/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_upload_large.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.uploadlarge;

import java.io.File;
import java.io.RandomAccessFile;

import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.kernel.enumerated.LetterCase;
import org.zhiqim.kernel.util.Bytes;
import org.zhiqim.kernel.util.DateTimes;
import org.zhiqim.kernel.util.Files;
import org.zhiqim.kernel.util.Hexs;
import org.zhiqim.kernel.util.Ids;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.kernel.util.codes.MD5;
import org.zhiqim.orm.ORM;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.orm.dbo.Updater;
import org.zhiqim.uploadlarge.dbo.UpllChunk;
import org.zhiqim.uploadlarge.dbo.UpllFile;

/**
 * 大文件上传处理器接口
 *
 * @version v1.0.0 @author zouzhigang 2015-10-23 新建与整理
 */
public class ZulUploader implements ZulConstants
{
    /**
     * 通过文件编号读取文件路径
     * 
     * @param request       HTTP请求
     * @param fileId        文件编号
     * @return              上传文件
     * @throws Exception    可能的异常
     */
    public static UpllFile getFile(HttpRequest request, String fileId) throws Exception
    {
        Selector selector = new Selector("fileId", fileId);
        selector.addReplace("id", request.getNestAttributeString(ORM.Z_ORM_ID));
        return ZulBootstrap.table(request).item(UpllFile.class, selector);
    }
    
    /**
     * 通过文件MD5码查询是否有已上传的文件
     * 
     * @param request       HTTP请求
     * @param fileMd5       文件MD5码 
     * @return              上传文件
     * @throws Exception    可能的异常
     */
    public static UpllFile queryFile(HttpRequest request, String fileMd5) throws Exception
    {
        Selector selector = new Selector("fileMd5", fileMd5);
        selector.addReplace("id", request.getNestAttributeString(ORM.Z_ORM_ID));
        return ZulBootstrap.table(request).item(UpllFile.class, selector);
    }
    
    /**
     * 通过文件MD5码和碎片大小查询是否有正在上传的文件
     * 
     * @param request       HTTP请求
     * @param fileMd5       文件MD5码 
     * @param chunkSize     碎片大小
     * @return              碎片编号，存在则返回chunkNo,不存在返回-1
     * @throws Exception    可能的异常
     */
    public static UpllChunk queryChunk(HttpRequest request, String fileMd5, int chunkSize) throws Exception
    {
        Selector selector = new Selector("fileMd5", fileMd5);
        selector.addMust("chunkSize", chunkSize);
        selector.addReplace("id", request.getNestAttributeString(ORM.Z_ORM_ID));
        return ZulBootstrap.table(request).item(UpllChunk.class, selector);
    }
    
    /**
     * 保存碎片数据
     * 
     * @param request       HTTP请求
     * @param data          碎片数据，第一个或中间的碎片
     * @param fileDir       文件目录
     * @param fileName      文件名，这里可能有中文名
     * @param fileMd5       文件MD5码
     * @param fileLength    文件总长度
     * @param chunkSize     碎片大小
     * @param chunkNum      碎片总数
     * @param chunkNo       碎片编号
     * @throws Exception    可能的异常
     */
    public static boolean saveChunkData(HttpRequest request, byte[] data, String fileDir, String fileName, String fileMd5, long fileLength, int chunkSize, int chunkNum, int chunkNo) throws Exception
    {
        String filePath = null;
        if (chunkNo == 1)
        {//第一块
            filePath = getChunkFilePath(fileDir, fileMd5);
            fileName = Files.getFileName(fileName);
        }
        else
        {//写入不是第一块时，取原来的filePath
            UpllChunk chunk = queryChunk(request, fileMd5, chunkSize);
            if (chunk == null)
            {//多人同时上传一个文件
                return false;
            }
            filePath = chunk.getFilePath();
        }
        
        //写入到文件
        RandomAccessFile file = new RandomAccessFile(new File(filePath), "rw");
        if (chunkNo == 1)
            file.setLength(fileLength);
        
        file.seek((chunkNo-1)*chunkSize);
        file.write(data);
        file.close();
        
        //保存到数据库
        if (chunkNo == 1)
        {//插入
            UpllChunk chunk = new UpllChunk();
            chunk.setId(request.getNestAttributeString(ORM.Z_ORM_ID));
            chunk.setChunkNo(chunkNo);
            chunk.setChunkNum(chunkNum);
            chunk.setChunkSize(chunkSize);
            chunk.setChunkTime(DateTimes.getDateTimeString());
            chunk.setFileExt(Files.getFileExt(fileName));
            chunk.setFileMd5(fileMd5);
            chunk.setFileName(fileName);
            chunk.setFilePath(filePath);
    
            ZulBootstrap.table(request).replace(chunk);
        }
        else
        {//更新
            Updater updater = new Updater();
            updater.addReplace("id", request.getNestAttributeString(ORM.Z_ORM_ID));
            updater.addField("chunkNo", chunkNo);
            updater.addField("chunkTime", DateTimes.getDateTimeString());
            updater.addMust("fileMd5", fileMd5);
            updater.addMust("chunkSize", chunkSize);
            
            ZulBootstrap.table(request).update(UpllChunk.class, updater);
        }
        
        return true;
    }
    
    /**
     * 清理碎片数据
     * 
     * @param request       HTTP请求
     * @param fileMd5       文件MD5码
     * @param chunkSize     碎片大小
     * @throws Exception    可能的异常
     */
    public static void clearChunkData(HttpRequest request, String fileMd5, int chunkSize) throws Exception
    {
        Selector selector = new Selector("fileMd5", fileMd5);
        selector.addMust("chunkSize", chunkSize);
        selector.addReplace("id", request.getNestAttributeString(ORM.Z_ORM_ID));
        ZulBootstrap.table(request).delete(UpllChunk.class, selector);
    }
    
    /**
     * 保存文件，正常从最后一个碎片中读取
     * 
     * @param request       HTTP请求
     * @param fileDir       文件目录
     * @param fileName      文件名，这里可能有中文名
     * @param fileLength    文件长度
     * @param fileMd5       文件MD5码
     * @param contentType   内容类型
     * @return              上传文件
     * @throws Exception    可能的异常
     */
    public static UpllFile saveFile(HttpRequest request, UpllChunk chunk, long fileLength, String contentType) throws Exception
    {
        //1.最后一个碎片后重命名文件
        String fileDir = Files.getFileDir(chunk.getFilePath());
        String filePath = getAvailableFilePath(fileDir, chunk.getFileName());
        String fileName = Files.getFileName(filePath);
        Files.renameFile(chunk.getFilePath(), filePath);
            
        //2.插入文件表
        String fileId = Ids.uuid();
        String fileExt = chunk.getFileExt();
        
        UpllFile file = new UpllFile();
        file.setId(request.getNestAttributeString(ORM.Z_ORM_ID));
        file.setFileId(fileId);
        file.setFileName(fileName);
        file.setFileLength(fileLength);
        file.setFilePath(filePath);
        file.setFileExt(chunk.getFileExt());
        file.setFileMd5(chunk.getFileMd5());
        file.setFileUrl(_PATH_UPLOAD_LARGE_PREFIX_ + fileId + Files.fixFileExt(fileExt));
        file.setFileTime(DateTimes.getDateTimeString());
        file.setContentType(contentType);
        ZulBootstrap.table(request).replace(file);
        
        //3.清理碎片表
        clearChunkData(request, chunk.getFileMd5(), chunk.getChunkSize());
        
        //4.返回结果
        return file;
    }
    
    /**
     * 提供给外部保存文件，文件大小不要太大
     * 
     * @param request           HTTP请求
     * @param data              文件数据
     * @param filePath          保存的文件路径
     * @param contentType       文件内容类型
     * @throws Exception        异常
     */
    public static UpllFile saveFile(HttpRequest request, byte[] data, String filePath, String contentType) throws Exception
    {
        String fileId = Ids.uuid();
        String fileExt = Files.getFileExt(filePath);
        String fileName = Files.getFileName(filePath);
        long fileLength = data.length;
        String fileMd5 = MD5.encode(data) + Hexs.toHexString(Bytes.BU.toBytes(fileLength), LetterCase.LOWER);
        
        //拷贝文件
        Files.write(filePath, data);
        
        //保存到数据库
        UpllFile file = new UpllFile();
        file.setId(request.getNestAttributeString(ORM.Z_ORM_ID));
        file.setFileId(fileId);
        file.setFileName(fileName);
        file.setFileExt(fileExt);
        file.setFileLength(fileLength);
        file.setFileMd5(fileMd5);
        file.setFilePath(filePath);
        file.setFileUrl(_PATH_UPLOAD_LARGE_PREFIX_ + fileId + Files.fixFileExt(fileExt));
        file.setFileTime(DateTimes.getDateTimeString());
        file.setContentType(contentType);
        
        ZulBootstrap.table(request).replace(file);
        return file;
    }
    
    /**
     * 保存文件，从另一个文件拷贝属性过来
     * 
     * @param request       HTTP请求
     * @param oFile         原始文件对象
     * @return              上传文件
     * @throws Exception    可能的异常
     */
    public static UpllFile saveFile(HttpRequest request, UpllFile oFile) throws Exception
    {//秒传
        String fileId = Ids.uuid();
        String filePath = oFile.getFilePath();
        String fileExt = oFile.getFileExt();
        String fileName = oFile.getFileName();
        
        UpllFile file = new UpllFile();
        file.setId(request.getNestAttributeString(ORM.Z_ORM_ID));
        file.setFileId(fileId);
        file.setFileName(fileName);
        file.setFileExt(fileExt);
        file.setFileLength(oFile.getFileLength());
        file.setFileMd5(oFile.getFileMd5());
        file.setFilePath(filePath);
        file.setFileUrl(_PATH_UPLOAD_LARGE_PREFIX_ + fileId + Files.fixFileExt(fileExt));
        file.setFileTime(DateTimes.getDateTimeString());
        file.setContentType(oFile.getContentType());
        ZulBootstrap.table(request).replace(file);
        
        return file;
    }
    
    /**
     * 拷贝文件，从另一个文件拷贝属性过来
     * 
     * @param request       HTTP请求
     * @param oFile         原始文件对象
     * @return              上传文件
     * @throws Exception    可能的异常
     */
    public static UpllFile copyFile(HttpRequest request, UpllFile oFile, String fileDir, String fileName) throws Exception
    {//秒传拷贝文件
        String fileId = Ids.uuid();
        String filePath = getAvailableFilePath(fileDir, fileName);
        String fileExt = oFile.getFileExt();
        
        //拷贝文件
        Files.copyFile(oFile.getFilePath(), filePath);
        
        //保存到数据库
        UpllFile file = new UpllFile();
        file.setId(request.getNestAttributeString(ORM.Z_ORM_ID));
        file.setFileId(fileId);
        file.setFileName(fileName);
        file.setFileExt(fileExt);
        file.setFileLength(oFile.getFileLength());
        file.setFileMd5(oFile.getFileMd5());
        file.setFilePath(filePath);
        file.setFileUrl(_PATH_UPLOAD_LARGE_PREFIX_ + fileId + Files.fixFileExt(fileExt));
        file.setFileTime(DateTimes.getDateTimeString());
        file.setContentType(oFile.getContentType());
        ZulBootstrap.table(request).replace(file);
        
        return file;
    }
    
    /** 获取一个碎片的文件名称，格式zhiqim_${fileMd5}.upll */
    private static String getChunkFilePath(String fileDir, String fileMd5)
    {
        return new StringBuilder(fileDir)
            .append(_CHUNK_FILE_NAME_PREFIX_).append(fileMd5).append(_CHUNK_FILE_NAME_EXT_)
            .toString();
    }
    
    /** 获取一个可用的文件名称，有重名的，后面加(n) */
    private static String getAvailableFilePath(String fileDir, String fileName)
    {
        String fileExt = Files.getFileExt(fileName);
        if (!Validates.isEmpty(fileExt))
        {
            fileExt = "." + fileExt;
            fileName = Strings.trimRight(fileName, fileExt);
        }
        
        String filePath = fileDir + fileName + fileExt;
        int i = 1;
        while (true)
        {//判断相同文件名
            File file = new File(filePath);
            if (!file.exists())
                return filePath;
            
            filePath = fileDir + fileName + "("+ i +")" + fileExt;
            i++;
        }
    }
}
