/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_upload_large.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.uploadlarge;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;

import org.zhiqim.uploadlarge.dbo.UpllChunk;
import org.zhiqim.uploadlarge.dbo.UpllFile;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.httpd.HttpExecutor;
import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.httpd.HttpResponse;
import org.zhiqim.httpd.util.Responses;
import org.zhiqim.kernel.enumerated.LetterCase;
import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.util.Bytes;
import org.zhiqim.kernel.util.Files;
import org.zhiqim.kernel.util.Hexs;
import org.zhiqim.kernel.util.Streams;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Urls;
import org.zhiqim.kernel.util.Validates;

/**
 * 上传服务，用于固定访问地址为/service/uploadelarge，实际处理在forward中
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class ZulService implements HttpExecutor, ZulConstants
{
    private static final Log log = LogFactory.getLog(ZulService.class);
    private static final String[] FILE_DIR_NOT_FORMAT = {".", "\\", "?", ":", "*", "\'", "\"", "<", ">", "|"};
    private String uploadRootDir;
    private String uploadOrm;
    
    @Override
    public boolean isMatch(String pathInContext)
    {
        return _PATH_SERVICE_UPLOAD_LARGE_.equals(pathInContext);
    }
    
    @Override
    public void handle(HttpRequest request, HttpResponse response) throws IOException
    {
        String sessionName = Strings.toString(request.getSessionName(), "guest");
        
        /**********************************************************/
        //第一步，检查上传根目录是否配置，如果未配置则禁止访问
        /**********************************************************/
        HttpContext context = request.getContext();
        
        if (uploadRootDir == null)
        {//先判断是否提供上传处理器
            String rootDir = context.getAttributeString(_SERV_UPLOAD_LARGE_ROOT_DIR_);
            if (Validates.isEmptyBlank(rootDir) || !Files.mkDirectory(rootDir))
            {
                log.error("USER:%s IP:%s [UploadLarge][配置处理器处理时必须配置上传根目录][%s]", sessionName, request.getRemoteAddr(), _SERV_UPLOAD_LARGE_ROOT_DIR_);
                response.sendError(_403_FORBIDDEN_);
                return;
            }
            
            uploadRootDir = Files.toLinuxPath(new File(rootDir).getCanonicalPath());
            uploadRootDir = Strings.addEndsWith(uploadRootDir, "/");
        }
        
        if (uploadOrm == null)
        {//数据库ID
            uploadOrm = context.getAttributeString(_SERV_UPLOAD_LARGE_ORM_, "");
        }
        
        /**********************************************************/
        //第二步，判断上传参数
        /**********************************************************/
        int mode = request.getHeaderInt(_X_RESPONSED_MODE_, 0);
        String fileDir = request.getHeader(_X_UPLOAD_FILE_DIR_);
        String fileName = request.getHeader(_X_UPLOAD_FILE_NAME_);
        String fileMd5 = request.getHeader(_X_UPLOAD_FILE_MD5_);
        int fileMd5Target = request.getHeaderInt(_X_UPLOAD_FILE_MD5_TARGET_, 0);
        int fileCopy = request.getHeaderInt("X-Upload-File-Copy", 0);
        long fileLength = request.getHeaderInt(_X_UPLOAD_FILE_LENGTH_);
        int chunkNum = request.getHeaderInt(_X_UPLOAD_CHUNK_NUM_);
        int chunkSize = request.getHeaderInt(_X_UPLOAD_CHUNK_SIZE_);
        int chunkNo = request.getHeaderInt(_X_UPLOAD_CHUNK_NO_);
        int contentLength = request.getContentLength();
        
        if (mode < 0 || mode > 1  || fileMd5Target < 0 || fileMd5Target > 2  || !Validates.isMD5String(fileMd5))
        {//两个值可以为空，默认0，MD5码必须是标准MD5码
            log.error("USER:%s IP:%s [UploadLarge][参数不正确][mode:%s][fileMd5Target:%s][fileMd5:%s]", sessionName, request.getRemoteAddr(), mode, fileMd5Target, fileMd5);
            response.sendError(_412_PRECONDITION_FAILED_);
            return;
        }
        
        if (fileLength == -1 || chunkNum < 1 || chunkNo < 1 || chunkSize < 1024 * 100  ||   //文件大小不能为空，碎片大小不能于小100K，碎片数和当前碎片不能<1
            contentLength <= 0 || fileLength < contentLength || chunkNum < chunkNo)         //当前碎片大小不能为空，总大小不能小于当前大小，当前碎片编号必须小于总数
        {
            log.error("USER:%s IP:%s [UploadLarge][参数不正确][fileLength:%s][contentLength:%s][chunkSize:%s][chunkNo:%s]][chunkNum:%s][%s]", sessionName, request.getRemoteAddr(), fileLength, contentLength, chunkSize, chunkNo, chunkNum, request.getUserAgent());
            response.sendError(_412_PRECONDITION_FAILED_);
            return;
        }
        
        if (contentLength > chunkSize || (chunkNo < chunkNum && contentLength != chunkSize))
        {
            log.error("USER:%s IP:%s [UploadLarge][上传文件内容长度不正确][contentLength:%s][chunkSize:%s][chunkNo:%s][chunkNum:%s][%s]", sessionName, request.getRemoteAddr(), contentLength, chunkSize, chunkNo, chunkNum, request.getUserAgent());
            response.sendError(_415_UNSUPPORTED_MEDIA_TYPE_);
            return;
        }
        
        fileDir = Urls.decodeUTF8(fileDir);
        fileDir = Strings.removeStartsWith(fileDir, "/");
        fileDir = Strings.removeEndsWith(fileDir, "/");
        fileName = Urls.decodeUTF8(fileName);
        fileMd5 = fileMd5.toLowerCase() + Hexs.toHexString(Bytes.BU.toBytes(fileLength), LetterCase.LOWER);
        
        if (Validates.isStrContainStrArr(fileDir, FILE_DIR_NOT_FORMAT) ||   //文件目录不支持.\:*?'"<>|
            Validates.isEmptyBlank(fileName))                               //文件名不能为空
        {
            log.error("USER:%s IP:%s [UploadLarge][参数不正确][fileDir:%s][fileName:%s]", sessionName, request.getRemoteAddr(), fileDir, fileName);
            response.sendError(_412_PRECONDITION_FAILED_);
            return;
        }
        
        /**********************************************************/
        //第三步，判断以处理器方式时检查处理器配置参数和文件目录和文件名限制
        //1.处理器检查，必须实现Upload接口
        //2.文件根目录配置必须
        //3.对文件目录进行UTF-8解码，且必须支持创建
        //4.对文件名进行UTF-8解码
        /**********************************************************/
        
        //对参数模式、文件目录和名称等作一下处理
        if (Validates.isEmptyBlank(fileDir))    
            fileDir = uploadRootDir;
        else
        {
            fileDir = Strings.addEndsWith(fileDir, "/");
            fileDir = uploadRootDir + fileDir;
            if (!Files.mkDirectory(fileDir))
            {
                log.error("USER:%s IP:%s [UploadLarge][上传的文件目录无法创建][%s]", sessionName, request.getRemoteAddr(), request.getHeader(_X_UPLOAD_FILE_DIR_));
                response.sendError(_403_FORBIDDEN_);
                return;
            }
        }
        
        if (fileName.length() > 512)
            fileName = fileName.substring(fileName.length()-512);
        
        /**********************************************************/
        //第四步，以接口方式处理，判断秒传、续传还是新传
        //1.秒传是指存在已上传的MD5相同的文件
        //2.续传是指有正在上传的MD5相同且块大小相同的碎片
        //3.新传是指没有已上传和MD5相同的文件，且没有MD5相同且块大小相同的碎片
        /**********************************************************/
        
        try
        {
            UpllFile file = ZulUploader.queryFile(request, fileMd5);
            if (file != null)
            {//文件已存在，秒传，返回204，头部包含URL，复制一条记录
                if (fileCopy == 0)
                    file = ZulUploader.saveFile(request, file);
                else
                    file = ZulUploader.copyFile(request, file, fileDir, fileName);
                
                log.info("USER:%s IP:%s [UploadLarge][秒传][%s]", sessionName, request.getRemoteAddr(), fileMd5);
                doUploadCompleted(response, mode, file.getFileId(), file.getFileName(), file.getFileUrl());
                return;
            }
    
            UpllChunk chunk = ZulUploader.queryChunk(request, fileMd5, chunkSize);
            if (chunk == null)
            {//文件碎片不存在
                if (chunkNo > 1)
                {//不是第一个块拒绝操作
                    log.error("USER:%s IP:%s [UploadLarge][碎片初始编号不正确]", sessionName, request.getRemoteAddr());
                    response.sendError(_400_BAD_REQUEST_);
                    return;
                }
            }
            else
            {//文件碎片已存在
                int lastChunkNo = chunk.getChunkNo();
                if (chunkNo > lastChunkNo+1)
                {//跳块拒绝操作
                    log.error("USER:%s IP:%s [UploadLarge][碎片跳编号]", sessionName, request.getRemoteAddr());
                    response.sendError(_400_BAD_REQUEST_);
                    return;
                }
                
                if (chunkNo <= lastChunkNo)
                {//已有块比当前多，续传
                    log.info("USER:%s IP:%s [UploadLarge][碎片续传][%s]", sessionName, request.getRemoteAddr(), chunkNo);
                    doUploadChunk(response, mode, lastChunkNo);
                    return;
                }
            }
            
            /**********************************************************/
            //第五步，上传碎片保存到存储器
            /**********************************************************/
            byte[] data = Streams.getBytes(request.getInputStream(), contentLength);
            
            boolean isSucc = ZulUploader.saveChunkData(request, data, fileDir, fileName, fileMd5, fileLength, chunkSize, chunkNum, chunkNo);
            if (!isSucc)
            {
                file = ZulUploader.queryFile(request, fileMd5);
                if (file == null)
                    throw new Exception("上传碎片时，文件不存在或已删除");
                
                //多人同时上传情况时，有一人已上传完成并删除了碎片，后续直接完成
                if (fileCopy == 0)
                    file = ZulUploader.saveFile(request, file);
                else
                    file = ZulUploader.copyFile(request, file, fileDir, fileName);
                
                log.info("USER:%s IP:%s [UploadLarge][上传完成][多人上传情况][%s][%s]", sessionName, request.getRemoteAddr(), fileMd5, file.getFileName());
                doUploadCompleted(response, mode, file.getFileId(), file.getFileName(), file.getFileUrl());
            }
            else
            {
                if (chunkNo < chunkNum)
                {//第一个和中间的碎片
                    log.info("USER:%s IP:%s [UploadLarge][碎片上传][总:%s][第:%s][%s]", sessionName, request.getRemoteAddr(), chunkNum, chunkNo, fileMd5);
                    doUploadChunk(response, mode, chunkNo);
                }
                else
                {//最后一个碎片，先清理碎片再保存文件
                    chunk = ZulUploader.queryChunk(request, fileMd5, chunkSize);
                    String contentType = request.getContentType();
                    
                    file = ZulUploader.saveFile(request, chunk, fileLength, contentType);
                    
                    log.info("USER:%s IP:%s [UploadLarge][上传完成][总:%s][最后一个碎片][%s][%s]", sessionName, request.getRemoteAddr(), chunkNum, fileMd5, file.getFileName());
                    doUploadCompleted(response, mode, file.getFileId(), file.getFileName(), file.getFileUrl());
                }
            }
        }
        catch(Exception e)
        {
            try{response.sendHeader(_500_INTERNAL_SERVER_ERROR_);}catch(Exception e2){}
            log.error(e);
        }
    }
    
    /**
     * 上传碎片返回200
     * 
     * @param response          响应
     * @param mode              响应模式,=1表示按内容发送，!=1表示按消息头发送
     * @param chunkNo           碎片编号
     * @throws IOException      可能的异常
     */
    private void doUploadChunk(HttpResponse response, int mode, int chunkNo) throws IOException
    {
        HashMap<String, String> headerMap = new HashMap<String, String>();
        headerMap.put(_X_UPLOAD_CHUNK_NO_, ""+chunkNo);
        Responses.doReturnMessage(response, mode, headerMap);
    }
    
    /**
     * 上传完成返回204
     * 
     * @param response          响应
     * @param mode              响应模式,=1表示按内容发送，!=1表示按消息头发送
     * @param fileId            文件编号
     * @param fileName          文件名
     * @param fileUrl           文件URL
     * @throws IOException      可能的异常
     */
    private void doUploadCompleted(HttpResponse response, int mode, String fileId, String fileName, String fileUrl) throws IOException
    {
        HashMap<String, String> headerMap = new HashMap<String, String>();
        headerMap.put(_X_UPLOAD_FILE_ID_, fileId);
        headerMap.put(_X_UPLOAD_FILE_NAME_, Urls.encodeUTF8(fileName));
        headerMap.put(_X_UPLOAD_FILE_URL_, fileUrl);
        Responses.doReturnMessage(response, mode, headerMap);
    }
}
