/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm.datasource;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringReader;
import java.io.Writer;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import oracle.sql.BLOB;
import oracle.sql.CLOB;

import org.zhiqim.kernel.extend.MapSO;
import org.zhiqim.kernel.util.Hexs;
import org.zhiqim.kernel.util.Replaces;
import org.zhiqim.kernel.util.Sqls;
import org.zhiqim.orm.ORMServer;

/**
 * 数据库处理工具类
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class ZStatement
{
    /**
     * 预处理语句设置参数，对SQL语句填充后返回SQL语句，用于打印日志
     * 
     * @param server        ORM服务
     * @param pstmt         预处理语句
     * @param index         索引
     * @param value         设置的值
     * @param str           SQL语句
     * @return              填充后的SQL语句
     * @throws SQLException SQL异常
     * @throws IOException  IO异常
     */
    public static String setParameter(ORMServer server, PreparedStatement pstmt, int index, Object value, String str) throws SQLException, IOException
    {
        if (value == null)
        {
            pstmt.setNull(index, Types.CHAR);//注意这里用Types.CHAR,可以保证string,int,long等值正确，其他的未测试
            str = str.replaceFirst("\\?", "null");
        }
        //基本类型
        else if (value instanceof Boolean)
        {
            pstmt.setBoolean(index, ((Boolean)value).booleanValue());
            str = str.replaceFirst("\\?", String.valueOf(value));
        }
        else if (value instanceof Byte)
        {
            pstmt.setByte(index, ((Byte)value).byteValue());
            str = str.replaceFirst("\\?", String.valueOf(value));
        }
        else if (value instanceof Short)
        {
            pstmt.setShort(index, ((Short)value).shortValue());
            str = str.replaceFirst("\\?", String.valueOf(value));
        }
        else if (value instanceof Integer)
        {
            pstmt.setInt(index, ((Integer)value).intValue());
            str = str.replaceFirst("\\?", String.valueOf(value));
        }
        else if (value instanceof Long)
        {
            pstmt.setLong(index, ((Long)value).longValue());
            str = str.replaceFirst("\\?", String.valueOf(value));
        }
        //日期格式
        else if (value instanceof Timestamp)
        {
            Timestamp time = (Timestamp)value;
            pstmt.setTimestamp(index, time);
            
            str = str.replaceFirst("\\?", "'" + Sqls.toDateTimeString(time) + "'");
        }
        else if (value instanceof Date)
        {
            Date date = (Date)value;
            pstmt.setDate(index, date);
            
            str = str.replaceFirst("\\?", "'" + Sqls.toDateString(date) + "'");
        }
        else if (value instanceof Time)
        {
            Time time = (Time)value;
            pstmt.setTime(index, time);
            
            str = str.replaceFirst("\\?", "'" + Sqls.toTimeString(time) + "'");
        }
        //字符串
        else if (value instanceof String)
        {
            String v = (String)value;
            if (v.length() <= 2000)
                pstmt.setString(index, v);
            else if (server.isOracle())
            {//oracle
                CLOB clob = CLOB.createTemporary(pstmt.getConnection(), false, CLOB.DURATION_SESSION);
                clob.open(CLOB.MODE_READWRITE);
                Writer writer = clob.getCharacterOutputStream();
                writer.write(v);
                writer.close();
                clob.close();
                
                pstmt.setClob(index, clob);
            }
            else
            {//其他
                StringReader reader = new StringReader(v);
                pstmt.setCharacterStream(index, reader, v.length());
            }
            
            //SQL转义
            v = v.replaceAll("'", "''");
            if (v.indexOf("?") != -1)
            {//转义
                v = v.replaceAll("\\?", "-%6-%-3%-");
            }
            
            v = Replaces.toReplaceEscape(v);
            str = str.replaceFirst("\\?", "'"+ v +"'");
        }
        //二进制
        else if (value instanceof byte[])
        {
            byte[] v = (byte[])value;
            if (!server.isOracle())
                pstmt.setBytes(index, v);
            else
            {
                BLOB blob = BLOB.createTemporary(pstmt.getConnection(), false, BLOB.DURATION_SESSION);
                blob.open(BLOB.MODE_READWRITE);
                OutputStream os = blob.getBinaryOutputStream();
                os.write(v);
                os.close();
                blob.close();
                
                pstmt.setBlob(index, blob);
            }
            
            str = str.replaceFirst("\\?", "'"+ Hexs.toHexString(v) +"'");
        }
        else
        {//其他类型
            pstmt.setObject(index, value);
            str = str.replaceFirst("\\?", String.valueOf(value));
        }
        
        return str;
    }
    
    /**
     * 预处理语句设置参数
     * 
     * @param server        ORM服务
     * @param pstmt         预处理语句
     * @param index         索引
     * @param value         设置的值
     * @throws SQLException SQL异常
     * @throws IOException  IO异常
     */
    public static void setParameter(ORMServer server, PreparedStatement pstmt, int index, Object value) throws SQLException, IOException
    {
        if (value == null)
            pstmt.setNull(index, Types.CHAR);//注意这里用Types.CHAR,可以保证string,int,long等值正确，其他的未测试
        
        //基本类型
        else if (value instanceof Boolean)
            pstmt.setBoolean(index, ((Boolean)value).booleanValue());
        else if (value instanceof Byte)
            pstmt.setByte(index, ((Byte)value).byteValue());
        else if (value instanceof Short)
            pstmt.setShort(index, ((Short)value).shortValue());
        else if (value instanceof Integer)
            pstmt.setInt(index, ((Integer)value).intValue());
        else if (value instanceof Long)
            pstmt.setLong(index, ((Long)value).longValue());
        //日期类型
        else if (value instanceof Timestamp)
            pstmt.setTimestamp(index, (Timestamp)value);
        else if (value instanceof Date)
            pstmt.setDate(index, (Date)value);
        else if (value instanceof Time)
            pstmt.setTime(index, (Time)value);
        //字符串
        else if (value instanceof String)
        {
            String v = (String)value;
            if (v.length() <= 2000)
                pstmt.setString(index, (String)value);
            else if (server.isOracle())
            {//oracle clob
                CLOB clob = CLOB.createTemporary(pstmt.getConnection(), false, CLOB.DURATION_SESSION);
                clob.open(CLOB.MODE_READWRITE);
                Writer writer = clob.getCharacterOutputStream();
                writer.write(v);
                writer.close();
                clob.close();
                
                pstmt.setClob(index, clob);
            }
            else
            {//其他
                StringReader reader = new StringReader(v);
                pstmt.setCharacterStream(index, reader, v.length());
            }
        }
        //二进制
        else if (value instanceof byte[])
        {
            byte[] v = (byte[])value;
            if (!server.isOracle())
                pstmt.setBytes(index, v);
            else
            {//oracle blob
                BLOB blob = BLOB.createTemporary(pstmt.getConnection(), false, BLOB.DURATION_SESSION);
                blob.open(BLOB.MODE_READWRITE);
                OutputStream os = blob.getBinaryOutputStream();
                os.write(v);
                os.close();
                blob.close();
                
                pstmt.setBlob(index, blob);
            }
        }
        else
        {//其他类型
            pstmt.setObject(index, value);
        }
    }
    
    
    /**
     * 替换SQL中##参数值为?
     * 
     * @param sql SQL语句
     * @param paramKeyList 参数KEY列表
     * @return 新的SQL语句
     */
    public static String paramMapString(String sql, List<String> paramKeyList)
    {
        for (int i=0,start=0,end=sql.indexOf('#');end!=-1;i++,start=end,end=sql.indexOf('#', start+1))
        {
            if (i % 2 == 0)
                continue;
            String paramKey = sql.substring(start + 1, end);
            paramKeyList.add(paramKey);
        }

        for (int i=0;i<paramKeyList.size();i++)
        {
            String paramKey = paramKeyList.get(i);
            sql = sql.replaceAll("#"+paramKey+"#", "?");//把#paramKey#换成?
        }
        
        return sql;
    }
    
    /**
     * 替换SQL中##参数值为结果值
     * 
     * @param sql       SQL语句
     * @param paramMap  参数KEY列表
     * @return          新的SQL语句
     */
    public static String paramMapString(String sql, MapSO paramMap)
    {
        List<String> paramKeyList = new ArrayList<String>();
        for (int i=0,start=0,end=sql.indexOf('#');end!=-1;i++,start=end,end=sql.indexOf('#', start+1))
        {
            if (i % 2 == 0)
                continue;
            String paramKey = sql.substring(start + 1, end);
            paramKeyList.add(paramKey);
        }

        for (int i=0;i<paramKeyList.size();i++)
        {
            String paramKey = paramKeyList.get(i);
            Object paramValue = paramMap.get(paramKey);
            sql = sql.replaceAll("#"+paramKey+"#", "'"+paramValue+"'");//把#paramKey#换成'paramValue'
        }
        
        return sql;
    }
}
