/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm.datasource;

import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import javax.naming.InitialContext;
import javax.sql.DataSource;

import org.zhiqim.kernel.control.ThreadLock;
import org.zhiqim.kernel.control.Threadx;
import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.util.Classes;
import org.zhiqim.kernel.util.Randoms;
import org.zhiqim.kernel.util.Threads;

/**
 * 数据库配置与管理类 <br>
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class ZDataSource extends Threadx implements DataSource, Runnable
{
    private static final Log log = LogFactory.getLog("database.log");
    
    private static final String JNDI    = "jndi";         //JNDI
    private static final int CHK_TIME   = 61 * 1000;      //每61秒检查一次线程池
    
    private final ThreadLock lock = new ThreadLock();      //数据库连接有效时通知锁
    private String id;                                      //服务编号
    private List<ZConnection> connList;                     //数据库连接池
    
    //数据库驱动配置
    private String driver;                                  //数据库驱动
    private String url;                                     //数据库URL
    private String user;                                    //数据库用户名
    private String pass;                                    //数据库密码
    
    //数据库连接池配置
    private int minPoolSize;                                //连接池最小数目
    private int maxPoolSize;                                //连接池最大数目
    private long maxKeepMilliTime;                          //数据库连接最大保持毫秒时间，超出则重建连接
    private long maxIdleMilliTime;                          //数据库连接最大空闲毫秒时间，超出则重建连接
    private int maxCompletedCount;                          //数据库连接最大处理数，超出则重建连接
    
    //数据库连接检查配置
    private boolean isChkConnOnTimer;                       //是否定时检查连接有效性
    private boolean isChkConnOnGet;                         //是否在获取连接时检查连接有效性
    private boolean isChkConnOnRelease;                     //是否在释放连接时检查连接有效性
    
    //数据库连接耗尽策略配置
    private int outOfConnWaitMilliTime;                     //数据库连接耗尽时，最大等待时长，单位毫秒，建议5000毫秒
    private int outOfConnRetryCount;                        //数据库连接耗尽时，重试次数，建议重试1次
    
    //数据库状态属性
    private boolean isDbBreak;                              //数据库是否已断开
    private ZConnectionTester tester;                        //数据库连接测试类
    
    /**
     * 默认参数构造函数，默认空闲时间=保持时间，最大100001次调用，定时检查连接有效性，获取和释放时检查有效性
     * 
     * @param driver                        数据库驱动
     * @param url                           数据库URL
     * @param user                          数据库用户名
     * @param pass                          数据库密码
     * @param minPoolSize                   连接池最小数目
     * @param maxPoolSize                   连接池最大数目
     * @param maxKeepTime                   数据库连接最大保持时长，超出则重建连接，单位:秒
     */
    public ZDataSource(String driver, String url, String user, String pass, int minPoolSize, int maxPoolSize, int maxKeepTime)
    {
        this(driver, url, user, pass, minPoolSize, maxPoolSize, maxKeepTime, maxKeepTime, 100001, true, false, false, 5, 1);
    }
    
    /**
     * 全参数构造函数
     * 
     * @param driver                        数据库驱动
     * @param url                           数据库URL
     * @param user                          数据库用户名
     * @param pass                          数据库密码
     * @param minPoolSize                   连接池最小数目
     * @param maxPoolSize                   连接池最大数目
     * @param maxKeepTime                   数据库连接最大保持时长，超出则重建连接，单位:秒
     * @param maxIdleTime                   数据库连接最大空闲时长，超出则重建连接，单位:秒
     * @param maxCompletedCount             数据库连接最大处理数，超出则重建连接
     * @param isChkConnOnTimer              是否定时检查连接有效性
     * @param isChkConnOnGet                是否在获取连接时检查连接有效性
     * @param isChkConnOnRelease            是否在释放连接时检查连接有效性
     * @param outOfConnWaitTime             数据库连接耗尽时，最大等待时长，单位秒
     * @param outOfConnRetryCount           数据库连接耗尽时，重试次数，建议重试1次
     */
    public ZDataSource(String driver, String url, String user, String pass, int minPoolSize, int maxPoolSize, int maxKeepTime, int maxIdleTime, int maxCompletedCount, 
        boolean isChkConnOnTimer, boolean isChkConnOnGet, boolean isChkConnOnRelease, int outOfConnWaitTime, int outOfConnRetryCount)
    {
        //配置
        this.id = Randoms.lettersDigits(2);
        this.connList = new ArrayList<ZConnection>(this.maxPoolSize);
        
        //传入参数
        this.driver = driver;
        this.url = url;
        this.user = user;
        this.pass = pass;
        this.minPoolSize = minPoolSize;
        this.maxPoolSize = maxPoolSize;
        this.maxKeepMilliTime = maxKeepTime * 1000;
        this.maxIdleMilliTime = maxIdleTime * 1000;
        this.maxCompletedCount = maxCompletedCount;
        this.isChkConnOnTimer = isChkConnOnTimer;
        this.isChkConnOnGet = isChkConnOnGet;
        this.isChkConnOnRelease = isChkConnOnRelease;
        this.outOfConnWaitMilliTime = outOfConnWaitTime * 1000;
        this.outOfConnRetryCount = outOfConnRetryCount;
    }
    
    /** 数据库信息 */
    public String toString()
    {
        return new StringBuilder("Database[")
            .append(url).append(";")
            .append("user:").append(user).append(";")
            .append("min:").append(minPoolSize).append(";")
            .append("max:").append(maxPoolSize).append(";")
            .append("cur:").append(getCurPoolSize()).append(";")
            .append("idle:").append(getIdlePoolSize()).append(";")
            .append("maxKeepTime:").append(maxKeepMilliTime/1000).append(";")
            .append("maxIdleTime:").append(maxIdleMilliTime/1000).append(";")
            .append("maxCompletedCount:").append(maxCompletedCount).append(";")
            .append("isChkOnTimer:").append(isChkConnOnTimer).append(";")
            .append("isChkOnGet").append(isChkConnOnGet).append(";")
            .append("isChkOnRelease:").append(isChkConnOnRelease).append(";")
            .append("outOfConnWaitTime:").append(outOfConnWaitMilliTime/1000).append(";")
            .append("outOfConnRetryCount:").append(outOfConnRetryCount).append("]")
            .toString();
    }
    
    /**********************************************************************************/
    //数据库服务线程开启&关闭&运行
    /**********************************************************************************/
    
    @Override /** 线程名 */
    protected String getThreadName()
    {
        return "ZDataSource-"+id;
    }
    
    @Override /** 线程开启前 */
    protected boolean openBefore()
    {
        if (!createConnectionTester())
        {//创建连接测试者
            return false;
        }
        
        //2.先删除已有的连接
        deleteConnections();
        
        //3.创建新的连接
        return createConnections();
    }

    @Override /** 线程关闭后 */
    protected void closeAfter()
    {
        deleteConnections();
        tester.shutdown();
        
        log.info("数据库[%s|%s]监视线程退出", url, user);
    }
    
    
    @Override /** 线程首次运行 */
    protected void first()
    {
        Threads.sleepIgnoreException(CHK_TIME);
    }
    
    @Override /** 线程持续运行 */
    protected void loop()
    {
        //1.打印数据库日记信息
        log.info(toString());
        
        //2.检查数据库是否断开
        isDbBreak = tester.isDbBreak();
        if (isDbBreak)
        {//如果数据库断开,则关闭所有连接
            deleteConnections();
            return;
        }
        
        //3.检查数据库连接有效性
        int closeNum=0,initNum=0;
        synchronized (connList)
        {
            //检查所有空闲线程
            for (Iterator<ZConnection> it=connList.iterator();it.hasNext();)
            {
                ZConnection conn = it.next();
                if (conn.isActive())
                {//活动中的连接不检查
                    continue;
                }
                
                if (conn.isClosed() || conn.isOvertimeOrCompletedCount() || (isChkConnOnTimer && !conn.isConnectionAvailable()))
                {//已关闭、已超出时长或数目、和要求定时检查连接连接不可用时，三种情况下销毁连接并从队列移除
                    String connId = conn.getId();
                    conn.shutdown();
                    it.remove();
                    conn = null;
                    
                    closeNum++;
                    log.info("监视关闭数据库连接[%s][%s|%s]成功", connId, url, user);
                }
            }
            
            //小于最小连接数时补上
            int num = minPoolSize - connList.size();
            for(int i=0;i<num;i++)
            {
                ZConnection conn = newProxyConnection("补足");
                if (conn != null)
                {
                    connList.add(conn);
                    initNum++;
                }
            }
        }
        
        //4.处理完如果有补上新建连接则进行通知和打印日志
        if (initNum > 0)
        {
            lock.unlock();
            log.info("DataBase[closeNum:%s;initNum:%s]", closeNum, initNum);
        }
        
        Threads.sleepIgnoreException(CHK_TIME);
    }

    /**创建测试连接*/
    private boolean createConnectionTester()
    {
        if (tester != null)
        {//如果已有测试器，先关闭
            tester.shutdown();
            tester = null;
        }
        
        try
        {
            if (!JNDI.equalsIgnoreCase(driver))
            {//判断驱动类是否能加载到
                Class<?> cls = Classes.forName(driver);
                if (cls == null)
                {
                    log.error("初始化数据库[%s|%s]连接池，未找到驱动类[%s]", url, user, driver);
                    return false;
                }
            }
            
            tester = new ZConnectionTester(this);
            if (tester.isDbBreak())
            {
                log.error("初始化数据库[%s|%s]连接池，发生异常：连接被断开", url, user);
                return false;
            }
        }
        catch (Exception e)
        {
            log.error("数据库[%s|%s]异常或读起初始化数据异常：%s", url, user, e.getMessage());
            return false;
        }
        
        return true;
    }

    /** 从池里获取一个连接，当连接耗尽时重复等待多少次 */
    public Connection getConnection() throws SQLException
    {
        return getConnection(0);
    }
    
    /** 从池里获取一个连接 */
    private Connection getConnection(int times) throws SQLException
    {
        //第一步，先看是否能从连接池找到或新建
        synchronized (connList)
        {
            //1.1循环寻找一个有效的空闲连接，并对无效连接进行处理
            for (Iterator<ZConnection> it=connList.iterator();it.hasNext();)
            {
                ZConnection conn = it.next();
                if (conn.isActive())
                {//1.1.1 先判断是否活动中，活动中继续找下一个
                    continue;
                }
                
                if (conn.isClosed() || conn.isOvertimeOrCompletedCount() || (isChkConnOnGet && !conn.isConnectionAvailable()))
                {//1.1.2 已关闭、已超出时长或数目、和要求获取时检查连接连接不可用时，三种情况下销毁连接并从队列移除
                    String connId = conn.getId();
                    conn.shutdown();
                    it.remove();
                    conn = null;
                    log.info("获取到已关闭或超时或超出调用数的连接[%s][%s|%s]关闭成功", connId, url, user);
                    continue;
                }
                
                //1.1.3 找到一个空闲并可用的连接即返回
                return conn.active();
            }
            
            //1.2如果没有空闲连接则检查是否达到最达连接数，没到最大值新建一个并返回
            if (connList.size() < maxPoolSize)
            {
                ZConnection conn = newProxyConnection("获取");
                if (conn != null)
                {
                    connList.add(conn);
                    return conn.active();
                }
            }
        }
        
        //第二步，如果没有空闲且连接达到最大数，则线程进入等待状态5秒，等待别的线程处理完释放连接
        lock.lock(outOfConnWaitMilliTime);
        
        //第三步，别的线程有释放或时间到达，检查是否有效线程，有则返回
        synchronized (connList)
        {
            for (ZConnection conn : connList)
            {
                if (!conn.isIdle())
                    continue;
                
                return conn.active();
            }
        }
        
        //第四步，最后还是没有获取到连接则抛出异常，防止线程锁死在等待数据库连接上，导致发现不了问题
        String fatal = "数据库连接["+url+"|"+user+"]连接耗尽，[max:"+getMaxPoolSize()+",cur:"+getCurPoolSize()+",idle:"+getIdlePoolSize()+"]";
        if (times < outOfConnRetryCount)
        {
            log.fatal("%s[第%s]", fatal, times+1);
            return getConnection(times + 1);
        }
        else
        {
            log.fatal("%s[第%s][全部失败抛出异常到业务层]", fatal, times+1);
            throw new SQLException("数据库连接耗尽，请与管理员联系检查数据库是否正常工作和连接池配置是否足够!");
        }
    }
    
    /** 释放一个连接到池内 */
    public void releaseConnection(ZConnection conn)
    {
        if (conn == null)
            return;
        
        if (conn.isClosed() || conn.isOvertimeOrCompletedCount() || (isChkConnOnRelease && !conn.isConnectionAvailable()))
        {//已关闭、已超出时长或数目、和要求释放时检查连接连接不可用时，三种情况下销毁连接并从队列移除
            String connId = conn.getId();
            conn.shutdown();
            synchronized (connList)
            {
                connList.remove(conn);
            }
            conn = null;
            
            log.info("释放关闭数据库连接[%s][%s|%s]成功", connId, url, user);
            return;
        }
        
        //连接没问题即设置为空闲并通知
        conn.idle();
        lock.unlock();
    }
    
    /** 创建所有连接 */
    private boolean createConnections()
    {
        synchronized (connList)
        {
            int num = minPoolSize - connList.size();
            for(int i=0;i<num;i++)
            {
                ZConnection conn = newProxyConnection("初始化");
                if (conn == null)
                    return false;
                
                connList.add(conn);
            }
        }
        
        lock.unlock();
        return true;
    }
    
    /** 删除所有连接 */
    private void deleteConnections()
    {
        synchronized (connList)
        {
            for (Iterator<ZConnection> it=connList.iterator();it.hasNext();)
            {
                ZConnection conn = it.next();
                String connId = conn.getId();
                conn.shutdown();
                it.remove();
                conn = null;
                
                log.info("销毁关闭数据库连接[%s][%s|%s]成功", connId, url, user);
            }
        }
    }

    /** 获取一个代理的连接 */
    ZConnection newProxyConnection(String type)
    {
        try
        {
            Connection conn = null;
            if (JNDI.equalsIgnoreCase(driver))
            {//JNDI
                DataSource  dataSource = (DataSource)new InitialContext().lookup(url);
                if (dataSource == null)
                    return null;
                
                dataSource.setLoginTimeout(10);
                conn = dataSource.getConnection();
            }
            else
            {//JDBC
                DriverManager.setLoginTimeout(10);
                conn = DriverManager.getConnection(url, user, pass);
            }
            
            if (conn == null)
            {
                log.error("创建数据库连接[%s|%s]失败，[%s]", url, user, type);
                return null;
            }
            
            ZConnection connection = new ZConnection(this, conn);
            //可选功能，事务等级暂未启用
            //connection.setTransactionIsolation(Connection.TRANSACTION_READ_UNCOMMITTED);
            log.info("创建数据库连接[%s][%s|%s]成功，[%s]", connection.getId(), url, user, type);
            return connection;
        }
        catch (SQLException e)
        {
            log.error("创建数据库连接[%s|%s]时，发生异常：[%s]", url, user, e.getMessage());
            return null;
        }
        catch(Exception e)
        {
            log.error("创建数据库连接[%s|%s]时，发生异常：[%s]", url, user, e.getMessage());
            return null;
        }
    }
    
    /***********************************/
    //DataSource 参数信息，密码不支持获取
    /***********************************/

    /** 判断数据库是否断开 */
    public boolean isDbBreak()
    {
        return tester.isDbBreak();
    }
    
    public String getId()
    {
        return id;
    }
    
    public String getDriver()
    {
        return driver;
    }
    
    public String getUrl()
    {
        return url;
    }

    public String getUser()
    {
        return user;
    }
    
    public int getMinPoolSize()
    {
        return minPoolSize;
    }

    public int getMaxPoolSize()
    {
        return maxPoolSize;
    }
    
    public int getCurPoolSize()
    {
        return connList.size();
    }
    
    public int getIdlePoolSize()
    {
        int size = 0;
        for (ZConnection conn : connList)
        {
            if (conn.isIdle())
                size++;
        }
        return size;
    }
    
    public int getMaxCompletedCount()
    {
        return maxCompletedCount;
    }
    
    public long getMaxKeepMilliTime()
    {
        return maxKeepMilliTime;
    }
    
    public long getMaxIdleMilliTime()
    {
        return maxIdleMilliTime;
    }

    public boolean isTestConnOnTimer()
    {
        return isChkConnOnTimer;
    }

    public boolean isTestConnOnGet()
    {
        return isChkConnOnGet;
    }

    public boolean isTestConnOnFree()
    {
        return isChkConnOnRelease;
    }
    
    /***********************************/
    //DataSource 支持动态修改的参数
    /***********************************/
    
    public void setMinPoolSize(int minPoolSize)
    {
        this.minPoolSize = minPoolSize;
    }

    public void setMaxPoolSize(int maxPoolSize)
    {
        this.maxPoolSize = maxPoolSize;
    }

    public void setMaxKeepTime(long maxKeepTime)
    {
        this.maxKeepMilliTime = maxKeepTime * 1000;
    }

    public void setMaxIdleTime(long maxIdleTime)
    {
        this.maxIdleMilliTime = maxIdleTime * 1000;
    }
    
    public void setMaxCompletedCount(int maxCompletedCount)
    {
        this.maxCompletedCount = maxCompletedCount;
    }

    public void setTestConnOnTimer(boolean isTestConnOnTimer)
    {
        this.isChkConnOnTimer = isTestConnOnTimer;
    }

    public void setTestConnOnGet(boolean isTestConnOnGet)
    {
        this.isChkConnOnGet = isTestConnOnGet;
    }

    public void setTestConnOnFree(boolean isTestConnOnFree)
    {
        this.isChkConnOnRelease = isTestConnOnFree;
    }

    /***********************************/
    //DataSource要求实现的方法，但没用上
    /***********************************/
    
    public Connection getConnection(String username, String password) throws SQLException
    {
        throw new SQLException("不支持传入用户名和密码获取连接");
    }
    
    public PrintWriter getLogWriter() throws SQLException
    {
        return DriverManager.getLogWriter();
    }

    public int getLoginTimeout() throws SQLException
    {
        return DriverManager.getLoginTimeout();
    }

    public void setLogWriter(PrintWriter out) throws SQLException
    {
        DriverManager.setLogWriter(out);
    }

    public void setLoginTimeout(int seconds) throws SQLException
    {
        DriverManager.setLoginTimeout(seconds);
    }
    
    /***********************************/
    //JDK1.6增加
    /***********************************/
    
    public boolean isWrapperFor(Class<?> iface) throws SQLException
    {
        return false;
    }

    public <T> T unwrap(Class<T> iface) throws SQLException
    {
        return null;
    }

    /***********************************/
    //JDK1.7增加
    /***********************************/
    
    public Logger getParentLogger() throws SQLFeatureNotSupportedException
    {
        return null;
    }
}