/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm;

import java.sql.SQLException;
import java.util.List;

import org.zhiqim.kernel.MultiInstance;
import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.paging.PageResult;
import org.zhiqim.orm.dbo.Selector;

/**
 * 视图类与数据库表多表映射调用，支持(count,sum,item,list,page)<br><br>
 * count,       统计视图类对应的关联表数目<br>
 * sum,         总计视图类对应的关联表数目<br>
 * item,        查询视图类对应的关联表一条数据，根据主键，或根据条件排序取第一条<br>
 * list,        查询视图类对应的关联表列表，或根据条件排序<br>
 * pag,         分页查询视图类对应的关联表列表，指定分页页码和页数目<br>
 * 
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 * @version v1.2.0 @author zouzhigang 2016-6-27 修改原类名FRHEL为ZView，正式使用视图名称
 */
@AnAlias("ZView")
public interface ZView extends MultiInstance
{
    /*****************************************************/
    //count 查询数目
    /*****************************************************/
    
    /**
     * 查询数目，查全视图
     * 
     * @param clazz         视图类
     * @return              返回数目
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int count(Class<?> clazz)throws ORMException, SQLException;
    
    /**
     * 查询数目，根据条件、可替换表查询
     * 
     * @param clazz         视图类
     * @param selector      对象查询器
     * @return              返回数目
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int count(Class<?> clazz, Selector selector)throws ORMException, SQLException;

    /*****************************************************/
    //sum 计算总和
    /*****************************************************/
    
    /**
     * 计算总和，全视图求和
     * 
     * @param clazz         视图类
     * @param field         视图字段
     * @return              返回总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long sum(Class<?> clazz, String field) throws ORMException, SQLException;
    
    /**
     * 计算总和，根据条件视图中求和
     * 
     * @param clazz         视图类
     * @param field         视图字段
     * @param selector      查询器
     * @return              返回总和(sum)
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long sum(Class<?> clazz, String field, Selector selector) throws ORMException, SQLException;
    
    /**
     * 计算多个总和，全视图中多个求和
     * 
     * @param clazz         视图类
     * @param fieldArr      多个视图字段
     * @return              返回计算多个总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long[] sum(Class<?> clazz, String[] fieldArr) throws ORMException, SQLException;
    
    /**
     * 计算多个总和，根据条件视图中多个求和
     * 
     * @param clazz         视图类
     * @param fieldArr      多个视图字段
     * @param selector      查询器
     * @return              返回计算多个总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long[] sum(Class<?> clazz, String[] fieldArr, Selector selector) throws ORMException, SQLException;
    
    /*****************************************************/
    //item 查询一条数据
    /*****************************************************/
    
    /**
     * 查询一个视图对象,全视图中取第一个
     * 
     * @param clazz         视图类
     * @return              返回视图对象
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> clazz) throws ORMException, SQLException;
    
    /**
     * 查询一个视图对象,可排序取第一个
     * 
     * @param clazz         视图类
     * @param selector      查询器
     * @return              返回视图对象
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> clazz, Selector selector) throws ORMException, SQLException;
    
    /*****************************************************/
    //list 查询列表
    /*****************************************************/
    
    /**
     * 查询列表，查全视图
     * 
     * @param clazz         视图类
     * @return              视图对象列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> clazz) throws ORMException, SQLException;
    
    /**
     * 查询列表，根据查询器
     * 
     * @param clazz         视图类
     * @param selector      查询器
     * @return              视图对象列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> clazz, Selector selector) throws ORMException, SQLException;
    
    /*****************************************************/
    //page 分页显示
    /*****************************************************/
    
    /**
     * 查询视图分页信息，查全视图
     * 
     * @param clazz         视图类
     * @param pageNo        页码
     * @param pageSize      页数
     * @return              分页信息,包括总页数,页码,页数和查询的记录
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> PageResult<T> page(Class<T> clazz, int pageNo, int pageSize) throws ORMException, SQLException;
    
    /**
     * 查询视图分页信息
     * 
     * @param clazz         视图类
     * @param pageNo        页码
     * @param pageSize      页数
     * @param selector      查询器
     * @return              分页信息,包括总页数,页码,页数和查询的记录
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> PageResult<T> page(Class<T> clazz, int pageNo, int pageSize, Selector selector) throws ORMException, SQLException;
}
