/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.sql.SQLException;
import java.util.Collection;
import java.util.List;

import org.zhiqim.kernel.Global;
import org.zhiqim.kernel.MultiInstancer;
import org.zhiqim.kernel.extend.HashMapCV;
import org.zhiqim.kernel.extend.HashMapSO;
import org.zhiqim.kernel.extend.HashMapSS;
import org.zhiqim.kernel.extend.MapSO;
import org.zhiqim.kernel.extend.MapSS;
import org.zhiqim.kernel.paging.PageBuilder;
import org.zhiqim.kernel.paging.PageResult;
import org.zhiqim.orm.annotation.AnIndex;
import org.zhiqim.orm.annotation.AnIndexValue;
import org.zhiqim.orm.annotation.AnTable;
import org.zhiqim.orm.annotation.AnTableField;
import org.zhiqim.orm.annotation.AnTableReplace;
import org.zhiqim.orm.dbo.Dbo;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.orm.dbo.Updater;
import org.zhiqim.orm.dbo.defined._Table;
import org.zhiqim.orm.dbo.defined._TableField;

/**
 * 标准类与数据库表映射调用，支持(truncate,exist,insert,replace,update,delete,count,sum,item,list,page)<br><br>
 * truncate,    清除表类对应的表<br>
 * exist,       检查表类对应的表是否存在<br>
 * create       创建表类对应的表
 * drop         删除表类对应的表
 * insert,      插入表类对应的表一条数据<br>
 * replace,     替换表类对应的表一条数据，根据主键<br>
 * update,      更新表类对应的表一条数据或根据条件更新数据<br>
 * delete,      删除表类对应的表一条数据或根据条件删除数据<br>
 * count,       统计表类对应的表数目<br>
 * sum,         总计表类对应的表数目<br>
 * item,        查询表类对应的表一条数据，根据主键，或根据条件排序取第一条<br>
 * list,        查询表类对应的表列表，或根据条件排序<br>
 * pag,         分页查询表类对应的表列表，指定分页页码和页数目<br>
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 * @version v1.2.0 @author zouzhigang 2016-6-27 修改原类名ZTableExcutor为ZTableExecutor，正式使用表类名称
 */
class ZTableImplement extends MultiInstancer implements ZTable, ORMConstants
{
    private ORMServer server;
    private ZSQL zSQL;
    private ZTabler zTabler;
    private ZTableCache zCache;
    private HashMapCV<_Table> tableMap;
    
    public ZTableImplement(ORMServer server)
    {
        this.server = server;
        this.zSQL = server.sql();
        this.zTabler = server.tabler();
        this.zCache = new ZTableCache(server);
        this.tableMap = new HashMapCV<>(true);
    }
    
    /**
     * 通过表类获取表对象
     * 
     * @param cls     表类
     * @return          表对象
     */
    public _Table getTable(Class<?> cls)
    {
        _Table _table = tableMap.get(cls);
        if (_table != null)
            return _table;
        
        AnTable anTable = cls.getAnnotation(AnTable.class);
        if (anTable == null)
            return null;
        
        _table = new _Table(cls.getName(), anTable.table(), anTable.key(), anTable.type());
        //1.增加索引
        AnIndex anIndex = cls.getAnnotation(AnIndex.class);
        if (anIndex != null)
        {
            AnIndexValue[] values = anIndex.value();
            for (AnIndexValue value : values)
            {
                _table.addIndex(value.name(), value.column(), value.unique());
            }
        }
        //2.增加字段和可替换字段
        Field[] fields = cls.getDeclaredFields();
        for (Field field : fields)
        {
            String fieldName = field.getName();
            AnTableField f = field.getAnnotation(AnTableField.class);
            if (f != null)
            {
                _table.addField(fieldName, f.column(), f.type(), f.notNull());
            }
            
            AnTableReplace r = field.getAnnotation(AnTableReplace.class);
            if (r != null)
            {
                _table.addReplace(fieldName, r.value());
            }
        }
        
        //3.放置到缓存
        tableMap.put(cls, _table);
        return _table;
    }
    
    /**
     * 通过表名获取表对象
     * 
     * @param tableName     表名
     * @return              表对象
     */
    public _Table getTableByTableName(String tableName)
    {
        for(_Table _table : tableMap.values())
        {
            if (tableName.equals(_table.getTable()))
                return _table;
        }
        
        //未找到则从全局找
        Collection<Class<?>> list = Global.getClassList();
        for (Class<?> clazz : list)
        {
            if (!clazz.isAnnotationPresent(AnTable.class))
                continue;
            
            _Table _table = getTable(clazz);
            if (tableName.equals(_table.getTable()))
                return _table;
        }
        
        return null;
    }
    
    /*****************************************************/
    //cache 表缓存
    /*****************************************************/
    
    /** 更新数据缓存任务安排 */
    void schedule()
    {
        zCache.schedule();
    }
    
    /**
     * 更新数据到缓存
     * 
     * @param cls           表类
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> void cache(Class<T> cls) throws ORMException, SQLException
    {
        if (!server.isCache(cls))
        {//未配置缓存的不处理
            return;
        }
        
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[cache]["+cls.getName()+"][未找到相应的配置]");
        
        //创建表当不存在时
        create(cls);
        
        //组装SQL
        StringBuilder sql = new StringBuilder("select * from ").append(_table.getTable());
        
        //执行SQL
        List<T> list = zSQL.executeQuery(sql.toString(), cls);
        zCache.cache(cls, list);
    }
    
    /*****************************************************/
    //truncate 清空表
    /*****************************************************/
    
    /**
     * 清空[表类]对应的[实际表]
     * 
     * @param cls           表类
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void truncate(Class<?> cls) throws ORMException, SQLException
    {
        truncate(cls, new HashMapSS());
    }
    
    /**
     * 清空[表类]对应的[实际表]分表，支持表名中替换字段指定为$ID$，如LOG_TRACE$ID$
     * 
     * @param clazz         表类
     * @param id            分表编号
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void truncate(Class<?> cls, String id) throws ORMException, SQLException
    {
        truncate(cls, new HashMapSS(_ID_, id));
    }
    
    /**
     * 清空[表类]对应的[实际表]，支持表名中有多个替换字段，如LOG_TRACE_$MONTH$_$ID$
     * 
     * @param cls           表类
     * @param replaceMap    替换表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void truncate(Class<?> cls, MapSS replaceMap) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[truncate]["+cls.getName()+"][未找到相应的配置]");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[truncate][可替换表对象不能为null]");
        
        zTabler.truncate(_table, replaceMap);
        
        if (server.isCache(cls))
        {//更新缓存
            cache(cls);
        }
    }
    
    /********************************************************************************************/
    //exist 表是否存在
    /********************************************************************************************/
    
    /**
     * 是否存在[表类]对应的[实际表]
     * 
     * @param cls           表类
     * @return              =true表示存在，=false表示不存在
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public boolean exist(Class<?> cls) throws ORMException, SQLException
    {
        return exist(cls, new HashMapSS());
    }

    /**
     * 是否存在[表类]对应的[实际表]分表，支持表名中替换字段指定为$ID$，如LOG_TRACE$ID$
     * 
     * @param cls           表类
     * @param id            分表编号
     * @return              =true表示存在，=false表示不存在
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public boolean exist(Class<?> cls, String id) throws ORMException, SQLException
    {
        return exist(cls, new HashMapSS(_ID_, id));
    }
    
    /**
     * 是否存在[表类]对应的[实际表]可替换表，支持表名中有多个替换字段，如LOG_TRACE_$MONTH$_$ID$
     * 
     * @param cls           表类
     * @param replaceMap    可替换字段表
     * @return              =true表示存在，=false表示不存在
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public boolean exist(Class<?> cls, MapSS replaceMap) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[exist]["+cls.getName()+"][未找到相应的配置]");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[exist][可替换表对象不能为null]");
        
        return zTabler.exist(_table, replaceMap);
    }
    
    /********************************************************************************************/
    //create 创建表
    /********************************************************************************************/
    
    /**
     * 创建[表类]对应的[实际表]
     * 
     * @param cls           表类
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void create(Class<?> cls) throws ORMException, SQLException
    {
        create(cls, new HashMapSS());
    }
    
    /**
     * 创建[表类]对应的[实际表]分表
     * 
     * @param cls           表类
     * @param id            分表编号
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void create(Class<?> cls, String id) throws ORMException, SQLException
    {
        create(cls, new HashMapSS(_ID_, id));
    }
    
    /**
     * 创建[表类]对应的[实际表]可替换表，支持表名中有多个替换字段，如LOG_TRACE_$MONTH$_$ID$
     * 
     * @param cls           表类
     * @param replaceMap    可替换字段表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void create(Class<?> cls, MapSS replaceMap) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[create]["+cls.getName()+"][未找到相应的配置]");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[create][可替换表对象不能为null]");
        
        zTabler.create(_table, replaceMap);
    }
    
    /********************************************************************************************/
    //drop 删除表
    /********************************************************************************************/
    
    /**
     * 创建[表类]对应的[实际表]
     * 
     * @param clazz         表类
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void drop(Class<?> cls) throws ORMException, SQLException
    {
        drop(cls, new HashMapSS());
    }
    
    /**
     * 创建[表类]对应的[实际表]分表
     * 
     * @param cls           表类
     * @param id            分表编号
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void drop(Class<?> cls, String id) throws ORMException, SQLException
    {
        drop(cls, new HashMapSS(_ID_, id));
    }
    
    /**
     * 创建[表类]对应的[实际表]可替换表，支持表名中有多个替换字段，如LOG_TRACE_$MONTH$_$ID$
     * 
     * @param cls           表类
     * @param replaceMap    可替换字段表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void drop(Class<?> cls, MapSS replaceMap) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[drop]["+cls.getName()+"][未找到相应的配置]");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[drop][可替换表对象不能为null]");
        
        zTabler.drop(_table, replaceMap);
    }
    
    /*****************************************************/
    //insert 插入数据
    /*****************************************************/
    
    /**
     * 增加数据,传入标准[表类]对象
     * 
     * @param data          表对象
     * @return int          表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int insert(Object data) throws ORMException, SQLException
    {
        return insert(data, new HashMapSS());
    }
    
    /**
     * 增加数据，支持表或字段中中有多个替换字段，如LOG_TRACE_$MONTH$_$ID$
     * 
     * @param data          表对象
     * @param replaceMap    适配表
     * @return              int 表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int insert(Object data, MapSS replaceMap) throws ORMException, SQLException
    {
        Class<?> cls = data.getClass();
        _Table _table = getTable(data.getClass());
        if (_table == null)
            throw new ORMException("ZTable[insert]["+cls.getName()+"][未找到相应的配置]");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[insert][可替换表对象不能为null]");
        
        //检查和尝试创建表
        _table.addReplaceToMap(data, replaceMap);
        create(cls, replaceMap);
        
        //组装SQL
        String sql = zTabler.toInsertOrReplaceSQL(_table, "insert");
        
        //执行SQL
        int ret = zSQL.executeUpdate(sql, data, replaceMap);
        
        if (ret > 0 && server.isCache(cls))
        {//更新缓存
            cache(cls);
        }
        
        return ret;
    }
    
    /**
     * 替换数据,传入标准[表类]对象和更新器（取fieldMap和replaceMap）当存在时指定修改的值
     * 
     * @param data          表对象
     * @param updater       更新器
     * @return              int 表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int insertOrUpdate(Object data, Updater updater) throws ORMException, SQLException
    {
        Class<?> clazz = data.getClass();
        _Table _table = getTable(data.getClass());
        if (_table == null)
            throw new ORMException("ZTable[insertOrUpdate]["+clazz.getName()+"][未找到相应的配置]");
        
        if (!server.isMysql())
            throw new ORMException("ZTable[insertOrUpdate]不支持当前数据库类型]");
        
        //检查和尝试创建表
        _table.addReplaceToMap(data, updater.getReplaceMap());
        create(clazz, updater.getReplaceMap());
        
        //组装字段和条件SQL和参数列
        MapSO paramMap = new HashMapSO();
        String insertSQL = toInsertOrUpdateSQL(_table, data, paramMap);
        String updateSQL = zTabler.toUpdateFieldSQL(_table, updater, paramMap);
        
        //组装SQL
        StringBuilder sql = new StringBuilder(insertSQL).append(" on duplicate key update ").append(updateSQL);
        
        //执行SQL
        int ret = zSQL.executeUpdate(sql.toString(), paramMap, updater.getReplaceMap());
        
        if (ret > 0 && server.isCache(clazz))
        {//更新缓存
            cache(clazz);
        }
        
        return ret;
    }
    
    /**
     * 批量增加数据
     * 
     * @param dataList      表对象列表
     * @return              int 表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int[] insertBatch(List<?> dataList) throws ORMException, SQLException
    {
        return insertBatch(dataList, new HashMapSS());
    }
    
    /**
     * 批量增加数据，支持表或字段中中有多个替换字段，如LOG_TRACE_$MONTH$_$ID$
     * 
     * @param dataList      表对象列表
     * @param replaceMap    适配表
     * @return              int 表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int[] insertBatch(List<?> dataList, MapSS replaceMap) throws ORMException, SQLException
    {
        if (dataList == null || dataList.isEmpty())
            return new int[0];
        
        Class<?> cls = dataList.get(0).getClass();
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[insertBatch]["+cls.getName()+"][未找到相应的配置]");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[insertBatch][可替换表对象不能为null]");
        
        //检查和尝试创建表
        _table.addReplaceToMap(dataList.get(0), replaceMap);
        zTabler.create(_table, replaceMap);
        
        //组装SQL
        String sql = zTabler.toInsertOrReplaceSQL(_table, "insert");
        
        //执行SQL
        int[] rets = zSQL.executeBatch(sql, dataList, replaceMap);
        
        if (server.isCache(cls))
        {//更新缓存
            cache(cls);
        }
        
        return rets;
    }
    
    /*****************************************************/
    //replace 替换数据，MYSQL支持，如果有数据先删除后增加
    /*****************************************************/
    
    /**
     * 替换数据,传入标准[表类]对象
     * 
     * @param data          表对象
     * @return              int 表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int replace(Object data) throws ORMException, SQLException
    {
        return replace(data, new HashMapSS());
    }
    
    /**
     * 替换数据，支持表或字段中中有多个替换字段，如LOG_TRACE_$MONTH$_$ID$
     * 
     * @param data          表对象
     * @param replaceMap    适配表
     * @return              int 表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int replace(Object data, MapSS replaceMap) throws ORMException, SQLException
    {
        Class<?> clazz = data.getClass();
        _Table _table = getTable(clazz);
        if (_table == null)
            throw new ORMException("ZTable[replace]["+clazz.getName()+"][未找到相应的配置]");
        
        if (!server.isMysql() && !server.isSqlite())
            throw new ORMException("ZTable[replace]不支持当前数据库类型]");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[replace][可替换表对象不能为null]");
        
        //检查和尝试创建表
        _table.addReplaceToMap(data, replaceMap);
        create(data.getClass(), replaceMap);
        
        //组装SQL
        String sql = zTabler.toInsertOrReplaceSQL(_table, "replace");
        
        //执行SQL
        int ret = zSQL.executeUpdate(sql, data, replaceMap);
        
        if (ret > 0 && server.isCache(clazz))
        {//更新缓存
            cache(clazz);
        }
        
        return ret;
    }
    
    /**
     * 批量替换数据
     * 
     * @param dataList      表对象列表
     * @return              int 表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int[] replaceBatch(List<?> dataList) throws ORMException, SQLException
    {
        return replaceBatch(dataList, new HashMapSS());
    }
    
    /**
     * 批量替换数据，支持表或字段中中有多个替换字段，如LOG_TRACE_$MONTH$_$ID$
     * 
     * @param dataList      表对象列表
     * @param replaceMap    适配表
     * @return              int 表示插入的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int[] replaceBatch(List<?> dataList, MapSS replaceMap) throws ORMException, SQLException
    {
        if (dataList == null || dataList.isEmpty())
            return new int[0];
        
        Class<?> clazz = dataList.get(0).getClass();
        String name = clazz.getName();
        _Table _table = getTable(clazz);
        if (_table == null)
            throw new ORMException("ZTable[replaceBatch]["+name+"][未找到相应的配置]");
        
        if (!server.isMysql() && !server.isSqlite())
            throw new ORMException("ZTable[replaceBatch]不支持当前数据库类型]");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[replaceBatch][可替换表对象不能为null]");
        
        //检查和尝试创建表
        _table.addReplaceToMap(dataList.get(0), replaceMap);
        create(dataList.get(0).getClass(), replaceMap);
        
        //组装SQL
        String sql = zTabler.toInsertOrReplaceSQL(_table, "replace");
        
        //执行SQL
        int[] rets = zSQL.executeBatch(sql, dataList, replaceMap);
        
        if (server.isCache(clazz))
        {//更新缓存
            cache(clazz);
        }
        
        return rets;
    }
    
    /*****************************************************/
    //update 更新数据
    /*****************************************************/
    
    /**
     * 更新数据，指定更新器需要更新的字段、条件和可替换表
     * 
     * @param clazz         表类
     * @param updater       更新器
     * @return              int 表示更新的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int update(Class<?> clazz, Updater updater) throws ORMException, SQLException
    {
        _Table _table = getTable(clazz);
        if (_table == null)
            throw new ORMException("ZTable[update]["+clazz.getName()+"][未找到相应的配置]");

        String result = _table.isValidUpdater(updater);
        if (result != null)
            throw new ORMException("ZTable[update]["+clazz.getName()+"][更新器字段["+result+"]类型和值要求不一致]");
        
        int ret = zTabler.update(_table, updater);
        
        if (ret > 0 && server.isCache(clazz))
        {//更新缓存
            cache(clazz);
        }
        
        return ret;
    }
    
    /**
     * 更新数据，指定对象，根据主键进行更新，主键值不更新
     * 
     * @param data          表对象
     * @return              int 表示更新的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int update(Object data) throws ORMException, SQLException
    {
        return update(data, new HashMapSS());
    }
    
    /**
     * 更新数据，指定对象，根据主键进行更新，主键值不更新，支持可替换表
     * 
     * @param data          表对象
     * @param replaceMap    适配表
     * @return              int 表示更新的条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int update(Object data, MapSS replaceMap) throws ORMException, SQLException
    {
        Class<?> clazz = data.getClass();
        _Table _table = getTable(clazz);
        if (_table == null)
            throw new ORMException("ZTable[update]["+clazz.getName()+"][未找到相应的配置]");

        if (_table.isAllKey())
            throw new ORMException("ZTable[update]["+clazz.getName()+"][所有字段都是关键属性,不支持此更新方法");
        
        if (replaceMap == null)
            throw new ORMException("ZTable[update][可替换表对象不能为null]");
        
        //检查和尝试创建表
        _table.addReplaceToMap(data, replaceMap);
        create(clazz, replaceMap);
        
        //组装SQL
        StringBuilder sql = new StringBuilder("update ").append(_table.getTable()).append(" set ");
        //field
        List<_TableField> fieldList = _table.getFieldListNoKey();
        //第一个不加逗号
        _TableField field = fieldList.get(0);
        sql.append(field.getColumn()).append("=#").append(field.getColumn()).append("#");
        //后面的加逗号
        for (int i=1;i<fieldList.size();i++)
        {
            field = fieldList.get(i);
            sql.append(", ").append(field.getColumn()).append("=#").append(field.getColumn()).append("#");
        }
        
        //where
        List<String> keyList = _table.getKeyList();
        //第一个不加 where
        StringBuilder where = new StringBuilder();
        where.append(" where ").append(keyList.get(0)).append("=#").append(keyList.get(0)).append("#");
        //后面的加 and
        for (int i=1;i<keyList.size();i++)
        {
            where.append(" and ").append(keyList.get(i)).append("=#").append(keyList.get(i)).append("#");
        }
        sql.append(where);
        
        //执行SQL
        int ret = zSQL.executeUpdate(sql.toString(), data, replaceMap);
        
        if (ret > 0 && server.isCache(clazz))
        {//更新缓存
            cache(clazz);
        }
        
        return ret;
    }
    
    /*****************************************************/
    //delete 删除数据
    /*****************************************************/
    
    /**
     * 删除数据，多个主键时使用
     * 
     * @param cls           表类
     * @param ids           关键属性为数组，多个主键
     * @return              返回删除数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int delete(Class<?> cls, Object... ids) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[delete]["+cls.getName()+"][未找到相应的配置]");
        
        if (!_table.isValidKeyObj(ids))
            throw new ORMException("ZTable[delete]["+cls.getName()+"][传入主键值类型和要求不一致]");
        
        int ret = zTabler.delete(_table, ids);
        
        if (ret > 0 && server.isCache(cls))
        {//更新缓存
            cache(cls);
        }
        
        return ret;
    }
    
    /**
     * 删除数据, 根据条件
     * 
     * @param cls           表类
     * @param selector      对象选择器
     * @return              返回删除数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int delete(Class<?> cls, Selector selector) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[delete]["+cls.getName()+"][未找到相应的配置]");

        String result = _table.isValidSelector(selector);
        if (result != null)
            throw new ORMException("ZTable[delete]["+cls.getName()+"][查询器字段["+result+"]类型和值要求不一致]");
        
        int ret = zTabler.delete(_table, selector);
        
        if (ret > 0 && server.isCache(cls))
        {//更新缓存
            cache(cls);
        }
        
        return ret;
    }
    
    /*****************************************************/
    //count 查询数目
    /*****************************************************/
    
    /**
     * 查询数目，多个主键时使用
     * 
     * @param cls           表类
     * @param ids           关键属性值
     * @return              存在的数目
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int count(Class<?> cls, Object... ids) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[count]["+cls.getName()+"][未找到相应的配置]");

        if (!_table.isValidKeyObj(ids))
            throw new ORMException("ZTable[count]["+cls.getName()+"][传入主键值和要求不一致]");
        
        return zTabler.count(_table, ids);
    }

    /**
     * 查询数目，整表查询
     * 
     * @param cls           表类
     * @return              int 数目值
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int count(Class<?> cls) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[count]["+cls.getName()+"][未找到相应的配置]");
        
        return zTabler.count(_table);
    }
    
    /**
     * 查询数目，根据条件、可替换表查询
     * 
     * @param cls           表类
     * @param selector      对象查询器
     * @return              int 数目值
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int count(Class<?> cls, Selector selector)throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[count]["+cls.getName()+"][未找到相应的配置]");
        
        String result = _table.isValidSelector(selector);
        if (result != null)
            throw new ORMException("ZTable[count]["+cls.getName()+"][查询器字段["+result+"]类型和值要求不一致]");
        
        return zTabler.count(_table, selector);
    }
    
    /*****************************************************/
    //sum 计算总和
    /*****************************************************/
    
    /**
     * 计算总和
     * 
     * @param cls           表类
     * @param field         表字段
     * @return              计算总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long sum(Class<?> cls, String field) throws ORMException, SQLException
    {
        return sum(cls, new Selector(), field);
    }
    
    /**
     * 计算总和
     * 
     * @param cls           表类
     * @param field         表字段
     * @param selector      对象查询器
     * @return              计算总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long sum(Class<?> cls, Selector selector, String field) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[sum]["+cls.getName()+"][未找到相应的配置]");
        
        String result = _table.isValidSelector(selector);
        if (result != null)
            throw new ORMException("ZTable[sum]["+cls.getName()+"][查询器字段["+result+"]类型和值要求不一致]");
        
        return zTabler.sum(_table, selector, field);
    }
    
    /**
     * 计算多个总和
     * 
     * @param cls           表类
     * @param fields        多个表字段
     * @param selector      对象查询器
     * @return              计算多个总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long[] sum(Class<?> cls, Selector selector, String... fields) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[sum]["+cls.getName()+"][未找到相应的配置]");
        
        String result = _table.isValidSelector(selector);
        if (result != null)
            throw new ORMException("ZTable[sum]["+cls.getName()+"][查询器字段["+result+"]类型和值要求不一致]");
        
        return zTabler.sum(_table, selector, fields);
    }
    
    /********************************************************************************************/
    //item 查询一条数据
    /********************************************************************************************/
    
    /**
     * 查询一个表对象，支持多个主键
     * 
     * @param cls           表类
     * @param ids           关键属性值
     * @return              返回表对象
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> cls, Object... ids) throws ORMException, SQLException
    {
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[item]["+cls.getName()+"][未找到相应的配置]");
        
        if (!_table.isValidKeyObj(ids))
            throw new ORMException("ZTable[item]["+cls.getName()+"]传参与主键要求不匹配]");
        
        //检查和尝试创建表
        create(cls);
        
        if (server.isCache(cls))
        {//从缓存中查找
            return zCache.item(cls, ids);
        }
        
        StringBuilder sql = new StringBuilder("select * from ").append(_table.getTable());

        String[] keyArr = _table.getKeyArr();
        //第一个where
        sql.append(" where ").append(keyArr[0]).append("=?");
        //后面的and
        for (int i=1;i<keyArr.length;i++)
        {
            sql.append(" and ").append(keyArr[i]).append("=?");
        }
        
        //执行SQL
        List<T> list = zSQL.executeQuery(sql.toString(), cls, ids);
        return list.isEmpty()?null:list.get(0);
    }
    
    /**
     * 查询一个表对象，并指定返回属性,查询第一行
     * 
     * @param cls           表类
     * @return              返回表对象
     * @throws ORMException 映射异常,如果传入的属性不在配置文件中则异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> cls) throws ORMException, SQLException
    {
        return item(cls, new Selector());
    }
    
    /**
     * 查询一个表对象，并指定返回属性,查询条件和排序条件
     * 
     * @param cls           表类
     * @param selector      对象查询器
     * @return              返回表对象
     * @throws ORMException 映射异常,如果传入的属性不在配置文件中则异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> cls, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZTable[item][selector]不允许为NULL]");
        
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[item]["+cls.getName()+"][未找到相应的配置]");
        
        String field = _table.isValidSelector(selector);
        if (field != null)
            throw new ORMException("ZTable[item]["+cls.getName()+"][查询器字段["+field+"]类型和值要求不一致]");
        
        //检查和尝试创建表
        create(cls, selector.getReplaceMap());
        
        if (server.isCache(cls))
        {//从缓存中查找
            return zCache.item(cls, selector);
        }
        
        //组装SQL
        MapSO paramMap = new HashMapSO();
        String fieldSQL = selector.getFieldSQL(_table);
        String whereSQL = selector.getWhereSQL(_table, paramMap);
        String orderbySQL = selector.getOrderbySQL(_table);
        String groupbySQL = selector.getGroupbySQL(_table);
        String sql = server.getPolicy().toItemSQL(fieldSQL, _table.getTable(), whereSQL, orderbySQL, groupbySQL);
        
        //执行SQL
        List<T> list = zSQL.executeQuery(sql, cls, paramMap, selector.getReplaceMap());
        if (list.isEmpty())
            return null;
        
        //检查可替换参数
        T item = list.get(0);
        _table.addReplaceToData(item, selector.getReplaceMap());
        return item;
    }

    /*****************************************************/
    //list 查询列表
    /*****************************************************/
    
    /**
     * 查询表对象列表，全表查询
     * 
     * @param cls           表类
     * @return              返回表对象列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> cls) throws ORMException, SQLException
    {
        return list(cls, new Selector());
    }
    
    /**
     * 查询表对象列表，并指定返回属性,查询条件和排序条件
     * 
     * @param cls           表类
     * @param selector      对象查询器
     * @return              返回表对象列表
     * @throws ORMException 映射异常,如果传入的属性不在配置文件中则异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> cls, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZTable[list][Selector]不允许为NULL]");
        
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[list]["+cls.getName()+"][未找到相应的配置]");
        
        String result = _table.isValidSelector(selector);
        if (result != null)
            throw new ORMException("ZTable[list]["+cls.getName()+"][查询器字段["+result+"]类型和值要求不一致]");
        
        //创建表当不存在时
        create(cls, selector.getReplaceMap());
        
        if (server.isCache(cls))
        {//从缓存中查找
            return zCache.list(cls, selector);
        }
        
        //组装SQL
        MapSO paramMap = new HashMapSO();
        String fieldSQL = selector.getFieldSQL(_table);
        String whereSQL = selector.getWhereSQL(_table, paramMap);
        String orderbySQL = selector.getOrderbySQL(_table);
        String groupbySQL = selector.getGroupbySQL(_table);
        StringBuilder sql = new StringBuilder("select ").append(fieldSQL).append(" from ").append(_table.getTable()).append(whereSQL).append(orderbySQL).append(groupbySQL);
        
        //执行SQL
        List<T> list = zSQL.executeQuery(sql.toString(), cls, paramMap, selector.getReplaceMap());
        
        //检查有没有可替换参数
        for (T item : list)
        {
            _table.addReplaceToData(item, selector.getReplaceMap());
        }
        return list;
    }
    
    /**
     * 查询表对象列表，全表查询指定位置的数据
     * 
     * @param cls           表类
     * @param pageNo        页码
     * @param pageSize      页数
     * @return              返回表对象列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> cls, int pageNo, int pageSize) throws ORMException, SQLException
    {
        return list(cls, pageNo, pageSize, new Selector());
    }
    
    /**
     * 查询表对象列表，并指定位置,查询条件和排序条件
     * 
     * @param cls           表类
     * @param pageNo        页码
     * @param pageSize      页数
     * @param selector      对象查询器
     * @return              返回表对象列表
     * @throws ORMException 映射异常,如果传入的属性不在配置文件中则异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> cls, int pageNo, int pageSize, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZTable[list][Selector]不允许为NULL]");
        
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[list]["+cls.getName()+"][未找到相应的配置]");

        String result = _table.isValidSelector(selector);
        if (result != null)
            throw new ORMException("ZTable[list]["+cls.getName()+"][查询器字段["+result+"]类型和值要求不一致]");
        
        //创建表当不存在时
        create(cls, selector.getReplaceMap());
        
        if (server.isCache(cls))
        {//从缓存中查找
            return zCache.list(cls, pageNo, pageSize, selector);
        }
        
        if (pageNo < 1) pageNo = 1;
        if (pageSize < 1) pageSize = 10;
        
        //执行SQL
        return list(cls, _table, pageNo, pageSize, selector);
    }
    
    
    /*****************************************************/
    //page 分页显示
    /*****************************************************/
    
    /**
     * 查询表对象分页信息
     * 
     * @param cls           表类
     * @param pageNo        页码
     * @param pageSize      页数
     * @return              分页信息,包括总页数,页码,页数和查询的记录
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> PageResult<T> page(Class<T> cls, int pageNo, int pageSize) throws ORMException, SQLException
    {
        return page(cls, pageNo, pageSize, new Selector());
    }

    /**
     * 查询表对象分页信息
     * 
     * @param cls           表类
     * @param pageNo        页码
     * @param pageSize      页数
     * @param selector      对象查询器
     * @return              分页信息,包括总页数,页码,页数和查询的记录
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> PageResult<T> page(Class<T> cls, int pageNo, int pageSize, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZTable[page][selector]不允许为NULL]");
        
        _Table _table = getTable(cls);
        if (_table == null)
            throw new ORMException("ZTable[page]["+cls.getName()+"][未找到相应的配置]");

        String result = _table.isValidSelector(selector);
        if (result != null)
            throw new ORMException("ZTable[page]["+cls.getName()+"][查询器字段["+result+"]类型和值要求不一致]");
        
        //创建表当不存在时
        create(cls, selector.getReplaceMap());
        
        if (server.isCache(cls))
        {//从缓存中查找
            return zCache.page(cls, pageNo, pageSize, selector);
        }
        
        if (pageNo < 1) pageNo = 1;
        if (pageSize < 1) pageSize = 10;
        
        int totalRecord = count(cls, selector);
        if (totalRecord == 0)
            return PageBuilder.newResult(pageNo, pageSize);
        
        //检查where
        int maxPage = (totalRecord-1) / pageSize + 1;
        if (pageNo > maxPage) pageNo = maxPage;
        
        //执行SQL
        List<T> resultSet = list(cls, _table, pageNo, pageSize, selector);
        return PageBuilder.newResult(totalRecord, pageNo, pageSize, resultSet);
    }
    
    /**************************************************************************/
    //内部方法
    /**************************************************************************/
    
    /** 
     * 内部方法，组装插入替换SQL 
     * 1.insert & insertOrUpdate & insertBatch
     * 2.replace & replaceBatch
     * 调用 
     */
    private String toInsertOrUpdateSQL(_Table _table, Object data, MapSO paramMap) throws ORMException
    {
        Class<?> clazz = data.getClass();
        Dbo dbo = server.getDbo(clazz);
        
        //组装SQL
        StringBuilder sql = new StringBuilder("insert into ").append(_table.getTable()).append("(");
        StringBuilder values = new StringBuilder("values (");
        List<_TableField> fieldList = _table.getFieldList();
        for (_TableField field : fieldList)
        {
            String fieldName = field.getField();
            String method = dbo.getGetMethod(fieldName);
            if (method == null)
                throw new ORMException("_Table/_View/_Dbo["+clazz.getName()+"]没有找到"+fieldName+"对应的方法]");
            
            try
            {//取值放置到MAP中，以fieldName方式
                Method m = clazz.getMethod(method, new Class[0]);
                Object value = m.invoke(data, new Object[0]);
                paramMap.put(fieldName, value);
                sql.append(field.getColumn()).append(",");
                values.append("#").append(fieldName).append("#").append(",");
            }
            catch (Exception e)
            {
                throw new ORMException(e);
            }
        }
        
        //去除最后一个逗号
        sql.setLength(sql.length()-1);
        values.setLength(values.length()-1);
        
        //最后加结尾
        sql.append(") ").append(values).append(")");
        
        return sql.toString();
    }
    
    /** 内部方法，list & page 调用 */
    private <T> List<T> list(Class<T> clazz, _Table _table, int pageNo, int pageSize, Selector selector) throws ORMException, SQLException
    {
        int minNum = (pageNo-1) * pageSize + 1;
        int maxNum = pageNo * pageSize;
        
        MapSO paramMap = new HashMapSO();
        paramMap.put("minNum", minNum);
        paramMap.put("maxNum", maxNum);
        paramMap.put("minSize", minNum - 1);
        paramMap.put("pageSize", pageSize);
        
        //组装SQL
        String fieldSQL = selector.getFieldSQL(_table);
        String whereSQL = selector.getWhereSQL(_table, paramMap);
        String orderbySQL = selector.getOrderbySQL(_table);
        String groupbySQL = selector.getGroupbySQL(_table);
        String sql = server.getPolicy().toPageSQL(fieldSQL, _table.getTable(), whereSQL, orderbySQL, groupbySQL, maxNum, pageNo);
        
        //执行SQL
        List<T> list = zSQL.executeQuery(sql, clazz, paramMap, selector.getReplaceMap());
        
        //检查有没有可替换参数
        for (T item : list)
        {
            _table.addReplaceToData(item, selector.getReplaceMap());
        }
        return list;
    }
}
