/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.zhiqim.kernel.extend.HashMapCO;
import org.zhiqim.kernel.extend.HashMapCV;
import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.paging.PageBuilder;
import org.zhiqim.kernel.paging.PageResult;
import org.zhiqim.kernel.schedule.Scheduler;
import org.zhiqim.kernel.schedule.TaskThreader;
import org.zhiqim.kernel.util.Classes;
import org.zhiqim.kernel.util.Lists;
import org.zhiqim.kernel.util.Objects;
import org.zhiqim.orm.dbo.Comparator;
import org.zhiqim.orm.dbo.Condition;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.orm.dbo.defined._Table;

/**
 * 表缓存
 *
 * @version v1.0.0 @author zhichenggang 2017-7-1 新建与整理
 */
final class ZTableCache
{
    private static final Log log = LogFactory.getLog(ZTableCache.class);
    
    private final ORMServer server;
    private final HashMapCO lockMap;
    private final HashMapCV<List<?>> cache;
    private final Scheduler scheduler;
    
    public ZTableCache(ORMServer server)
    {
        this.server = server;
        this.lockMap = new HashMapCO();
        this.cache = new HashMapCV<>();
        
        //增加计划任务
        this.scheduler = new Scheduler();
        this.scheduler.create();
    }
    
    /** 安排任务 */
    public void schedule()
    {
        for (TaskThreader task : server.getCacheMap().values())
        {
            this.scheduler.addTask(task);
        }
    }
    
    /** 缓存回调刷新 */
    public <T> void cache(Class<?> clazz, List<T> list) throws ORMException, SQLException
    {
        Object lock = lock(clazz);
        synchronized (lock)
        {//表内锁
            List<?> oList = cache.remove(clazz);
            if (oList != null)
            {
                oList.clear();
                oList = null;
            }
            
            cache.put(clazz, list);
        }
        
        log.info("刷新缓存[%s]", clazz.getName());
    }
    
    /********************************************************************************************/
    //item 查询一条数据
    /********************************************************************************************/
    
    /**
     * 查询一个表对象，支持多个主键
     * 
     * @param clazz         表类
     * @param ids           关键属性值
     * @return              返回表对象
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> clazz, Object... ids) throws ORMException, SQLException
    {
        _Table _table = ((ZTableImplement)server.table()).getTable(clazz);
        String[] keys = _table.getKeyArr();
        Selector selector = new Selector();
        for (int i=0;i<keys.length;i++)
        {
            String field = _table.getField(keys[i]).getField();
            selector.addMust(field, ids[i]);
        }
        
        return item(clazz, selector);
    }
    
    /**
     * 查询一个表对象，并指定返回属性,查询第一行
     * 
     * @param clazz         表类
     * @return              返回表对象
     * @throws ORMException 映射异常,如果传入的属性不在配置文件中则异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> clazz) throws ORMException, SQLException
    {
        List<T> list = fetch(clazz, null);
        if (list.isEmpty())
            return null;
        
        return Objects.copy(list.get(0), Classes.newInstance(clazz));
    }
    
    /**
     * 查询一个表对象，并指定返回属性,查询条件和排序条件
     * 
     * @param clazz         表类
     * @param selector      对象查询器
     * @return              返回表对象
     * @throws ORMException 映射异常,如果传入的属性不在配置文件中则异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> clazz, Selector selector) throws ORMException, SQLException
    {
        List<T> list = fetch(clazz, selector);
        if (list.isEmpty())
            return null;
        
        if (!selector.hasCondition())
            return Objects.copy(list.get(0), Classes.newInstance(clazz));
    
        for (T item : list)
        {
            if (match(item, selector))
                return Objects.copy(item, Classes.newInstance(clazz));
        }
        
        return null;
    }
    
    /********************************************************************************************/
    //list 查询列表
    /********************************************************************************************/
    
    /**
     * 查询表对象列表，全表查询
     * 
     * @param clazz         表类
     * @return              返回表对象列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> clazz) throws ORMException, SQLException
    {
        return copyList(clazz, fetch(clazz, null));
    }
    
    /**
     * 查询表对象列表，并指定返回属性,查询条件和排序条件
     * 
     * @param clazz         表类
     * @param selector      对象查询器
     * @return              返回表对象列表
     * @throws ORMException 映射异常,如果传入的属性不在配置文件中则异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> clazz, Selector selector) throws ORMException, SQLException
    {
        List<T> list = fetch(clazz, selector);
        if (list.isEmpty())
            return list;
        
        if (!selector.hasCondition())
            return copyList(clazz, list);
        
        ArrayList<T> sList = new ArrayList<>();
        for (T item : list)
        {
            if (match(item, selector))
                sList.add(Objects.copy(item, Classes.newInstance(clazz)));
        }
        
        return Lists.trim(sList);
    }
    
    /**
     * 查询表对象列表，查询指定的位置的数据
     * 
     * @param clazz         表类
     * @param pageNo        页码
     * @param pageSize      页数
     * @return              返回表对象列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> clazz, int pageNo, int pageSize) throws ORMException, SQLException
    {
        List<T> list = fetch(clazz, null);
        if (list.isEmpty())
            return list;
        
        return PageBuilder.pageList(pageNo, pageSize, copyList(clazz, list));
    }
    
    /**
     * 查询表对象列表，并指定位置的,条件和排序条件
     * 
     * @param clazz         表类
     * @param pageNo        页码
     * @param pageSize      页数
     * @param selector      对象查询器
     * @return              返回表对象列表
     * @throws ORMException 映射异常,如果传入的属性不在配置文件中则异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> clazz, int pageNo, int pageSize, Selector selector) throws ORMException, SQLException
    {
        List<T> list = fetch(clazz, selector);
        if (list.isEmpty())
            return list;
        
        if (!selector.hasCondition())
            return PageBuilder.pageList(pageNo, pageSize, copyList(clazz, list));
        
        ArrayList<T> sList = new ArrayList<>();
        for (T item : list)
        {
            if (match(item, selector))
                sList.add(Objects.copy(item, Classes.newInstance(clazz)));
        }
        
        return PageBuilder.pageList(pageNo, pageSize, sList);
    }
    
    /********************************************************************************************/
    //page 分页显示
    /********************************************************************************************/
    
    /**
     * 查询表对象分页信息
     * 
     * @param clazz         表类
     * @param pageNo        页码
     * @param pageSize      页数
     * @return              分页信息,包括总页数,页码,页数和查询的记录
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> PageResult<T> page(Class<T> clazz, int pageNo, int pageSize) throws ORMException, SQLException
    {
        List<T> list = fetch(clazz, null);
        if (list.isEmpty())
            return PageBuilder.newResult(0, pageNo, pageSize, list);
        
        return PageBuilder.pageResult(pageNo, pageSize, copyList(clazz, list));
    }

    /**
     * 查询表对象分页信息
     * 
     * @param clazz         表类
     * @param pageNo        页码
     * @param pageSize      页数
     * @param selector      对象查询器
     * @return              分页信息,包括总页数,页码,页数和查询的记录
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> PageResult<T> page(Class<T> clazz, int pageNo, int pageSize, Selector selector) throws ORMException, SQLException
    {
        List<T> list = fetch(clazz, selector);
        if (list.isEmpty())
            return PageBuilder.newResult(0, pageNo, pageSize, list);
        
        if (!selector.hasCondition())
            return PageBuilder.pageResult(pageNo, pageSize, copyList(clazz, list));
        
        ArrayList<T> sList = new ArrayList<>();
        for (T item : list)
        {
            if (match(item, selector))
                sList.add(Objects.copy(item, Classes.newInstance(clazz)));
        }
        
        return PageBuilder.pageResult(pageNo, pageSize, sList);
    }
    
    /********************************************************************************************/
    //私用共享方
    /********************************************************************************************/
    
    /** 抓取列表，如果没有缓存则从数据库读取一次 */
    @SuppressWarnings("unchecked")
    private <T> List<T> fetch(Class<T> clazz, Selector selector) throws ORMException, SQLException
    {
        List<T> list = (List<T>) cache.get(clazz);
        if (list == null)
        {//第一次没有数据，执行更新缓存操作
            server.table().cache(clazz);
            
            //然后再查一次
            list = (List<T>) cache.get(clazz);
        }
        
        if (selector == null || !selector.hasOrderby())
        {//无需排序，返回当前列表即可
            return list;
        }
        else
        {//保持原列表不动，新建列表，再排序，最后去除多余的占用空间
            ArrayList<T> nList = new ArrayList<T>(list);
            Collections.sort(nList, new Comparator<>(selector));
            return Lists.trim(nList);
        }
    }
    
    /** 匹配对象 */
    private <T> boolean match(T item, Selector selector)
    {
        for (Condition c : selector.getConditionList())
        {
            if (!c.match(item))
                return false;
        }
        
        return true;
    }
    
    /** 读取表锁 */
    private Object lock(Class<?> clazz)
    {
        synchronized (this)
        {
            Object lock = lockMap.get(clazz);
            if (lock == null)
            {
                lock = new Object();
                lockMap.put(clazz, lock);
            }
            return lock;
        }
    }
    
    /** 拷贝列表，数据全部重新创建 */
    private <T> List<T> copyList(Class<T> clazz, List<T> list)
    {
        if (list.isEmpty())
            return list;
        
        ArrayList<T> cList = new ArrayList<>(list.size());
        for (T item : list)
        {
            T cItem = Objects.copy(item, Classes.newInstance(clazz));
            cList.add(cItem);
        }
        
        return Lists.trim(cList);
    }
}
