/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm;

import java.sql.SQLException;

import org.zhiqim.kernel.MultiInstancer;
import org.zhiqim.kernel.util.Validates;

/**
 * SQL(DML)调用实现<br><br>
 *                      
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
class ZDDLImplement extends MultiInstancer implements ZDDL, ORMConstants
{
    private ORMServer server;
    private ZSQL zSQL;
    
    public ZDDLImplement(ORMServer server)
    {
        this.server = server;
        this.zSQL = server.sql();
    }
    
    /**
     * 获取ORM服务器
     * 
     * @return              数据库服务器
     */
    public ORMServer getServer()
    {
        return server;
    }
    
    /**
     * ORM定义的类型，转化为SQL类型，如string,15转化为varchar(15)
     * 
     * @param type          ORM定义的类型如string,15
     * @return              SQL类型，如varchar(15)
     */
    public String toColumnType(String type)
    {
        return server.getPolicy().toColumnType(type);
    }
    
    /**
     * ORM定义的类型，转化为SQL类型，如string,15转化为varchar(15)
     * 
     * @param fieldType     ORM定义的类型如string,int
     * @param fieldLength   ORM定义的类型长度，string和decimal时有效，如19,char | 10,2
     * @return              SQL类型，如varchar(15)
     */
    public String toColumnType(String fieldType, String fieldLength)
    {
        return toColumnType(Validates.isEmptyBlank(fieldLength)?fieldType:(fieldType+","+fieldLength));
    }
    
    /*****************************************************/
    //alter DML语句
    /*****************************************************/
    
    /**
     * 修改一个表的名称
     * 
     * @param table             表名
     * @param newTable          新表名
     * @throws SQLException     数据库异常
     */
    public void alterTableName(String table, String newTable) throws SQLException
    {
        zSQL.execute(new StringBuilder("alter table ").append(table).append(" rename to ").append(newTable.toUpperCase()).toString());
    }
    
    /**
     * 增加一个表的一列
     * 
     * @param table             表名
     * @param column            列名
     * @param columnType        列类型
     * @param notNull           是否允许为null
     * @throws SQLException     数据库异常
     */
    public void alterColumnAdd(String table, String column, String columnType, boolean notNull) throws SQLException
    {
        zSQL.execute(server.getPolicy().toAlertColumnAdd(table, column, columnType, notNull));
    }
    
    
    /**
     * 删除一个表的一列
     * 
     * @param table             表名
     * @param column            列名
     * @throws SQLException     数据库异常
     */
    public void alterColumnDrop(String table, String column) throws ORMException, SQLException
    {
        server.getPolicy().executeAlertColumnDrop(table, column);
    }
    
    /**
     * 修改一个表的一列信息
     * 
     * @param table             表名
     * @param column            列名
     * @param newColumn         新列名
     * @param newColumnType     新列类型
     * @param newNotNull        新列是否不为null
     * @throws SQLException     数据库异常
     */
    public void alterColumnInfo(String table, String column, String newColumn, String newColumnType, boolean newNotNull) throws ORMException, SQLException
    {
        if (column.equalsIgnoreCase(newColumn))
        {
            alterColumnType(table, column, newColumnType, newNotNull);
            return;
        }
        
        server.getPolicy().executeAlertColumnInfo(table, column, newColumn, newColumnType, newNotNull);
    }
    
    /**
     * 修改一个表的列名
     * 
     * @param table             表名
     * @param column            列名
     * @param newColumn         新列名
     * @throws SQLException     数据库异常
     */
    public void alterColumnName(String table, String column, String newColumn) throws ORMException, SQLException
    {
        server.getPolicy().executeAlertColumnName(table, column, newColumn);
    }
    
    /**
     * 修改一个表的一列类型
     * 
     * @param table             表名
     * @param column            列名
     * @param newType           新类型
     * @throws SQLException     数据库异常
     */
    public void alterColumnType(String table, String column, String newColumnType, boolean newNotNull) throws ORMException, SQLException
    {
        server.getPolicy().executeAlertColumnType(table, column, newColumnType, newNotNull);
    }
    
    /**
     * 增加表的主键
     * 
     * @param table             表名
     * @param columns           列名，多个用逗号隔开如AAA_AAA,BBB_BBB
     * @throws SQLException     数据库异常
     */
    public void alertPrimaryKeyAdd(String table, String columns) throws ORMException, SQLException
    {
        server.getPolicy().executeAlertPrimaryKeyAdd(table, columns);
    }
    
    /**
     * 删除表的主键
     * 
     * @param table             表名
     * @throws SQLException     数据库异常
     */
    public void alertPrimaryKeyDrop(String table) throws ORMException, SQLException
    {
        server.getPolicy().executeAlertPrimaryKeyDrop(table);
    }
}
