/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm;

import java.io.File;

import org.zhiqim.kernel.Global;
import org.zhiqim.kernel.config.Group;
import org.zhiqim.kernel.config.Item;
import org.zhiqim.kernel.extend.HashMapSS;
import org.zhiqim.kernel.extend.HashMapSV;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Resources;
import org.zhiqim.kernel.util.Systems;
import org.zhiqim.kernel.util.codes.Base64;
import org.zhiqim.kernel.util.codes.HEX;
import org.zhiqim.orm.datasource.ZDataSource;

/**
 * ORM服务需要配置的参数
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class ORMParameter implements ORMConstants
{
    //必填五项
    private String dbType;
    private String driver;
    private String url;
    private String user;
    private String pass;
    
    //常用配置三项
    private int minPoolSize = 2;
    private int maxPoolSize = 10;
    private int maxKeepTime = 7001;
    
    //连接时效两项
    private int maxIdleTime = 7001;
    private int maxCompletedCount = 100001;
    
    //密码类型
    private String passType = null;

    //检查连接三项
    private boolean isChkConnOnTimer = false;
    private boolean isChkConnOnGet = false;
    private boolean isChkConnOnRelease = false;
    
    //耗尽重试两项
    private int outOfConnWaitTime = 5;
    private int outOfConnRetryCount = 1;
    
    //SQL输出日志两项
    private boolean isUpdateSqlLog = false;
    private boolean isQuerySqlLog = false;
    
    //sql文件表
    private HashMapSV<Integer> sqlMap = new HashMapSV<>();
    private HashMapSS cacheMap = new HashMapSS();
    
    /***********************************************************************************/
    //构造函数
    /***********************************************************************************/
    
    /** 空构造，然后手动设置参数 */
    public ORMParameter()
    {
    }
    
    /** 通过配置生成参数对象 */
    public ORMParameter(Group group)
    {
        //5项必须
        this.dbType = group.getString("dbType");
        this.driver = group.getString("driver");
        this.url = group.getString("url");
        this.user = group.getString("user");
        this.pass = group.getString("pass");
        
        //其他有默认值
        this.passType = group.getString("passType");
        this.minPoolSize = group.getInt("minPoolSize", 2);
        this.maxPoolSize = group.getInt("maxPoolSize", 10);
        
        this.maxKeepTime = group.getInt("maxKeepTime", 7001);
        this.maxIdleTime = group.getInt("maxIdleTime", this.maxKeepTime);
        this.maxCompletedCount = group.getInt("maxCompletedCount", 100001);
        this.isChkConnOnTimer = group.isTrue("isChkConnOnTimer", false);
        this.isChkConnOnGet = group.isTrue("isChkConnOnGet", false);
        this.isChkConnOnRelease = group.isTrue("isChkConnOnRelease", false);
        
        this.outOfConnWaitTime = group.getInt("outOfConnWaitTime", 5);
        this.outOfConnRetryCount = group.getInt("outOfConnRetryCount", 1);
        
        this.isUpdateSqlLog = group.isTrue("isUpdateSqlLog", false);
        this.isQuerySqlLog = group.isTrue("isQuerySqlLog", false);
        
        //SQL文件
        Group grp = Global.getGroup(group.getId()+".sql");
        if (grp != null)
        {
            for (Item item : grp.list())
            {
                addSqlConfig(item.getKey(), item.getString());
            }
        }
        
        //表缓存
        grp = Global.getGroup(group.getId()+".cache");
        if (grp != null)
        {
            for (Item item : grp.list())
            {
                addCacheConfig(item.getKey(), item.getString());
            }
        }
    }
    
    /** 初始化连接池 */
    public ZDataSource newDatabase()
    {
        //对密码支持简单加密，HEX/Base64两种
        if (_HEX_.equalsIgnoreCase(passType))
            pass = HEX.decrypt(pass);
        else if (_BASE64_.equalsIgnoreCase(passType))
            pass = Base64.decodeUTF8(pass);
            
        return new ZDataSource(driver, url, user, pass, 
            minPoolSize, maxPoolSize, maxKeepTime, maxIdleTime, maxCompletedCount, 
            isChkConnOnTimer, isChkConnOnGet, isChkConnOnRelease, 
            outOfConnWaitTime, outOfConnRetryCount);
    }
    
    /***********************************************************************************/
    //判断数据库类型
    /***********************************************************************************/
    
    /** 获取数据库类型 */
    public int getDatabaseType()
    {
        return ORMType.getDatabaseType(dbType).value();
    }
    
    /** 判断是否是Oracle数据库 */
    public boolean isOracle()
    {
        return getDatabaseType() == Z_ORM_ORACLE.value();
    }
    
    /** 判断是否是Microsoft SQLServer数据库 */
    public boolean isMssql()
    {
        return getDatabaseType() == Z_ORM_MSSQL.value();
    }
    
    /** 判断是否是SQLite数据库 */
    public boolean isSqlite()
    {
        return getDatabaseType() == Z_ORM_SQLITE.value();
    }
    
    /** 判断是否是Hsql数据库 */
    public boolean isHsql()
    {
        return getDatabaseType() == Z_ORM_HSQL.value();
    }
    
    /** 判断是否是MySql数据库 */
    public boolean isMysql()
    {
        return getDatabaseType() == Z_ORM_MYSQL.value();
    }
    
    /** 获取数据库名 */
    public String getDbName()
    {
        if (isMysql())
        {
            if ("jndi".equalsIgnoreCase(driver))
                return user;//JNDI 时配置成用户
            
            String dbUrl = url;
            int ind = dbUrl.indexOf("?");
            if (ind != -1)
                dbUrl = dbUrl.substring(0, ind);
            
            int ind2 = dbUrl.lastIndexOf("/");
            return dbUrl.substring(ind2+1);
        }
        
        return null;
    }
    
    /***********************************************************************************/
    //SQL文件表
    /***********************************************************************************/
    
    public HashMapSV<Integer> getSqlConfig()
    {
        return sqlMap;
    }
    
    public void addSqlConfig(String key, String value)
    {
        int type = Z_SQL_FILE.equals(value)?Z_SQL_FILE_INT:
                   Z_SQL_FOLDER.equals(value)?Z_SQL_FOLDER_INT:
                   Z_SQL_CLASS.equals(value)?Z_SQL_CLASS_INT:
                   Z_SQL_PACKAGE.equals(value)?Z_SQL_PACKAGE_INT:-1;
        
        addSqlConfig(key, type);
    }
    
    public void addSqlConfig(String key, int type)
    {
        Asserts.notNull(key, _KEY_);
        Asserts.as((type >= 0 && type <= 3)?null:"增加ORM的SQL配置["+key+"]的类型不正确");
        
        switch (type)
        {
        case Z_SQL_FILE_INT:
        {
            Asserts.as(key.endsWith(Z_SQL_ENDSWITH)?null:"增加ORM的SQL配置["+key+"]必须以[.sql.xml]结尾的SQL配置文件名");
            
            File file = new File(key);
            Asserts.as((file.isFile() && file.canRead())?null:"增加ORM的SQL配置["+key+"]必须是文件并可读");
            break;
        }
        case Z_SQL_FOLDER_INT:
        {
            File folder = new File(key);
            Asserts.as((folder.isDirectory() && folder.canRead())?null:"增加ORM的SQL配置["+key+"]必须是文件夹并可读");
            break;
        }
        case Z_SQL_CLASS_INT:
        {
            Asserts.as(key.endsWith(Z_SQL_ENDSWITH)?null:"增加ORM的SQL配置["+key+"]必须以[.sql.xml]结尾的SQL配置文件名");
            Asserts.as(Resources.exists(ORMServer.class, key)?null:"增加ORM的SQL配置["+key+"]必须是文件并可读");
            break;
        }
        case Z_SQL_PACKAGE_INT:
        {
            Asserts.as(Resources.exists(ORMServer.class, key)?null:"增加ORM的SQL配置["+key+"]必须是存在");
            break;
        }
        }
        
        key = Systems.replacePropertyPath(key);
        sqlMap.put(key, type);
    }
    
    /***********************************************************************************/
    //Cache配置表
    /***********************************************************************************/
    
    public void addCacheConfig(String key, String value)
    {
        cacheMap.put(key, value);
    }
    
    public HashMapSS getCacheConfig()
    {
        return cacheMap;
    }
    
    /***********************************************************************************/
    //参数的设置和获取
    /***********************************************************************************/
    
    public String getDbType()
    {
        return dbType;
    }
    
    public void setDbType(String dbType)
    {
        this.dbType = dbType;
    }
    
    public boolean isUpdateSqlLog()
    {
        return isUpdateSqlLog;
    }
    
    public void setUpdateSqlLog(boolean isUpdateSqlLog)
    {
        this.isUpdateSqlLog = isUpdateSqlLog;
    }
    
    public boolean isQuerySqlLog()
    {
        return isQuerySqlLog;
    }
    
    public void setQuerySqlLog(boolean isQuerySqlLog)
    {
        this.isQuerySqlLog = isQuerySqlLog;
    }
    
    public String getDriver()
    {
        return driver;
    }
    
    public void setDriver(String driver)
    {
        this.driver = driver;
    }
    
    public String getUrl()
    {
        return url;
    }
    
    public void setUrl(String url)
    {
        this.url = url;
    }
    
    public String getUser()
    {
        return user;
    }
    
    public void setUser(String user)
    {
        this.user = user;
    }
    
    public String getPass()
    {
        return pass;
    }
    
    public void setPass(String pass)
    {
        this.pass = pass;
    }
    
    public String getPassType()
    {
        return passType;
    }
    
    public void setPassType(String passType)
    {
        this.passType = passType;
    }
    
    public int getMinPoolSize()
    {
        return minPoolSize;
    }
    
    public void setMinPoolSize(int minPoolSize)
    {
        this.minPoolSize = minPoolSize;
    }
    
    public int getMaxPoolSize()
    {
        return maxPoolSize;
    }
    
    public void setMaxPoolSize(int maxPoolSize)
    {
        this.maxPoolSize = maxPoolSize;
    }
    
    public int getMaxKeepTime()
    {
        return maxKeepTime;
    }
    
    public void setMaxKeepTime(int maxKeepTime)
    {
        this.maxKeepTime = maxKeepTime;
    }
    
    public int getMaxIdleTime()
    {
        return maxIdleTime;
    }
    
    public void setMaxIdleTime(int maxIdleTime)
    {
        this.maxIdleTime = maxIdleTime;
    }
    
    public int getMaxCompletedCount()
    {
        return maxCompletedCount;
    }
    
    public void setMaxCompletedCount(int maxCompletedCount)
    {
        this.maxCompletedCount = maxCompletedCount;
    }
    
    public boolean isChkConnOnTimer()
    {
        return isChkConnOnTimer;
    }
    
    public void setChkConnOnTimer(boolean isChkConnOnTimer)
    {
        this.isChkConnOnTimer = isChkConnOnTimer;
    }
    
    public boolean isChkConnOnGet()
    {
        return isChkConnOnGet;
    }
    
    public void setChkConnOnGet(boolean isChkConnOnGet)
    {
        this.isChkConnOnGet = isChkConnOnGet;
    }
    
    public boolean isChkConnOnRelease()
    {
        return isChkConnOnRelease;
    }
    
    public void setChkConnOnRelease(boolean isChkConnOnRelease)
    {
        this.isChkConnOnRelease = isChkConnOnRelease;
    }
    
    public int getOutOfConnWaitTime()
    {
        return outOfConnWaitTime;
    }
    
    public void setOutOfConnWaitTime(int outOfConnWaitTime)
    {
        this.outOfConnWaitTime = outOfConnWaitTime;
    }
    
    public int getOutOfConnRetryCount()
    {
        return outOfConnRetryCount;
    }
    
    public void setOutOfConnRetryCount(int outOfConnRetryCount)
    {
        this.outOfConnRetryCount = outOfConnRetryCount;
    }
}
