/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.util;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.constants.RegexpConstants;
import org.zhiqim.kernel.constants.SignConstants;

/**
 * 基础Script，从Scripts提取出来
 *
 * @version v1.0.0 @author zouzhigang 2016-6-24 新建与整理
 */
@AnAlias("Scriptx")
public class Scriptx implements RegexpConstants, SignConstants
{
    /**
     * 构造正则表达式函数
     * 
     * @param function      指定函数名称
     * @param regexp        正则表达式
     * @param allowEmpty    是否允许为空
     * @return              得到指定函数名称的函数内容
     */
    public static String isRegexp(String function, String regexp, boolean allowEmpty)
    {
        if (!regexp.startsWith("/"))
            regexp = "/" + regexp;
        
        if (!regexp.endsWith("/"))
            regexp = regexp + "/";
        
        StringBuilder strb = new StringBuilder()
            .append("function ").append(function).append("(value)").append(_BR_)
            .append("{").append(_BR_)
            .append(_FOUR_).append("if (value === undefined || value == '' || value.length < 1)").append(_BR_);
        if (allowEmpty)
            strb.append(_FOUR_).append(_FOUR_).append("return true;").append(_BR_);
        else
            strb.append(_FOUR_).append(_FOUR_).append("return false;").append(_BR_);
        
        strb.append(_BR_)
            .append(_FOUR_).append("var reg = ").append(regexp).append(";").append(_BR_)
            .append(_FOUR_).append("return reg.test(value);").append(_BR_)
            .append("}").append(_BR_).append(_BR_);
    
        return strb.toString();
    }
    
    /**
     * 是否符号正则表达式
     * 
     * @param regexp        正则表达式
     * @param allowEmpty    是否允许为空
     * @return              得到指定函数名称的函数内容
     */
    public static String isRegexp(String regexp, boolean allowEmpty)
    {
        return isRegexp(IS_REGEXP, regexp, allowEmpty);
    }
    
    /**
     * 是否是数字
     * 
     * @param allowEmpty    是否允许为空
     * @return isNumeric
     */
    public static String isNumeric(boolean allowEmpty)
    {
        return isRegexp(IS_NUMERIC, NUMERIC, allowEmpty);
    }
    
    /**
     * 是否是整数，包括正整数、负整数、和0，支持-号、不支持+号
     * 
     * @param allowEmpty    是否允许为空
     * @return isInteger
     */
    public static String isInteger(boolean allowEmpty)
    {
        return isRegexp(IS_INTEGER, INTEGER, allowEmpty);
    }
    
    /**
     * 是否是字母和数字组合
     * 
     * @param allowEmpty    是否允许为空
     * @return isAlphaNumeric
     */
    public static String isAlphaNumeric(boolean allowEmpty)
    {
        return isRegexp(IS_ALPHA_NUMERIC, ALPHA_NUMERIC, allowEmpty);
    }
    
    /**
     * 是否是字母和数字组合，字母开头，且指定长度大小
     * 
     * @param allowEmpty    是否允许为空
     * @param min           最小长度
     * @param max           最大长度
     * @return isAlphaNumeric
     */
    public static String isAlphaNumericLenPA(boolean allowEmpty, int min, int max)
    {
        if (min < 1) min = 1;if (max < min) max = min;
        String regexp = ALPHA_NUMERIC_PA_N_M.replaceFirst("n", ""+(min-1)).replaceFirst("m", ""+(max-1));
        return isRegexp(IS_ALPHA_NUMERIC_PA_LEN, regexp, allowEmpty);
    }
    
    /**
     * 是否浮点型，支持多位小数，支持负数，如-12,-12.3,-12.35,-12.356,12,12.3,12.35,0,0.3,0.35,-0.3,-0.35，不支持+12,+12,35之类的前缀+号
     * 
     * @param allowEmpty    是否允许为空
     * @return isFloat
     */
    public static String isFloat(boolean allowEmpty)
    {
        return isRegexp(IS_FLOAT, FLOAT, allowEmpty);
    }
    
    /**
     * 是否是最长两位小数的金额，支持负数，如-12,-12.3,-12.35,12,12.3,12.35,0,0.3,0.35,-0.3,-0.35，不支持+12,+12,35之类的前缀+号
     * 
     * @param allowEmpty    是否允许为空
     * @return isAmount2R
     */
    public static String isAmount2R(boolean allowEmpty)
    {
        return isRegexp(IS_AMOUNT_2R, AMOUNT_2R, allowEmpty);
    }
    
    /**
     * 是否是邮箱地址，支持类似于xxx@xxx.xxx的邮箱格式，支持[-_.]等连接符，不支持中文等特殊符号
     * 
     * @param allowEmpty    是否允许为空
     * @return isEmail
     */
    public static String isEmail(boolean allowEmpty)
    {
        return isRegexp(IS_EMAIL, EMAIL, allowEmpty);
    }
    
    /**
     * 是否是IP地址，支持类似于192.168.1.1/192.168.01.01/192.168.001.001的IP格式
     * 
     * @param allowEmpty    是否允许为空
     * @return isIP
     */
    public static String isIP(boolean allowEmpty)
    {
        return isRegexp(IS_IP, IP, allowEmpty);
    }
    
    /**
     * 是否是MAC地址，格式为23:ad:87:be(十六进制)
     * 
     * @param allowEmpty    是否允许为空
     * @return isIP
     */
    public static String isMacAddress(boolean allowEmpty)
    {
        return isRegexp(IS_MAC, MAC, allowEmpty);
    }
    
    /**
     * 是否是泛手机号码，支持前缀+86,86开头,后面接手机号
     * 
     * @param allowEmpty    是否允许为空
     * @return isMsisdn
     */
    public static String isMobile(boolean allowEmpty)
    {
        return isRegexp(IS_MOBILE, MOBILE, allowEmpty);
    }
    
    /**
     * 是否是中国区11位的手机号码
     * 
     * @param allowEmpty    是否允许为空
     * @return isMsisdn
     */
    public static String isMobile11(boolean allowEmpty)
    {
        return isRegexp(IS_MOBILE11, MOBILE_11, allowEmpty);
    }
    
    /**
     * 是否是日期格式，格式为yyyy-MM-dd
     * 
     * @param allowEmpty    是否允许为空
     * @return isDate
     */
    public static String isDate(boolean allowEmpty)
    {
        return isRegexp(IS_DATE, DATE, allowEmpty);
    }
    
    /**
     * 是否是时间格式，格式为HH:mm:ss
     * 
     * @param allowEmpty    是否允许为空
     * @return isTime
     */
    public static String isTime(boolean allowEmpty)
    {
        return isRegexp(IS_TIME, TIME, allowEmpty);
    }
    
    /**
     * 是否是日期时间格式，格式为yyyy-MM-dd HH:mm:ss
     * 
     * @param allowEmpty    是否允许为空
     * @return isDateTime
     */
    public static String isDateTime(boolean allowEmpty)
    {
        return isRegexp(IS_DATE_TIME, DATE_TIME, allowEmpty);
    }
    
    /**
     * 是否是身份证号，格式为[0-9]{6}yyyyMMdd[0-9]{3}[0-9xX*]，前6位数字，中间8位1900-2099年的日期，末4位为3个数字和[0-9]或xX*。
     * 
     * @param allowEmpty    是否允许为空
     * @return isIdcard
     */
    public static String isIdcard(boolean allowEmpty)
    {
        return isRegexp(IS_DATE_TIME, DATE_TIME, allowEmpty);
    }
    
    /**
     * 是否是邮政编码
     * 
     * @param allowEmpty    是否允许为空
     * @return isPostalCode
     */
    public static String isPostalCode(boolean allowEmpty)
    {
        return isRegexp(IS_POSTAL_CODE, POSTAL_CODE, allowEmpty);
    }
    
    /**
     * 是否是合适的用户名
     * 
     * @param allowEmpty    是否允许为空
     * @return isUserName
     */
    public static String isUserCode(boolean allowEmpty)
    {
        return isRegexp(IS_USER_CODE, USER_CODE, allowEmpty);
    }
    
    /**
     * 是否是合适的用户密码
     * 
     * @param allowEmpty    是否允许为空
     * @return isUserPass
     */
    public static String isUserPass(boolean allowEmpty)
    {
        return isRegexp(IS_USER_PASS, USER_PASS, allowEmpty);
    }
}
