/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.sessioner;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.httpd.HttpSession;
import org.zhiqim.httpd.HttpSessionManager;
import org.zhiqim.httpd.HttpSessionUser;
import org.zhiqim.kernel.httpclient.HttpRMI;
import org.zhiqim.kernel.json.Jsons;
import org.zhiqim.kernel.paging.PageBuilder;
import org.zhiqim.kernel.paging.PageResult;
import org.zhiqim.kernel.util.Arrays;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Classes;
import org.zhiqim.kernel.util.Ints;
import org.zhiqim.kernel.util.Lists;

/**
 * Hash会话管理器，通过HashMap对会话进行管理
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class RmiSessionManager implements HttpSessionManager
{
    private static final String SESSIONER = "sessioner";
    private HttpContext context;
    
    //四个接口定义参数
    private String url;
    private String key;
    private String secret;
    
    //两个会话参数
    private String sessionIdName;
    private int sessionTimeout;
    
    /** 新建会话 */
    public HttpSession newSession(String sessionId)
    {
        return new RmiSession().setSessionManager(this).setSessionId(sessionId);
    }
    
    /** 获取上下文环境对象 */
    public HttpContext getContext()
    {
        return context;
    }
    
    /** 设置上下文环境对象 */
    public void setContext(HttpContext context)
    {
        this.context = context;
    }
    
    /**************************************************************************************/
    //开启和关闭
    /**************************************************************************************/
    
    /** 开启会话管理器 */
    public void open()
    {
    }
    
    /** 关闭会话管理器 */
    public void close()
    {
    }
    
    /**************************************************************************************/
    //设置参数部分
    /**************************************************************************************/
    
    /** 设置会话编号名称 */
    public void setSessionIdName(String sessionIdName)
    {
    }
    
    /** 设置会话超时时长 */
    public void setSessionTimeout(int sessionTimeout)
    {
    }
    
    /** 设置会话管理器参数，Hash模式无需参数 */
    public void setSessionParameter(String sessionParameter)
    {
        String[] params = Arrays.toStringArray(sessionParameter);
        this.url = params[0];
        this.key = params[1];
        this.secret = params[2];
    }
    
    public String getUrl()
    {
        return url;
    }
    
    /**************************************************************************************/
    //获取参数部分
    /**************************************************************************************/
    
    /** 获取会话编号名称 */
    public String getSessionIdName()
    {
        if (sessionIdName != null)
            return sessionIdName;
        
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("getSessionIdName");
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
        
        sessionIdName = conn.getResponseText();
        return sessionIdName;
    }
    
    /** 获取超时时长 */
    public int getSessionTimeout()
    {
        if (sessionTimeout != 0)
            return sessionTimeout;
        
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("getSessionTimeout");
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
        
        sessionTimeout = Ints.toInt(conn.getResponseText());
        return sessionTimeout;
    }
    
    /**************************************************************************************/
    //设置 & 获取 & 失效 会话
    /**************************************************************************************/
    
    /** 设置一个会话 */
    public void setSession(HttpSession session)
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("setSession");
        conn.addParam(Jsons.toString(session));
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
    }
    
    /** 通过会话编号获取会话对象 */
    public HttpSession getSession(String sessionId)
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("getSession");
        conn.addParam(sessionId);
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
        
        String responseText = conn.getResponseText();
        RmiSession session = new RmiSession().setSessionManager(this);
        session.parse(responseText);
        return session;
    }
    
    /** 判断会话是否存在 */
    public boolean hasSession(String sessionId)
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("hasSession");
        conn.addParam(sessionId);
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
        
        return _TRUE_.equals(conn.getResponseText());
    }
    
    /** 删除一个会话，指定会话编号 */
    public void invalidateSession(String sessionId)
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("invalidateSession");
        conn.addParam(sessionId);
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
    }
    
    /** 更新会话最后访问时间 */
    public void updateLastAccessed(String sessionId)
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("updateLastAccessed");
        conn.addParam(sessionId);
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
    }
    
    /** 更新会话信息 */
    public void updateInfo(String sessionId, String sessionIp, String userAgent)
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("updateInfo");
        conn.addParam(sessionId);
        conn.addParam(sessionIp);
        conn.addParam(userAgent);
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
    }
    
    /**************************************************************************************/
    //会话查询和统计
    /**************************************************************************************/
    
    /** 查会话数 */
    public int size()
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("size");
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }

        return Ints.toInt(conn.getResponseText());
    }
    
    /** 获取会话列表 */
    public List<HttpSession> getSessionList()
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("getSessionList");
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }

        List<RmiSession> list = Jsons.toList(conn.getResponseText(), RmiSession.class);
        ArrayList<HttpSession> nList = new ArrayList<>(list.size());
        for (HttpSession s : list)
        {
            nList.add(s);
        }
        
        return Collections.unmodifiableList(Lists.trim(nList));
    }
    
    /** 获取会话分页信息 */
    public PageResult<HttpSession> getSessionPage(int pageNo, int pageSize)
    {
        int size = size();
        if (size == 0)
            return PageBuilder.newResult(pageNo, pageSize);
        
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("getSessionList");
        conn.addParam(pageNo);
        conn.addParam(pageSize);
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
        
        List<RmiSession> list = Jsons.toList(conn.getResponseText(), RmiSession.class);
        ArrayList<HttpSession> nList = new ArrayList<>(list.size());
        for (HttpSession s : list)
        {
            nList.add(s);
        }
        
        return PageBuilder.newResult(size, pageNo, pageSize, nList);
    }
    
    /**************************************************************************************/
    //会话用户查询和统计
    /**************************************************************************************/
    
    /** 查用户数，有登录状态 */
    public int sizeUser()
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("sizeUser");
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }

        return Ints.toInt(conn.getResponseText());
    }
    
    /** 是否有会话名称对应的会话 */
    public boolean hasSessionName(String sessionName)
    {
        Asserts.notNull(sessionName, "sessionName");
        
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("hasSessionName");
        conn.addParam(sessionName);
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }
        
        return _TRUE_.equals(conn.getResponseText());
    }
    
    /** 获取会话登录用户列表，指定会话用户类名（如前后端共用session管理器的情况） */
    public List<HttpSessionUser> getSessionUserList()
    {
        HttpRMI conn = new HttpRMI(url);
        conn.setServiceId(SESSIONER);
        conn.setKeySecret(key, secret);
        conn.setMethodName("getSessionUserList");
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }

        List<HttpSessionUser> list = new ArrayList<>();
        List<String> strList = Jsons.toList(conn.getResponseText(), String.class);
        for (String str : strList)
        {
            String className = Jsons.getString(str, "className");
            Class<?> clazz = Classes.forName(className);
            if (clazz == null)
                continue;
            
            HttpSessionUser sessionUser = (HttpSessionUser)Jsons.toObject(str, clazz);
            list.add(sessionUser);
        }
        
        return Lists.trim(list);
    }
    
    /** 获取会话登录用户分页信息，指定会话用户类名（如前后端共用session管理器的情况） */
    public PageResult<HttpSessionUser> getSessionUserPage(int pageNo, int pageSize)
    {
        int size = sizeUser();
        if (size == 0)
            return PageBuilder.newResult(pageNo, pageSize);
        
        HttpRMI conn = new HttpRMI(url);
        conn.setKeySecret(key, secret);
        conn.setServiceId(SESSIONER);
        conn.setMethodName("getSessionUserList");
        conn.addParam(pageNo);
        conn.addParam(pageSize);
        conn.execute();
        
        if (!conn.isResponseSuccess())
        {
            throw Asserts.exception(conn.getResponseText());
        }

        List<HttpSessionUser> list = new ArrayList<>();
        List<String> strList = Jsons.toList(conn.getResponseText(), String.class);
        for (String str : strList)
        {
            String className = Jsons.getString(str, "className");
            Class<?> clazz = Classes.forName(className);
            if (clazz == null)
                continue;
            
            HttpSessionUser sessionUser = (HttpSessionUser)Jsons.toObject(str, clazz);
            list.add(sessionUser);
        }
        
        return PageBuilder.newResult(size, pageNo, pageSize, list);
    }
}
