/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.context.service;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.httpd.HttpExecutor;
import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.httpd.HttpResponse;
import org.zhiqim.httpd.constants.HttpMimeType;
import org.zhiqim.httpd.context.ZmlContextConstants;

import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.util.DateTimes;
import org.zhiqim.kernel.util.Files;
import org.zhiqim.kernel.util.Streams;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.kernel.util.codes.Base64;

/**
 * 上传文件访问服务，用于固定访问地址为/service/upl/*，中间*=${fileDir}/${fileId}，实际处理在forward中
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class UploadAccessService implements HttpExecutor, ZmlContextConstants
{
    private static final Log log = LogFactory.getLog(UploadAccessService.class);
    private String uploadRootDir;
    
    @Override
    public boolean isMatch(String pathInContext)
    {
        return Validates.isMatch(pathInContext, _PATH_SERVICE_UPLOAD_ACCESS_);
    }
    
    @Override
    public void handle(HttpRequest request, HttpResponse response) throws IOException
    {
        HttpContext context = request.getContext();
        String forward = (String)context.getAttribute(_SERV_UPLOAD_FORWARD_);
        if (Validates.isNotEmpty(forward))
        {//跳转优先，成功则跳转到对应的连接
            try{request.forwardTo(forward);}
            catch(Exception e){response.sendHeader(_500_INTERNAL_SERVER_ERROR_);}
            return;
        }

        /**************************************************************/
        //当前处理方式
        /**************************************************************/
        if (uploadRootDir == null)
        {
            String rootDir = (String)context.getAttribute(_SERV_UPLOAD_ROOT_DIR_);
            if (Validates.isEmptyBlank(rootDir) || !Files.mkDirectory(rootDir))
            {
                log.error("[UploadAccessService][配置处理器处理时必须配置上传根目录][%s]", _SERV_UPLOAD_ROOT_DIR_);
                response.sendError(_403_FORBIDDEN_);
                return;
            }
            
            uploadRootDir = Files.toLinuxPath(new File(rootDir).getCanonicalPath());
            uploadRootDir = Strings.addEndsWith(uploadRootDir, "/");
        }
        
        String filePath = Strings.removeStartsWith(request.getPathInContext(), _PATH_UPLOAD_PREFIX_);
        String fileDir = Files.getFileDirPath(filePath, "/");
        String fileName = Files.getFileNameLinux(filePath);
        
        if (Validates.isNotEmpty(fileDir))
        {
            fileDir = Base64.decodeFileUrlUTF8(fileDir);
        }
        
        if (Validates.isEmpty(fileDir))
            filePath = uploadRootDir + fileName;
        else
            filePath = uploadRootDir + fileDir + "/" + fileName;
        
        File file = new File(filePath);
        if (!file.exists() || !file.isFile())
        {//不存在或不是文件
            response.sendError(_404_NOT_FOUND_);
            return;
        }
        
        if (file.isHidden() || !file.canRead())
        {//隐藏、不可读和文件大小超限
            response.sendError(_403_FORBIDDEN_);
            return;
        }
        
        //设置文件类型
        String fileExt = Files.getFileExt(fileName);
        response.setContentType(HttpMimeType.getResourceMimeType(fileExt));
        
        long lastModified = file.lastModified();
        String lastModifiedTime = DateTimes.getDateTimeHttp(lastModified);
        String ifModifiedSince = request.getHeader(_IF_MODIFIED_SINCE_);
        if (ifModifiedSince != null && ifModifiedSince.indexOf(lastModifiedTime) != -1)
        {//时间相等
            response.sendHeader(_304_NOT_MODIFIED_);
            return;
        }
        
        response.setHeader(_ACCEPT_RANGES_, _BYTES_);
        response.setHeader(_LAST_MODIFIED_, lastModifiedTime);
        
        if (request.isMethodResponseContent())
        {//GET,POST
            try(FileInputStream fis = new FileInputStream(file))
            {
                Streams.putBytes(fis, response.getOutputStream());
            }
        }
        
        response.commit();
    }
}
