/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.context.config;

import java.util.Collection;
import java.util.List;

import org.zhiqim.httpd.context.ZmlConfig;
import org.zhiqim.httpd.context.ZmlContextConstants;
import org.zhiqim.kernel.annotation.AnNullable;
import org.zhiqim.kernel.extend.LinkedMapSV;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Lists;
import org.zhiqim.kernel.util.Objects;
import org.zhiqim.kernel.util.Replaces;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.zml.Zmls;

/**
 * ZAction表，目标是把精确匹配和模糊匹配区分开来
 * 1. exactMap      精确匹配表
 * 2. fuzzyMap      模糊匹配表
 *
 * @version v1.0.0 @author zhichenggang 2016-12-31 新建与整理
 */
public class ZActionMap implements ZmlContextConstants
{
    private final ZmlConfig zmlConfig;
    private final ZConfig config;
    
    private LinkedMapSV<ZAction> exactMap = new LinkedMapSV<>();
    private LinkedMapSV<ZAction> fuzzyMap = new LinkedMapSV<>();
    
    public ZActionMap(ZmlConfig zmlConfig, ZConfig config)
    {
        this.zmlConfig = zmlConfig;
        this.config = config;
    }
    
    /**********************************************************************/
    //list & add & toString
    /**********************************************************************/
    
    /** 获取精确表 */
    public Collection<ZAction> getExactActionList()
    {
        return exactMap.values();
    }
    
    /** 获取模糊表 */
    public Collection<ZAction> getFuzzyActionList()
    {
        return fuzzyMap.values();
    }
    
    /**
     * 添加一个ZAction
     * 
     * @param path      路径，精确或模糊pattern
     * @param action    ZAction类
     */
    public void addAction(ZAction action)
    {
        action.setConfig(config);
        String path = action.getPath();
        if (path.indexOf("*") == -1)
            exactMap.put(path, action);
        else
            fuzzyMap.put(path, action);
    }
    
    /** 删除一个ZAction */
    public void removeAction(String path)
    {
        exactMap.remove(path);
        fuzzyMap.remove(path);
    }
    
    /** 检查变量名 */
    public void chkVariable()
    {
        LinkedMapSV<ZAction> newExactMap = new LinkedMapSV<>();
        for (ZAction zAction : exactMap.values())
        {
            ZAction newAction = chkVariable(zAction);
            newExactMap.put(newAction.getPath(), newAction);
        }
        
        exactMap.clear();
        exactMap = newExactMap;
        
        LinkedMapSV<ZAction> newFuzzyMap = new LinkedMapSV<>();
        for (ZAction zAction : fuzzyMap.values())
        {
            ZAction newAction = chkVariable(zAction);
            newFuzzyMap.put(newAction.getPath(), newAction);
        }
        
        fuzzyMap.clear();
        fuzzyMap = newFuzzyMap;
    }
    
    /** 检查变量名 */
    private ZAction chkVariable(ZAction zAction)
    {
        String path = zAction.getPath();
        if (path.contains("${") || path.contains("<#"))
        {
            try
            {
                zAction.setPath(Zmls.parseContext(path, zmlConfig));
            }
            catch (Exception e)
            {
                throw Asserts.exception("action配置[%s]，path变量解析失败", path);
            }
        }
        
        String redirect = zAction.getRedirect();
        if (Validates.isNotEmpty(redirect) && path.contains("${") || path.contains("<#"))
        {
            try
            {
                zAction.setRedirect(Zmls.parseContext(redirect, zmlConfig));
            }
            catch (Exception e)
            {
                throw Asserts.exception("action配置[%s]，redirect变量解析失败", path);
            }
        }
        
        String forward = zAction.getForward();
        if (Validates.isNotEmpty(forward) && path.contains("${") || path.contains("<#"))
        {
            try
            {
                zAction.setForward(Zmls.parseContext(forward, zmlConfig));
            }
            catch (Exception e)
            {
                throw Asserts.exception("action配置[%s]，forward变量解析失败", path);
            }
        }
        
        return zAction;
    }
    
    /** toString */
    public String toString()
    {
        StringBuilder strb = new StringBuilder();
        for (ZAction action : fuzzyMap.values())
        {
            strb.append(_FOUR_).append(action.toString()).append(_BR_);
        }
        for (ZAction action : exactMap.values())
        {
            strb.append(_FOUR_).append(action.toString()).append(_BR_);
        }
        
        return strb.toString();
    }
    
    /**********************************************************************/
    //通过类名方式
    /**********************************************************************/
    
    /**
     * 获取精确匹配的ZAction的path
     * 
     * @param clazz         传入的类名
     * @return              path
     */
    @AnNullable
    public ZAction getExactActionByClass(String clazz)
    {
        if (clazz == null)
            return null;
        
        for (ZAction action : exactMap.values())
        {
            if (clazz.equals(action.getClazz()))
                return action;
        }
        
        return null;
    }
    
    /**
     * 获取模糊匹配的ZAction的path
     * 
     * @param clazz         传入的类名
     * @return              path
     */
    @AnNullable
    public ZAction getFuzzyActionByClass(String clazz)
    {
        if (clazz == null)
            return null;
        
        for (ZAction action : fuzzyMap.values())
        {
            if (clazz.equals(action.getClazz()))
                return action;
        }
        
        return null;
    }
    
    /**********************************************************************/
    //通过 path 获取 ZAction
    /**********************************************************************/
    
    /**
     * 获取精确匹配的ZAction
     * 
     * @param path  传入的上下文环境下的路径
     * @return      ZAction
     */
    @AnNullable
    public ZAction getExactAction(String path)
    {
        return exactMap.get(path);
    }
    
    /**
     * 获取模糊匹配的ZAction
     * 
     * @param path  传入的上下文环境下的路径
     * @return      ZAction
     */
    @AnNullable
    public ZAction getFuzzyAction(String path)
    {
        for (String p : fuzzyMap.keySet())
        {
            List<String> matchList = Lists.getMatchList(path, p);
            if (matchList == null)
                continue;
            
            return getActionMatchParam(p, fuzzyMap.get(p), path, matchList);
        }
        
        return null;
    }
    
    /** 获取含通匹符的Action */
    private ZAction getActionMatchParam(String path, ZAction action, String pathInContext, List<String> paramList)
    {
        //生成新的Action配置
        ZAction newAction = Objects.copy(action, new ZAction());
        newAction.setPath(pathInContext);//重置path为新的pathInContext
        
        for (int i=0;i<paramList.size();i++)
        {//对参数表替换{}
            String regex = "{"+(i+1)+"}";
            String replacement = paramList.get(i);
            
            newAction.setInterceptor(Replaces.replaceAllEscape(newAction.getInterceptor(), regex, replacement));
            newAction.setForward(Replaces.replaceAllEscape(newAction.getForward(), regex, replacement));
            newAction.setRedirect(Replaces.replaceAllEscape(newAction.getRedirect(), regex, replacement));
            newAction.setView(Replaces.replaceAllEscape(newAction.getView(), regex, replacement));
            newAction.setInclude(Replaces.replaceAllEscape(newAction.getInclude(), regex, replacement));
            newAction.setClazz(Replaces.replaceAllEscape(newAction.getClazz(), regex, replacement));
            newAction.setMethod(Replaces.replaceAllEscape(newAction.getMethod(), regex, replacement));
            newAction.setSuccess(Replaces.replaceAllEscape(newAction.getSuccess(), regex, replacement));
        }
        
        return newAction.setMatchList(paramList);
    }
}
