/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd;

import java.io.IOException;

import org.zhiqim.kernel.constants.CodeConstants;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Bytes;
import org.zhiqim.kernel.util.Strings;

/**
 * HttpWebsocket数据格式：
 * 第一字节，第1位：FIN，标识是否为此消息的最后一个数据包
 * 第一字节，第5,6,7,8位，opcode，操作码（0－15）：
 * 0x00：标识一个中间数据包
 * 0x01：标识一个text类型数据包
 * 0x02：标识一个binary类型数据包
 * 0x08：标识一个断开连接类型数据包
 * 0x09：标识一个ping类型数据包
 * 0x0A：表示一个pong类型数据包
 * 0x03-07：保留
 * 0x0B-0F：保留
 * 
 * 一共分5步读取数据
 * 1、读first字节
 * 2、读second字节
 * 3、根据length读2字节或8字节
 * 4、读masks
 * 5、读length
 *
 * @version v1.0.0 @author zouzhigang 2017-6-1 新建与整理
 */
public class HttpWebsocketMessageAsync extends HttpWebsocketMessage implements CodeConstants
{
    private int step;
    private int first;
    private int second;
    private byte[] buffer;
    private int bufferOffset;
    private int masksOffset;
    private int contentOffset;
    
    public boolean parse(HttpInputStream in) throws IOException
    {
        if (step < 1)
        {//第一步
            first = in.read();
            if (first == -1)
            {//异步未读完，等待下次再读
                return false;
            }
            
            step = 1;
        }
        
        if (step < 2)
        {//第二步
            second = in.read();
            if (second == -1)
            {//异步未读完，等待下次再读
                return false;
            }
            
            step = 2;
        }
        
        //1.FIN
        fin = (first & 0x80) != 0;
        
        //2.opcode
        opcode = first & 0x0F;
        
        //3.MASK
        mask = (second & 0x80) != 0;
        
        //4.length
        contentLen = length = second & 0x7F;
        if (length == 126)
        {
            if (step < 3)
            {
                if (buffer == null)
                {//初始化
                    buffer = new byte[2];
                    bufferOffset = 0;
                }
                
                byte[] buf = in.read(2 - bufferOffset);
                if (buf != null && buf.length > 0)
                {
                    bufferOffset = Bytes.putBytes(buffer, bufferOffset, buf);
                }
                
                if (bufferOffset < 2)
                {//异步未读完，等待下次再读
                    return false;
                }
                
                contentLen = Bytes.BU.getShortUnsigned(buffer, 0);
                step = 3;
            }
        }
        else if (length == 127)
        {
            if (step < 3)
            {
                if (buffer == null)
                {//初始化
                    buffer = new byte[8];
                    bufferOffset = 0;
                }
                
                byte[] buf = in.read(8 - bufferOffset);
                if (buf != null && buf.length > 0)
                {
                    bufferOffset = Bytes.putBytes(buffer, bufferOffset, buf);
                }
                
                if (bufferOffset < 8)
                {//异步未读完，等待下次再读
                    return false;
                }
                
                long len = Bytes.BU.getLong(buf, 0);
                if (len > Integer.MAX_VALUE)
                    throw Asserts.exception("超出支持的能力");
                
                contentLen = (int)len;
                step = 3;
            }
        }
        
        //5.masks
        if (step < 4)
        {
            if (mask)
            {
                if (masks == null)
                {//初始化
                    masks = new byte[4];
                    masksOffset = 0;
                }
                
                byte[] buf = in.read(4 - masksOffset);
                if (buf != null && buf.length > 0)
                {
                    masksOffset = Bytes.putBytes(masks, masksOffset, buf);
                }
                
                if (masksOffset < 4)
                {//异步未读完，等待下次再读
                    return false;
                }
            }
            
            step = 4;
        }
        
        //6.内容
        if (step < 5)
        {
            if (contentLen > 0)
            {//有内容
                if (content == null)
                {//初始化
                    content = new byte[contentLen];
                    contentOffset = 0;
                }
                
                byte[] buf = in.read(contentLen - contentOffset);
                if (buf != null && buf.length > 0)
                {
                    contentOffset = Bytes.putBytes(content, contentOffset, buf);
                }
                
                if (contentOffset < contentLen)
                {//异步未读完，等待下次再读
                    return false;
                }
                
                if (mask)
                {
                    for (int i=0;i<content.length;i++)
                    {
                        content[i] = (byte)(content[i] ^ masks[i % 4]);
                    }
                }
                
                if (isText())
                {//文本的解析成字符串
                    text = Strings.newStringUTF8(content);
                }
                else if (isClose())
                {//结束符的有状态码，CloseEvent.code，如1001
                    text = String.valueOf(Bytes.BU.getShortUnsigned(content, 0));
                }
            }
            
            step = 5;
        }

        return true;
    }
    
    /***********************************************************************/
    // 获取&设置
    /***********************************************************************/
    
    public boolean isParsed()
    {
        return step == 5;
    }
}
