/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd;

import java.io.IOException;

/**
 * Websocket连接抽象类
 *
 * @version v1.0.0 @author zouzhigang 2018-6-30 新建与整理
 */
public abstract class HttpWebsocketConnectionAbs implements HttpWebsocketConnection
{
    protected HttpContext context;
    protected HttpWebsocketManager manager;
    protected String protocol;
    protected String sessionId;
    protected HttpWebsocket websocket;
    protected volatile boolean isRunning;
    
    public HttpWebsocketConnectionAbs(HttpContext context, String protocol, String sessionId)
    {
        this.context = context;
        this.manager = context.getWebsocketManager();
        this.protocol = protocol;
        this.sessionId = sessionId;
        
        this.websocket = manager.getWebsocket(protocol);
        if (this.websocket != null)
        {//开启回调
            this.websocket.onOpen(this);
        }
    }
    
    /** 子类根据不同的结构提供不同的连接类 */
    public abstract HttpConnection getConnection();
    
    /***********************************************************************************/
    //打开&关闭
    /***********************************************************************************/
    
    @Override /** 开启，子类实现 */
    public abstract void open();
    
    @Override /** 正常关闭 */
    public void close()
    {
        try
        {
            if (websocket != null)
                websocket.onClose(this);
        }
        finally
        {
            websocket = null;
            getWebsocketManager().remove(this);
        }
    }

    @Override /** 异常关闭 */
    public void exception(Throwable e)
    {
        try
        {
            if (websocket != null)
                websocket.onException(this, e);
        }
        finally
        {
            websocket = null;
            getWebsocketManager().remove(this);
        }
    }
    
    /***********************************************************************************/
    //发送数据
    /***********************************************************************************/
    
    public boolean send(String text)
    {
        HttpConnection conn = getConnection();
        if (conn == null || conn.isClosed())
            return false;
        
        HttpWebsocketMessage message = new HttpWebsocketMessage().build(text);
        return send(message.toBytes());
    }
    
    public boolean sendPing()
    {
        HttpConnection conn = getConnection();
        if (conn == null || conn.isClosed())
            return false;
        
        HttpWebsocketMessage message = new HttpWebsocketMessage().buildPing();
        return send(message.toBytes());
    }
    
    public boolean send(byte[] data)
    {
        HttpConnection conn = getConnection();
        if (conn == null || conn.isClosed())
            return false;
        
        try
        {
            conn.write(data, 0, data.length);
            return true;
        }
        catch (IOException e)
        {
            exception(e);
            return false;
        }
    }
    
    /***********************************************************************************/
    //获取属性
    /***********************************************************************************/

    public String getProtocol()
    {
        return protocol;
    }
    
    public HttpSessionManager getSessionManager()
    {
        return context.getSessionManager();
    }
    
    public String getSessionId()
    {
        return sessionId;
    }
    
    public HttpSession getSession()
    {
        return getSessionManager().getSession(sessionId);
    }
    
    public HttpWebsocket getWebsocket()
    {
        return websocket;
    }
    
    public HttpWebsocketManager getWebsocketManager()
    {
        return context.getWebsocketManager();
    }
}
