/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd;

import java.io.IOException;
import java.io.InputStream;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.zhiqim.kernel.util.consts.Int;
import org.zhiqim.httpd.validate.Validate;

/**
 * HTTP请求接口定义
 *
 * @version v1.0.0 @author zouzhigang 2016-7-15 新建与整理
 */
public interface HttpRequest extends HttpHeader
{
    /** 获取请求编号 */
    public String getId();
    
    /** 获取产生时间戳 */
    public long getTimeMillis();
    
    /***********************************************************************/
    //请求状态处理和判断
    /***********************************************************************/
    
    /** 设置拦截中 */
    public void setStepInterceptor();
    
    /** 设置Action处理中 */
    public void setStepAction();

    /** 获取当前处理环节 */
    public int getStep();
    
    /** 获取当前处理环节描述 */
    public String getStepDesc();

    /** 是否可读，已读头部之后可读 */
    public boolean isRead();
    
    /** 是否可编辑，提交之前可编辑 */
    public boolean isEditable();
    
    /** 是否已提交，提交之后不允许再操作request和response */
    public boolean isCommitted();
    
    /***********************************************************************/
    //获取和判断请求行信息，包括协议、方法、版本、URI等
    /***********************************************************************/
    
    /** 获取协议格式 */
    public String getScheme();
    
    /** 获取端口信息 */
    public int getPort();
    
    /** 获取URI信息 */
    public String getRequestURI();
    
    /** 获取URL信息 */
    public String getRequestURL();
    
    /***********************************************************************/
    //获取和判断请求头部信息
    /***********************************************************************/
    
    /** 获取消息头名称迭代器 */
    public Iterator<String> getHeaderNames();
    
    /** 获取请求头字符串 */
    public String getHeaderString();
    
    /** 获取请求头属性 */
    public String getHeader(String key, String defaultValue);
    
    /** 获取请求头属性整型值 */
    public int getHeaderInt(String key);
    
    /** 获取请求头属性整型值 */
    public int getHeaderInt(String key, int defaultValue);
    
    /** 获取请求头属性长整型值 */
    public long getHeaderLong(String key);
    
    /** 获取请求头属性长整型值 */
    public long getHeaderLong(String key, long defaultValue);
    
    /** 获取组装内容类型 */
    public String getContentTypeMimeEncoding();
    
    /** 获取来源地址 */
    public String getReferer();
    
    /** 获取知启蒙定义的客户端类型（mobile|www），该值也会设置到request的属性中 */
    public String getZhiqimClientType();
    
    /** 是否是异isXMLHttpRequest */
    public boolean isXMLHttpRequest();
    
    /** 是否手机端 */
    public boolean isMobile();
    
    /** 是否微信 */
    public boolean isMicroMessenger();
    
    /** 是否IE浏览器 */
    public boolean isIE();
    
    /** 是否是火狐浏览器 */
    public boolean isFirefox();
    
    /** 是否是Webkit浏览器 */
    public boolean isWebkit();
    
    /** 是否是chrome浏览器 */
    public boolean isChrome();

    /** 是否是Safari浏览器 */
    public boolean isSafari();
    
    /***********************************************************************/
    //获取监听、服务器、上下文环境、响应、会话相关
    /***********************************************************************/
    
    /** 获取请求对应的响应对象 */
    public HttpResponse getResponse();
    
    /** 获取请求所在的服务 */
    public HttpServer getServer();
    
    /** 获取上下文环境路径 */
    public String getContextPath();
    
    /** 获取上下文环境名称 */
    public String getContextName();
    
    /** 获取上下文环境版本 */
    public String getContextVersion();
    
    /** 设置上下文环境属性 */
    public void setContextAttribute(String key, Object value);
    
    /** 设置上下文环境属性值（通过类结构） */
    public <T> void setContextAttribute(Class<T> key, T value);
    
    /** 获取上下文环境属性值 */
    public Object getContextAttribute(String key);
    
    /** 获取上下文环境属性值（通过类结构） */
    public <T> T getContextAttribute(Class<T> key);
    
    /** 获取上下文环境属性值 */
    public Object getContextAttribute(String key, Object defaultValue);
    
    /** 获取上下文环境属性值字符串 */
    public String getContextAttributeString(String key);
    
    /** 获取上下文环境属性值字符串 */
    public String getContextAttributeString(String key, String defaultValue);
    
    /** 获取上下文环境属性值整型 */
    public int getContextAttributeInt(String key);
    
    /** 获取上下文环境属性值整型 */
    public int getContextAttributeInt(String key, int defaultValue);
    
    /** 获取上下文环境属性值长整型 */
    public long getContextAttributeLong(String key);
    
    /** 获取上下文环境属性值长整型 */
    public long getContextAttributeLong(String key, long defaultValue);
    
    /** 获取上下文环境属性值布尔型 */
    public boolean getContextAttributeBoolean(String key, boolean defaultValue);
    
    /** 判断上下文环境是否有属性值 */
    public boolean hasContextAttribute(String key);
    
    /** 获取当前WEB目录根目录，注意没有/结尾 */
    public String getResourcePath();
    
    /** 获取上下文环境下绝对路径转为根环境下的绝对路径，如contextPath=/doc,path=/index.htm，得到/doc/index.htm */
    public String getRootPath(String path);
    
    /** 获取当前WEB目录路径转换成绝对路径 */
    public String getRealPath(String path);
    
    /** 获取SSL证书 */
    public X509Certificate[] getCertificates();
    
    /***********************************************************************/
    //设置和获取Session中属性
    /***********************************************************************/
    
    /** 获取请求用户名 */
    public String getRequestName();
    
    /** 设置请求用户名 */
    public void setRequestName(String name);
    
    /** 获取会话编号 */
    public String getSessionId();
    
    /** 获取请求对应的会话环境 */
    public HttpSession getSession();
    
    /** 删除会话 */
    public void invalidateSession();
    
    /** 判断是否绑定了登录用户 */
    public boolean hasSessionUser();
    
    /** 绑定会话登录用户 */
    public void bindSessionUser(HttpSessionUser sessionUser);
    
    /** 解除绑定会话登录用户 */
    public void unbindSessionUser();
    
    /** 判断是否绑定了登录用户，且指定的会话用户类（HttpSessionUser子类） */
    public <T extends HttpSessionUser> boolean hasSessionUser(Class<T> clazz);
    
    /** 获取会话用户信息，且指定的会话用户类（HttpSessionUser子类） */
    public <T extends HttpSessionUser> T getSessionUser(Class<T> clazz);
    
    /** 获取会话默认的用户信息 */
    public HttpSessionUser getSessionUser();
    
    /** 获取会话默认的登录名 */
    public String getSessionName();
    
    /** 设置会话属性 */
    public void setSessionAttribute(String name, Object value);
    
    /** 获取会话属性值 */
    public Object getSessionAttribute(String name);
    
    /** 获取会话属性值 */
    public Object getSessionAttribute(String name, Object defaultValue);
    
    /** 获取会话属性值字符串 */
    public String getSessionAttributeString(String name);
    
    /** 获取会话属性值字符串 */
    public String getSessionAttributeString(String name, String defaultValue);
    
    /** 获取会话属性值整型 */
    public int getSessionAttributeInt(String name);
    
    /** 获取会话属性值整型 */
    public int getSessionAttributeInt(String name, int defaultValue);
    
    /** 获取会话属性值长整型 */
    public long getSessionAttributeLong(String name);
    
    /** 获取会话属性值长整型 */
    public long getSessionAttributeLong(String name, long defaultValue);
    
    /** 判断会话是否有属性值 */
    public boolean hasSessionAttribute(String name);
    
    /***********************************************************************/
    //获取Cookie数据或数组
    /***********************************************************************/
    
    /** 获取指定名称的Cookie */
    public String getCookie(String name);
    
    /** 获取Cookie，如果没有值或值为空字符串返回缺省值 */
    public String getCookie(String name, String defaultValue);
    
    /** 获取请求中Cookie数组 */
    public HttpCookie[] getCookies();
    
    /** 删除请求中的Cookie值 */
    public void removeCookie(String name);
    
    /** 增加Cookie到响应中 */
    public void addCookieToResponse(String name, String value);
    
    /** 增加Cookie到响应中 */
    public void addCookieToResponse(String name, String value, int seconds);
    
    /** 增加Cookie到响应中 */
    public void addCookieToResponse(HttpCookie cookie);
    
    /** 删除Cookie到响应中 */
    public void removeCookieToResponse(String name);
    
    /***********************************************************************/
    //获取请求中URL匹配的参数
    /***********************************************************************/
    
    /** 获取通配符参数列表 */
    public List<String> getParameterMatch();
    
    /**
     * 获取通配符参数值
     * 
     * @param index 参数匹配位置
     * @return      有过滤的参数值
     */
    public String getParameterMatch(int index);
    
    /**
     * 获取通配符参数值
     * 
     * @param index 参数匹配位置
     * @return      有过滤的参数值
     */
    public long getParameterMatchLong(int index);
    
    /**
     * 获取通配符参数值
     * 
     * @param index 参数匹配位置
     * @return      有过滤的参数值
     */
    public int getParameterMatchInt(int index);
    
    /***********************************************************************/
    //获取请求中URL和Content中的数据，包括URL和Content中的数据
    /***********************************************************************/
    
    /**
     * 获取参数哈唏表（未安全过滤），先填入url中参数表，再用content参数表覆盖，得到全部参数表
     * 
     * @return  得到全部参数表
     */
    public HashMap<String, String> getParameterMap();
    
    /** 判断是否有参数 */
    public boolean hasParameter(String name);
    
    /**
     * 获取参数值并安全过滤，优先检查content中是否存在，如果不存在才检查url中
     * 
     * @param name  参数名
     * @return      有过滤的参数值
     */
    public String getParameter(String name);
    
    /**
     * 获取动态参数对象信息
     * 
     * @param clazz     类结构
     * @return          返回赋值后的对象
     * @throws          异常需要捕捉
     */
    public <T> T getParameter(Class<T> clazz) throws IllegalAccessException;
    
    /**
     * 获取参数值（未过滤），优先检查content中是否存在，如果不存在才检查url中
     * 
     * @param name  参数名
     * @return      有过滤的参数值
     */
    public String getParameterNoFilter(String name);
    
    /**
     * 获取参数值并安全过滤，优先检查content中是否存在，如果不存在才检查url中，都不存在或值为空白返回缺省值
     * 
     * @param name          参数名
     * @param defaultValue  不存在或为空时的缺省值
     * @return              有过滤的参数值
     */
    public String getParameter(String name, String defaultValue);
    
    /**
     * 获取参数值（未过滤），优先检查content中是否存在，如果不存在才检查url中，都不存在或值为空白返回缺省值
     * 
     * @param name          参数名
     * @param defaultValue  不存在或为空时的缺省值
     * @return              有过滤的参数值
     */
    public String getParameterNoFilter(String name, String defaultValue);
    
    /**
     * 获取参数整型，非整型返回-1的值
     * 
     * @param name          参数名
     * @return              整数值
     */
    public int getParameterInt(String name);
    
    /**
     * 获取参数整型，非整型返回缺省值
     * 
     * @param name          参数名
     * @param defaultValue  缺省值
     * @return              整数值
     */
    public int getParameterInt(String name, int defaultValue);
    
    /**
     * 获取参数长整型，非整型返回-1的值
     * 
     * @param name          参数名
     * @return              长整型
     */
    public long getParameterLong(String name);
    
    /**
     * 获取参数长整型，非整型返回-1的值
     * 
     * @param name          参数名
     * @param defaultValue  缺省值
     * @return              长整型
     */
    public long getParameterLong(String name, long defaultValue);
    
    /**
     * 获取参数布尔型，非布尔型返回false的值
     * 
     * @param name          参数名
     * @return              布尔型
     */
    public boolean getParameterBoolean(String name);
    
    /**
     * 获取参数布尔型，非布尔型返回false的值
     * 
     * @param name          参数名
     * @param defaultValue  缺省值
     * @return              布尔型
     */
    public boolean getParameterBoolean(String name, boolean defaultValue);
    
    /**
     * 获取两位小数的金额值，格式不对返回-1
     * 
     * @param name          参数名
     * @return              金额值，如果从请求中获取的格式不对，则返回-1
     */
    public int getParameterAmount2R(String name);
    
    /**
     * 获取两位小数的金额值
     * 
     * @param name          参数名
     * @param defaultValue  缺省值
     * @return              金额值，如果从请求中获取的格式不对，则返回缺省值
     */
    public int getParameterAmount2R(String name, int defaultValue);
    
    /**
     * 获取两位小数的金额值长整型，格式不对返回-1
     * 
     * @param name          参数名
     * @param defaultValue  缺省值
     * @return              金额值，如果从请求中获取的格式不对，则返回-1
     */
    public long getParameterAmount2RLong(String name);
    
    /**
     * 获取两位小数的金额值长整型
     * 
     * @param name          参数名
     * @param defaultValue  缺省值
     * @return              金额值，如果从请求中获取的格式不对，则返回缺省值
     */
    public long getParameterAmount2R(String name, long defaultValue);
    
    /**
     * 获取参数值并安全过滤，并转化为数组（逗号格式），优先检查content中是否存在，如果不存在才检查url中
     * 
     * @param name  参数名
     * @return      有过滤后的逗号格式数组值
     */
    public String[] getParameterValues(String name);
    
    /**
     * 获取参数值（未过滤），并转化为数组（逗号格式），优先检查content中是否存在，如果不存在才检查url中
     * 
     * @param name  参数名
     * @return      有过滤后的逗号格式数组值
     */
    public String[] getParameterValuesNoFilter(String name);
    
    /***********************************************************************/
    //获取请求中URL中的数据
    /***********************************************************************/
    
    /** 获取URL参数哈唏表（值未过滤） */
    public HashMap<String,String> getParameterMapOnURL();
    
    /** 判断是否有URL参数 */
    public boolean hasParameterOnURL(String name);
    
    /** 获取URL参数值，并安全过滤 */
    public String getParameterOnURL(String name);
    
    /** 获取URL参数值，并安全过滤 */
    public String getParameterOnURL(String name, String defaultValue);
    
    /** 获取URL参数值(未过滤) */
    public String getParameterNoFilterOnURL(String name);
    
    /** 获取URL参数值(未过滤) */
    public String getParameterNoFilterOnURL(String name, String defaultValue);
    
    /** 获取URL参数整型，非整型返回-1的值 */
    public int getParameterIntOnURL(String name);
    
    /** 获取URL参数整型，非整型返回缺省值 */
    public int getParameterIntOnURL(String name, int defaultValue);
    
    /** 获取URL参数长整型，非整型返回-1的值 */
    public long getParameterLongOnURL(String name);
    
    /** 获取URL参数长整型，非整型返回-1的值 */
    public long getParameterLongOnURL(String name, long defaultValue);
    
    /** 获取两位小数的金额值 */
    public int getParameterAmount2ROnURL(String name, int defaultValue);
    
    /** 获取两位小数的金额值长整型 */
    public long getParameterAmount2ROnURL(String name, long defaultValue);

    /** 获取URL参数值数组，并安全过滤 */
    public String[] getParameterValuesOnURL(String name);
    
    /***********************************************************************/
    //获取请求中Content中的数据
    /***********************************************************************/

    /** 获取内容输入流 */
    public InputStream getInputStream();
    
    /** 获取内容输入流字符串 */
    public String getInputStreamString();
    
    /** 获取内容参数哈唏表（值未过滤） */
    public HashMap<String,String> getParameterMapOnCNT();
    
    /** 判断是否有内容参数 */
    public boolean hasParameterOnCNT(String name);
    
    /** 获取内容参数值，并安全过滤 */
    public String getParameterOnCNT(String name);
    
    /** 获取内容参数值，并安全过滤 */
    public String getParameterOnCNT(String name, String defaultValue);
    
    /** 获取内容参数值(未过滤) */
    public String getParameterNoFileterOnCNT(String name);
    
    /** 获取内容参数值(未过滤) */
    public String getParameterNoFileterOnCNT(String name, String defaultValue);
    
    /** 获取内容参数整型，非整型返回-1的值 */
    public int getParameterIntOnCNT(String name);
    
    /** 获取内容参数整型，非整型返回缺省值 */
    public int getParameterIntOnCNT(String name, int defaultValue);
    
    /** 获取内容参数长整型，非整型返回-1的值 */
    public long getParameterLongOnCNT(String name);
    
    /** 获取内容参数长整型，非整型返回-1的值 */
    public long getParameterLongOnCNT(String name, long defaultValue);
    
    /** 获取内容参数两位小数的金额值 */
    public int getParameterAmount2ROnCNT(String name, int defaultValue);
    
    /** 获取内容参数两位小数的金额值长整型 */
    public long getParameterAmount2ROnCNT(String name, long defaultValue);
    
    /** 获取内容参数值数组 */
    public String[] getParameterValuesOnCNT(String name);
    
    /** 设置参数值到URL表中 */
    public void setParameterOnURL(String name, Object value);
    
    /** 设置参数值到CNT表中 */
    public void setParameterOnCNT(String name, Object value);
    
    /***********************************************************************/
    //设置和获取请求中属性
    /***********************************************************************/
    
    /** 获取属性列表 */
    public Map<String, Object> getAttributeMap();
    
    /** 获取属性值 */
    public Object getAttribute(String name);
    
    /** 获取属性值，如果无该属性则返回缺省值 */
    public Object getAttribute(String name, Object defaultValue);
    
    /** 获取属性值字符串 */
    public String getAttributeString(String name);
    
    /** 获取属性值字符串，如果无该属性则返回缺省值 */
    public String getAttributeString(String name, String defaultValue);
    
    /** 获取属性值整型 */
    public int getAttributeInt(String name);
    
    /** 获取属性值整型，如果无该属性则返回缺省值 */
    public int getAttributeInt(String name, int defaultValue);
    
    /** 获取属性值整型 */
    public long getAttributeLong(String name);
    
    /** 获取属性值整型，如果无该属性则返回缺省值 */
    public long getAttributeLong(String name, int defaultValue);
    
    /** 判断属性是否存在 */
    public boolean hasAttribute(String name);
    
    /** 设置属性 */
    public void setAttribute(String name, Object value);
    
    /***********************************************************************/
    //获取请求/会话/上下文环境中嵌套属性，有则返回
    /***********************************************************************/
    
    /** 获取属性值 */
    public Object getNestAttribute(String name);
    
    /** 获取属性值，如果无该属性则返回缺省值 */
    public Object getNestAttribute(String name, Object defaultValue);
    
    /** 获取属性值字符串 */
    public String getNestAttributeString(String name);
    
    /** 获取属性值字符串，如果无该属性则返回缺省值 */
    public String getNestAttributeString(String name, String defaultValue);
    
    /** 获取属性值整型 */
    public int getNestAttributeInt(String name);
    
    /** 获取属性值整型，如果无该属性则返回缺省值 */
    public int getNestAttributeInt(String name, int defaultValue);
    
    /** 获取属性值整型 */
    public long getNestAttributeLong(String name);
    
    /** 获取属性值整型，如果无该属性则返回缺省值 */
    public long getNestAttributeLong(String name, int defaultValue);
    
    /** 判断属性是否存在 */
    public boolean hasNestAttribute(String name);
    
    /***********************************************************************/
    //内部重定向请求和上下文环境下绝对路径
    /***********************************************************************/
    
    /**
     * 内部重定向请求，支持从一个Action内部转到新的Action处理
     * 
     * @param pathInContext     上下文内部路径，即对应的新Action的配置路径
     * @throws IOException      IO异常
     * @throws HttpException    HTTP异常
     */
    public void forwardTo(String pathInContext) throws HttpException, IOException;
    
    /** 是否已标志为已内部重定向 */
    public boolean hasForward();
    
    /***********************************************************************/
    //以下为响应相关配置
    /***********************************************************************/
    
    
    /***********************************************************************/
    //从请求中设置属性到响应中
    /***********************************************************************/
    
    /**
     * 设置响应配置信息
     * 
     * @param view          转向到视图页
     * @param include       转向到视图页的嵌套地址
     * @param forward       转向到新的路径
     * @param redirect      重定向到新的路径
     * @param method        执行类的方法
     * @param alert         执行后的警告信息
     */
    public void setResponseConfig(String view, String include, String forward, String redirect, String method, String alert);
    
    /** 设置响应私用缓存 */
    public void setResponsePrivateCache();
    
    /** 设置响应头不缓存 */
    public void setResponseNoCache();
    
    /** 设置响应为UTF8编码*/
    public void setResponseEncodingUTF8();
    
    /***********************************************************************/
    //设置响应的配置信息
    /***********************************************************************/
    
    /** 修改为是否转向 */
    public void setReturn(boolean isReturn);
    
    /** 设置成功失败信息 */
    public void setAlertMsg(String errorMsg);
    
    /** 修改视图页和嵌套地址，和转向页互斥 */
    public void setView(String view, String include);
    
    /** 修改为内部转向地址 */
    public void setForward(String forwardTo);
    
    /** 修改重定向，状态码为602 */
    public void setRedirect(String redirect);
    
    /** 修改重定向，和视图页互斥，但不改变responseStatus值 */
    public void setRedirectInner(String redirect);
    
    /** 修改父页重定向，状态码为601 */
    public void setRedirectParent(String redirect);
    
    /** 修改顶页重定向 ，状态码为603*/
    public void setRedirectTop(String redirect);
    
    /** 修改重定向前提示错误，状态码为602 */
    public void setRedirect(String redirect, String error);
    
    /** 修改父页重定向前提示错误，状态码为601  */
    public void setRedirectParent(String redirect, String error);
    
    /** 修改顶页重定向前提示错误，状态码为603  */
    public void setRedirectTop(String redirect, String error);
    
    /***********************************************************************/
    //从请求中设置响应状态和文本属性
    /***********************************************************************/
    
    /** 获取响应属性码 */
    public int getResponseStatus();
    
    /** 获取响应属性文本 */
    public String getResponseText();
    
    /** 判断响应属性是不是成功 */
    public boolean isResponseSuccess();
    
    /** 判断响应属性是不是重定向 */
    public boolean isResponseRedirect();
    
    /** 设置响应结果属性，指定状态码但不能小于0，不能填70-699之间已知的状态码 */
    public void setResponseResult(int responseStatus, String responseText);
    
    /** 设置响应结果属性，指定状态码但不能小于0，不能填70-699之间已知的状态码 */
    public void setResponseResult(Int result);
    
    /** 设置响应结果属性，状态码为0，表示成功 */
    public void setResponseResult(String responseText);
    
    /** 设置响应结果属性，状态码为699，表示通用的失败 */
    public void setResponseError(String responseText);
    
    /**********************************************/
    //加上四个常用特性方法，用于运行时修改设置
    /**********************************************/
    
    /** 增加一个返回上一页快捷方式 */
    public void returnHistory();
    
    /** 增加一个错误返回上一页并提示错误的快捷方式 */
    public void returnHistory(String errorMsg);
    
    /** 增加一个弹出窗口时关闭快捷方式 */
    public void returnCloseWindow();
    
    /** 增加一个弹出窗口时关闭并提示错误的快捷方式 */
    public void returnCloseWindow(String errorMsg);
    
    /** 增加一个弹出对话框时关闭快捷方式 */
    public void returnCloseDialog();
    
    /** 增加一个弹出对话框时关闭并提示错误的快捷方式 */
    public void returnCloseDialog(String errorMsg);

    /** 增加参数 */
    public void addParam(String key, Object value);
    
    /*******************************************************************************/
    //转向控制字段相关属性
    /*******************************************************************************/

    /** 
     * 检查是否转向，两种情况<br><br>
     * 
     * 1、当设置成强制不转向时，即不转向<br>
     * 2、当为转向时，检查视图页和转向页是否存在，存在一个即转向，否则不转向<br>
      */
    public boolean isReturn();
    
    /** 获取Action在execute中转向的方法名 */
    public String getExecuteMethod();
    
    /** 获取转向定制的提示信息 */
    public String getAlertMsg();
    
    /** 获取视图信息 */
    public String getView();
    
    /** 获取视图信息嵌套地址 */
    public String getInclude();

    /** 获取内转向地址 */
    public String getForward(String encoding);
    
    /** 获取重定向地址 */
    public String getRedirect(String encoding);
    
    /** 获取重定向内部定义的地址 */
    public String getRedirectInner();
    
    /***********************************************/
    //以下判断是否强制要求转向
    /***********************************************/
    
    /** 
     * 是否强制转向
     * 
     * 1.如果有失败提示或成功提示，需强制转向
     * 2.由业务指定返回上一页或关闭，需强制转向
     */
    public boolean isForceReturn();
    
    /**
     * 获取强制转向类型
     * 
     * 1.返回上一页
     * 2.关闭当前窗口
     * 3.关闭对话框
     * 4.调用Javascript
     * 0.表示正常连接
     * @return int 类型
     */
    public int getForceReturnType();
    
    /*******************************************************************************/
    //验证相关属性
    /*******************************************************************************/
    
    /** 获取验证列表 */
    public List<Validate> getValidateList();
    
    /** 获取验证脚本 */
    public String getValidateScript();
    
    /** 获取验证提交脚本 */
    public String getValidateSubmitScript();
    
    /** 增加验证信息 */
    public void addValidate(Validate validate);
    
    /** 清除所有的验证信息 */
    public void clearValidate();
    
    /** 检查验证信息 */
    public boolean chkValidate();
    
    /** 设置验证确认信息 */
    public void setValidateConfirm(String confirm);
    
    /** 设置验证提交时是否按钮变灰，默认变灰 */
    public void setValidateGray(boolean isGray);
}
