/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd;

import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Urls;

/**
 * HTTP Cookie类
 * 1.name       Cookie名称
 * 2.value      Cookie值
 * 3.Version    Cookie遵循的版本
 * 4.Max-Age    Cookie有效期，为0表示不设置关闭浏览器即失效，单位：秒，替换Expires字段，告之浏览器保留时间，而不是到期时间
 * 5.Discard    Cookie有效期，是否在会话结束后丢弃该 Cookie 项，默认为 fasle
 * 6.Domain     Cookie指定的域，如果有域建议添加
 * 7.Port       Cookie指定的端口列表，多个逗号隔开，我们这里只支持一个，Version=1时引入，不过多大还不支持该属性
 * 8.Path       Cookie的虚拟目录，取值为contextPath，由HttpServer决定，不需要传入
 * 9.Secure     Cookie安全SSL时才传送
 * 10.HttpOnly   Cookie仅HTTP使用，由IE引入，大多都支持，强制填写
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class HttpCookie implements HttpdConstants
{
    private String name;
    private String value;
    private int maxAge;
    private boolean secure;
    
    public HttpCookie(String name, String value)
    {
        this.name = name;
        this.value = value;
    }
    
    public HttpCookie(String name, String value, int maxAge)
    {
        this.name = name;
        this.value = value;
        this.maxAge = maxAge;
    }
    
    public String getName()
    {
        return name;
    }
    
    public String getValue()
    {
        return value;
    }
    
    public int getMaxAge()
    {
        return maxAge;
    }
    
    public void setMaxAge(int maxAge)
    {
        this.maxAge = maxAge;
    }
    
    public void setName(String name)
    {
        this.name = name;
    }
    
    public void setValue(String value)
    {
        this.value = value;
    }
    
    public void setSecure(boolean secure)
    {
        this.secure = secure;
    }
    
    public boolean isSecure()
    {
        return secure;
    }
    
    /**************************************************************/
    //以下为自定义方法
    /**************************************************************/
    
    /**
     * 响应Cookie，支持PATH,DOMAIN和PORT
     * 
     * @param contextPath   上下文路径
     * @param domain        域名
     * @param port          端口
     * @return              Cookie响应
     */
    public String getResponse(String contextPath, String domain, int port)
    {
        return new StringBuilder(_SET_COOKIE_).append(_COLON_).append(getResponseValue(contextPath, domain, port)).toString();
    }
    
    /**
     * 响应Cookie，支持PATH,DOMAIN和PORT
     * 
     * @param contextPath   上下文路径
     * @param domain        域名
     * @param port          端口
     * @return              Cookie响应
     */
    public String getResponseValue(String contextPath, String domain, int port)
    {
        StringBuilder strb = new StringBuilder();
        //强制规定采用UTF8进行URL编码，读取时解码，保证统一，防止更换编码时读取异常
        strb.append(name).append("=").append(Urls.encodeUTF8(value));
        strb.append(";").append(_COOKIE_VERSION_).append("=").append(1);
        
        //默认不设置有效期即浏览器关闭即失效，如果有设置要求，则加入
        if (maxAge == 0)
            strb.append(";").append(_COOKIE_DISCARD_).append("=").append("\"true\"");
        else
            strb.append(";").append(_COOKIE_MAX_AGE_).append("=").append(maxAge);
        
        //强制加上域和端口
        if (domain != null)
            strb.append(";").append(_COOKIE_DOMAIN_).append("=").append(domain);
        
        if (port != -1)
            strb.append(";").append(_COOKIE_PORT_).append("=").append(port);
        
        //强制加上路径
        strb.append(";").append(_COOKIE_PATH_).append("=").append(Strings.addEndsWith(contextPath, "/"));
        
        //默认不增加SSL时才传递
        if (secure)
            strb.append(";").append(_COOKIE_SECURE_);
        
        //强制加上仅HTTP操作
        strb.append(";").append(_COOKIE_HTTP_ONLY);
        return strb.toString();
    }
}
