/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util.seqs;

import org.zhiqim.kernel.util.Arrays;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.DateTimes;
import org.zhiqim.kernel.util.Ints;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Validates;

/**
 * 整型序列值，默认从1从0x7FFFFFFF(最大值)，每天重新开始，并支持保存文件<br><br>
 * 1、最小值必须>=1<br>
 * 2、如果设置最大值，则最大值必须大小最小值，否则取整数最大值0x7FFFFFFF<br>
 * 3、是否自动保存
 * 
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class SequenceDateFile extends SequenceFile
{
    private String date = DateTimes.getDateString();
    
    /***********************************************************************************/
    //默认自动加载
    /***********************************************************************************/
    
    /** 从1到0x7FFFFFFF，自动加载 */
    public SequenceDateFile(String filePath)
    {
        super(filePath);
    }
    
    /** 从1到len个9，自动加载 */
    public SequenceDateFile(int len, String filePath)
    {
        super(len, filePath);
    }
    
    /** 从min到max，自动加载 */
    public SequenceDateFile(long min, long max, String filePath)
    {
        super(min, max, min, filePath);
    }
    
    /** 从min到max，指定初始值，自动加载 */
    public SequenceDateFile(long min, long max, long seq, String filePath)
    {
        super(min, max, seq, filePath);
    }
    
    /***********************************************************************************/
    //设置自动/手动加载，一般用于手动加载
    /***********************************************************************************/
    
    /** 从1到0x7FFFFFFF */
    public SequenceDateFile(String filePath, boolean auto)
    {
        super(filePath, auto);
    }
    
    /** 从1到len个9 */
    public SequenceDateFile(int len, String filePath, boolean auto)
    {
        super(len, filePath, auto);
    }
    
    /** 从min到max */
    public SequenceDateFile(long min, long max, String filePath, boolean auto)
    {
        super(min, max, filePath, auto);
    }
    
    /** 从min到max，指定初始值 */
    public SequenceDateFile(long min, long max, long seq, String filePath, boolean auto)
    {
        super(min, max, seq, filePath, auto);
    }
    
    public String getDate()
    {
        return date;
    }

    /********************************************************************/
    //判断日期加载和保存序列号，自动标志打开时，新建即加载，每次next即保存
    /********************************************************************/
    
    @Override
    public SequenceDateFile load()
    {
        String seqs = readFile();
        if (Validates.isEmpty(seqs))
            return this;
        
        String[] seqArr = Arrays.toStringArray(seqs);
        if (seqArr.length != 2)
            return this;
        
        String sdate = Strings.trim(seqArr[0]);
        if (!sdate.equals(date))
            return this;
            
        String seq = Strings.trim(seqArr[1]);
        if (!Validates.isIntegerPositive(seq))
            return this;
        
        setSequence(Ints.toInt(seq));
        return this;
    }
    
    @Override
    public void save()
    {////19位最大长整型，不足后补空格，保证每次写入时完整
        String seq = Strings.suffixLen(sequence, 19, _SPACE_);
        writeFile(new StringBuilder(date).append(",").append(seq).toString());
    }
    
    /********************************************************************/
    //重写获取方法，按天判断，和IntSequenceDate中的方法相同
    /********************************************************************/
    
    @Override
    public long next()
    {
        if (!isValid())
        {//创建就失败，但不在构造函数中报错，而在使用时报异常
            throw Asserts.exception("创建SequenceFile失败，文件目录不存在且创建失败");
        }
        
        synchronized (this)
        {
            String curDate = DateTimes.getDateString();
            if (curDate.equals(date)){
                chkSequence();
            }else{
                resetSequence();
                date = curDate;
            }
            
            long seq = sequence++;
            if (isAuto())
            {//保存下一个，返回当前
                save();
            }
            return seq;
        }
    }
}
