/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.util.seqs.Sequence;

/**
 * 线程相关工具类
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Threads")
public class Threads
{
    
    /**
     * 获取调用者堆栈对象
     * 
     * @return  调用者堆栈
     */
    public static StackTraceElement getInvoker()
    {
        StackTraceElement[] elems = Thread.currentThread().getStackTrace();
        return elems[3];//0是Thread本身，1是Threads，2是getInvoker方法的使用者，3才是正主，引入getInvoker使用的调用者
    }
    
    /**
     * 获取调用者类名、方法名和行数
     * 
     * @return  调用者堆栈字符串
     */
    public static String getInvokerString()
    {
        StackTraceElement[] elems = Thread.currentThread().getStackTrace();
        StackTraceElement elem = elems[3];
        return new StringBuilder()
            .append("[").append(elem.getClassName()).append("]")
            .append("[").append(elem.getMethodName()).append("]")
            .append("[第").append(elem.getLineNumber()).append("行]")
            .toString();
    }
    
    /**
     * 获取调用者类名、方法名和行数简写
     * 
     * @return  调用者堆栈字符串
     */
    public static String getInvokerAbbr()
    {
        StackTraceElement[] elems = Thread.currentThread().getStackTrace();
        StackTraceElement elem = elems[3];
        String className = Strings.removeLeftByLast(elem.getClassName(), ".");
        
        return new StringBuilder()
            .append("[")
            .append(className).append(".").append(elem.getMethodName())
            .append("(第").append(elem.getLineNumber()).append("行)")
            .append("]")
            .toString();
    }

    /***********************************************************************************/
    //新建线程
    /***********************************************************************************/
    
    
    /**
     * 生成一个新的线程
     * 
     * @param runnable 线程运行类
     * @return Thread
     */
    public static Thread newThread(Runnable runnable)
    {
        return new Thread(runnable);
    }
    
    /**
     * 生成一个新的线程
     * 
     * @param runnable  线程运行类
     * @param name      线程名
     * @return          Thread
     */
    public static Thread newThread(Runnable runnable, String name)
    {
        return new Thread(runnable, name);
    }
    
    /**
     * 生成一个新的线程
     * 
     * @param runnable  线程运行类
     * @param name      线程名
     * @param sequence  序号
     * @return          Thread
     */
    public static Thread newThread(Runnable runnable, String name, int sequence)
    {
        return new Thread(runnable, name+"-"+sequence);
    }
    
    
    /**
     * 生成一个新的线程
     * 
     * @param runnable  线程运行类
     * @param name      线程名
     * @param sequence  序号表
     * @return          Thread
     */
    public static Thread newThread(Runnable runnable, String name, Sequence sequence)
    {
        return new Thread(runnable, name+"-"+sequence.next());
    }
    
    /**
     * 生成一个新的线程
     * 
     * @param group     线程组
     * @param runnable  线程运行类
     * @param sequence  序号
     * @return          Thread
     */
    public static Thread newThread(ThreadGroup group, Runnable runnable, int sequence)
    {
        return new Thread(group, runnable, group.getName()+"-"+sequence);
    }
    
    /**
     * 生成一个新的线程
     * 
     * @param group     线程组
     * @param runnable  线程运行类
     * @param sequence  序号表
     * @return          Thread
     */
    public static Thread newThread(ThreadGroup group, Runnable runnable, Sequence sequence)
    {
        return new Thread(group, runnable, group.getName()+"-"+sequence.next());
    }
    
    /***********************************************************************************/
    //中断线程
    /***********************************************************************************/
    
    /**
     * 忽略线程睡眠时的中断异常，使用时放置到while最下端
     * 
     * @param millis    毫秒数
     */
    public static void sleepIgnoreException(int millis)
    {
        if (millis <= 0)
            return;
        
        try{Thread.sleep(millis);}catch (InterruptedException e){}
    }
    
    /** 关闭关忽略异常 */
    public static void closeIgnoreException(AutoCloseable closeable)
    {
        if (closeable == null)
            return;
        
        try{closeable.close();}catch (Throwable e){}
    }
}
