/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.zip.Deflater;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.control.Filter;
import org.zhiqim.kernel.control.FilterEndsWith;

/**
 * JAR包相关工具类
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Jars")
public class Jars
{
    /**
     * 通过JarEntry获取类名
     * 
     * @param jarEntry  类实体
     * @return          类名
     */
    public static String getClassName(JarEntry jarEntry)
    {//格式如:org/zhiqim/kernel/Global.class
     //1.去掉.class的后缀，2.把/替换成 .
        String className = jarEntry.getName();
        className = Strings.trimRight(className, ".class");
        className = Strings.replaceAll(className, "/", ".");
        return className;
    }
    
    /**
     * 查询JAR包中后缀列表
     * 
     * @param file      JAR文件
     * @param filter    过滤器
     * @return          查询到的实体文件
     */
    public static List<JarEntry> queryEndsWithList(JarFile file, Filter filter)
    {
        List<JarEntry> list = new ArrayList<JarEntry>();
        Enumeration<JarEntry> enumeration = file.entries(); 
        while (enumeration.hasMoreElements()) 
        { 
            JarEntry entry = enumeration.nextElement();
            String name = entry.getName(); 
            if (filter.access(name))
                list.add(entry);
        }
        
        return list;
    }
    
    /**
     * 查询JAR包中后缀列表
     * 
     * @param file      JAR文件
     * @param endsWith  后缀名
     * @return          查询到的实体文件
     */
    public static List<JarEntry> queryEndsWithList(JarFile file, String endsWith)
    {
        List<JarEntry> list = new ArrayList<JarEntry>();
        Enumeration<JarEntry> enumeration = file.entries(); 
        while (enumeration.hasMoreElements()) 
        { 
            JarEntry entry = enumeration.nextElement();
            String name = entry.getName(); 
            if (Validates.isEmpty(endsWith) || name.endsWith(endsWith))
                list.add(entry);
        }
        
        return list;
    }
    
    /**
     * 查询JAR包中后缀列表
     * 
     * @param entryList 要求传入列表，匹配到数据放入该列表
     * @param file      JAR文件
     * @param endsWith  后缀名
     */
    public static void queryFilterList(List<JarEntry> entryList, JarFile file, Filter filter)
    {
        List<JarEntry> list = new ArrayList<JarEntry>();
        Enumeration<JarEntry> enumeration = file.entries(); 
        while (enumeration.hasMoreElements()) 
        { 
            JarEntry entry = enumeration.nextElement();
            String name = entry.getName(); 
            if (filter.access(name))
                list.add(entry);
        }
    }
    
    /**
     * 查询JAR包中后缀列表
     * 
     * @param entryList 要求传入列表，匹配到数据放入该列表
     * @param file      JAR文件
     * @param endsWith  后缀名
     */
    public static void queryEndsWithList(List<JarEntry> entryList, JarFile file, String endsWith)
    {
        Filter filter = new FilterEndsWith().param(endsWith);
        
        queryEndsWithList(file, filter);
    }
    
    /**
     * jar打包压缩文件
     * 
     * @param fileDirPath           文件目录路径
     * @param jarExportPath         JAR导出路径
     * @throws Exception
     */
    public static void jar(String fileDirPath, String jarExportPath) throws Exception
    {
        Asserts.notEmpty(fileDirPath, "文件目录路径");
        Asserts.notEmpty(jarExportPath, "JAR导出路径");
        
        File fileDir = new File(fileDirPath);
        Asserts.as(fileDir.exists()?null:"文件目录路径不存在");

        //转化为/方式的绝对路径
        String fileDirCanonicalPath = Files.toLinuxPath(fileDir.getCanonicalPath());
        
        FileOutputStream fos = new FileOutputStream(jarExportPath);
        JarOutputStream jos = new JarOutputStream(fos);
        jos.setLevel(Deflater.BEST_COMPRESSION);
        
        List<File> fileList = new ArrayList<>();
        Files.queryFileList(fileList, fileDir, 0);
        
        for (File file : fileList)
        {
            String name = Files.toLinuxPath(file.getCanonicalPath());
            if (file.isDirectory())
                name = name.substring(fileDirCanonicalPath.length()+1) + "/";
            else
                name = name.substring(fileDirCanonicalPath.length()+1);
            
            JarEntry ze = new JarEntry(name);
            ze.setSize(file.length());
            ze.setTime(file.lastModified());
            jos.putNextEntry(ze);
            
            if (ze.isDirectory())
            {//目录
                jos.closeEntry();
            }
            else
            {//文件
                try (InputStream is = new FileInputStream(file))
                {
                    Streams.putBytes(is, jos);
                }
                jos.closeEntry();
            }
        }

        jos.close();
    }
}
