/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.json;

import java.util.ArrayList;
import java.util.List;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.constants.AsciiConstants;
import org.zhiqim.kernel.extend.HashMapSO;
import org.zhiqim.kernel.extend.HashMapSS;
import org.zhiqim.kernel.extend.HashMapSV;
import org.zhiqim.kernel.extend.MapSO;
import org.zhiqim.kernel.json.parser.ArrayParser;
import org.zhiqim.kernel.json.parser.MapParser;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Validates;

/**
 * JSON相关工具类
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Jsons")
public class Jsons implements AsciiConstants
{
    /********************************************************/
    //提供静态的toString和toObject方法,取标准JSON参数
    /********************************************************/
    
    /**
     * 默认的对象转JSON字符串方法，不同的参数请单独使用Json类
     * 
     * @param obj   对象
     * @return      返回JSON字符串
     */
    public static String toString(Object obj)
    {
        return Json.json.toString(obj);
    }
    
    /**
     * 默认的JSON字符串转换成对象，不同的参数请单独使用Json类
     * 
     * @param str   JSON字符串
     * @param cls   转换成对象的类
     * @return      对应的对象
     */
    public static <T> T toObject(String str, Class<T> cls)
    {
        return Json.json.toObject(str, cls, true);
    }
    
    
    /********************************************************/
    //以下解析JSON时需要的字符串分析方法
    /********************************************************/
    
    /**
     * 判断字符串是否是由引号包括，以判断是否是一个指定的字符串
     * 
     * @param str   字符串
     * @return      =true表示是由引号包括
     */
    public static boolean isStartEndQuotation(String str)
    {
        return str != null && str.length() >= 2 && ((str.startsWith("\"") && str.endsWith("\"")) || (str.startsWith("\'") && str.endsWith("\'")));
    }
    
    /***
     * 去除JSON键和值的前后成对引号
     * 
     * @param str   原字符串
     * @return      去除成对引号之后的字符串
     */
    public static String removeStartEndQuotation(String str)
    {
        if (str == null)
            return null;
        
        if (str.length() >= 2 && str.startsWith("\"") && str.endsWith("\""))
        {//有双引号删除退出
            str = str.substring(1, str.length()-1);
            return str;
        }
        
        //没有双引号则判断单引号
        if (str.length() >= 2 && str.startsWith("\'") && str.endsWith("\'"))
            str = str.substring(1, str.length()-1);
        
        return str;
    }
    
    /**
     * 增加JSON中的转义字符，使用双引号时，单引号不转义，使用单引号时双引号不转义，不使用引号时都转义
     * 
     * @param str       原字符串
     * @param quotation 使用的引号 =0表示未使用,='表示单引号,="表示双引号
     * @return          对字符中需要转义的字符，增加转义符
     */
    public static String addEscapeChar(String str, char quotation)
    {
        if (str == null)
            return null;
        
        StringBuilder strb = new StringBuilder();
        for(int i=0;i<str.length();i++)
        {
            char c = str.charAt(i);
            switch (c)
            {
            case '\\':strb.append("\\\\");break;
            case '\"':if (quotation == 0 || quotation == '\"')strb.append("\\\"");else strb.append(c);break;
            case '\'':if (quotation == 0 || quotation == '\'')strb.append("\\\'");else strb.append(c);break;//单引号或无引号时单引号要转义，双引号下的单引号无需处理
            case '\b':strb.append("\\b");break;
            case '\f':strb.append("\\f");break;
            case '\r':strb.append("\\r");break;
            case '\n':strb.append("\\n");break;
            case '\t':strb.append("\\t");break;
            case '/':strb.append("\\/");break;
            default:strb.append(c);break;
            }
        }

        return strb.toString();
    }
    
    /***
     * 去除JSON中的转义字符
     * 
     * @param str   原字符串
     * @return      去除成对引号之后的字符串
     */
    public static String removeEscapeChar(String str)
    {
        if (str == null)
            return null;
        
        StringBuilder strb = new StringBuilder();
        boolean isEscape = false;//是否前一字符是转义字符
        for(int i=0;i<str.length();i++)
        {
            char c = str.charAt(i);
            if (!isEscape)
            {//未转义
                if (c == '\\')
                    isEscape = true;//设为有转义
                else
                    strb.append(c);
            }
            else
            {//有转义
                switch (c)
                {
                case '\\':strb.append('\\');break;
                case '\"':strb.append('\"');break;
                case '\'':strb.append('\'');break;
                case 'b':strb.append('\b');break;
                case 'f':strb.append('\f');break;
                case 'n':strb.append('\n');break;
                case 'r':strb.append('\r');break;
                case 't':strb.append('\t');break;
                case '/':strb.append('/');break;
                default:strb.append("\\").append(c);break;//如果未找到匹配,则返原值
                }
                isEscape = false;//重置转义为结束
            }
        }
        
        if (isEscape)
        {//最后一个字符是\
            strb.append("\\");
        }
        
        return strb.toString();
    }
    
    /**
     * 去除JSON的空字符串，包括空格,\t,\r,\n
     * 
     * @param str   原字符串
     * @return      去除空字符串之后的字符串
     */
    public static String removeBlankSpace(String str)
    {
        if (str == null) 
            return null;
        
        str = Strings.trim(str);
        StringBuilder strb = new StringBuilder();
        boolean isRemove = false;
        for (int i=0;i<str.length();i++)
        {
            char c = str.charAt(i);
            
            if (isRemove)
            {
                if (Validates.isWhitespace(c))
                    continue;//如果允许删除（不是内容的一部分），则<=32的字符都删除
            }
            
            switch (c)
            {
            case '{':
            case '[':
            {//开始标志,下一个遇到空字符串删除
                isRemove = true;
                break;
            }
            case '}':
            case ']':
            case ',':
            case ':':
            {//结束标志(含KEY,VALUE的结束),对之前的空格处理，如{"abc" : "abc"}第一个空格
                int count = 0;
                for (int j=strb.length()-1;j>=0;j--)
                {
                    char cj = strb.charAt(j);
                    if (!Validates.isWhitespace(cj))
                        break;
                    count++;
                }
                
                if (count > 0)
                {//存在空格则删除
                    strb.setLength(strb.length()-count);
                }
                isRemove = true;
                break;
            }
            default:
            {//非标志的不删除
                isRemove = false;
                break;
            }
            }
            
            strb.append(c);
        }
        
        return strb.toString();
    }
    
    /**
     * 获取字段对应的值起始索引值
     * 
     * @param json      JSON字符串
     * @param field     字段名
     * @return          索引值
     */
    public static int getIndex(String json, String field)
    {
        int fieldLen = field.length();
        
        //先判断双引号情况
        int ind = json.indexOf("\""+field+"\":");
        if (ind != -1)
            return ind + 1 + fieldLen + 1 + 1;//"field":
        
        //再判断无引号情况
        ind = json.indexOf(field+":");
        if (ind != -1)
            return ind + fieldLen + 1;//field:
        
        //最后判断单引号情况
        ind = json.indexOf("\'"+field+"\':");
        if (ind != -1)
            return ind + 1 + fieldLen + 1 + 1;//'field':
        
        return -1;
    }
    
    /** 
     * 获取字段对应的值，支持{}对象格式，[]数组格式和("",'',无引号)字符串格式。
     * 
     * @param json      JSON字符串
     * @param field     字段名
     * @return          得到的值
     */
    public static final String getValue(String json, String field)
    {
        json = removeBlankSpace(json);
        if (Validates.isEmpty(json))
            return null;
        
        int ind = getIndex(json, field);
        if (ind == -1)
            return null;
        
        char firstChar = json.charAt(ind);
        switch (firstChar)
        {
        case '{':return getObject(json, field);//对象
        case '[':return getArray(json, field);//数组
        case '\"'://双引号字符串
        case '\''://单引号字符串
        default://无引号字符串
            return getString(json, field);
        }
    }
    
    /**
     * 判断是否是字符串开始
     * 
     * @param quotation 引号值，取值为=0,='\''和='\"'三种可能=0表示当前没有引号
     * @param c         当前字符
     * @param lastChar  当前字符的前一个字符
     * @return          =true表示字符串开始
     */
    public static boolean isStringStart(char quotation, char c, char lastChar)
    {//字符串开始,
        //1.要求前面没有引号,
        //2.当前是引号,
        //3.前第一字符为开始字符(这样前第一字符肯定不是转义符,保证引号有效)
        return (quotation == 0 && (c == '\"' || c == '\'') && (lastChar == ':' || lastChar == ',' || lastChar == '{' || lastChar == '['));
    }
    
    /**
     * 判断是否是字符串结束
     * 
     * @param i         当前字符索引
     * @param lastIndex 上次字符串开始索引
     * @param quotation 引号值，取值为=0,='\''和='\"'三种可能=0表示当前没有引号
     * @param c         当前字符
     * @param lastChar  当前字符的前一个字符
     * @param last2Char 当前字符的前二个字符
     * @return          =true表示字符串结束
     */
    public static boolean isStringEnd(int i, int lastIndex, char quotation, char c, char lastChar, char last2Char)
    {//字符串结束,
        //1.要求位置大于1(否则会出同一个引号后面跟逗号问题),
        //2.当前字符为结束字符,
        //3.前第一个字符为引号,
        //4.前第二个字符不是转义字符
        return (lastIndex < (i-1) && (c == ':' || c == ',' || c == '}' || c == ']') && lastChar == quotation && last2Char != '\\');
    }
    
    /**
     * 获取对象或数组内的字段列表
     * 1.对象格式：{"abc":"a{b}c"},{"abc":"a{b}c","bcd":"bcd"},和{"abc":"{'bcd':'efg','aaa':'bbb'}"}
     * 2.数组格式：[1,2],["1",2],["a{b}c","bcd"],[[1,2],["[","]","}","{"]]
     * 
     * @param str   字符串
     * @return      得到分离成数组列表或对象字段列表
     */
    public static List<String> getFieldList(String str)
    {
        ArrayList<String> fieldList = new ArrayList<String>();
        
        //定义对象和数组成对出现的数据,表示现{则objNum++,}则objNum--，通过objNum>0来检查当前是否是在该对象中,[]类似
        int objNum = 0, arrNum = 0;StringBuilder temp = new StringBuilder();
        boolean isString = false;char quotation =0;char lastChar = 0;char last2Char = 0;int lastIndex = 0;
        
        for (int i=0;i<str.length();i++)
        {
            char c = str.charAt(i);
            if (isStringStart(quotation, c, lastChar))
            {
                quotation = c;
                lastIndex = i;
                isString = true;
            }
            
            if (isStringEnd(i, lastIndex, quotation, c, lastChar, last2Char))
            {
                quotation = 0;
                lastIndex = 0;
                isString = false;
            }
            
            //置历史数据
            last2Char = lastChar;
            lastChar = c;
            
            if (isString)
            {//还是该字段的字符串值
                temp.append(c);
                continue;
            }
            
            switch (c)
            {
            case '{':objNum++;break;
            case '}':objNum--;break;
            case '[':arrNum++;break;
            case ']':arrNum--;break;
            }
                
            if (objNum > 0 || arrNum > 0)
            {//还是该字段中的对象或数组值
                temp.append(c);
                continue;
            }

            if (c != ',')
            {//字段未结束
                temp.append(c);
                continue;
            }
            
            //字段结束
            if (temp.length() > 0)
                fieldList.add(temp.toString());
            
            //重置值
            temp = null;
            temp = new StringBuilder();
        }
        
        //最后一个结束的加上
        if (temp.length() > 0)
            fieldList.add(temp.toString());
        
        fieldList.trimToSize();
        return fieldList;
    }
    
    /********************************************************/
    //以下根据field获取json中对应的value的简单解析字符串方法
    //1.getObject,  格式如{"abc":{}}，其中getMap为getMap的变化
    //2.getArray,   格式如{"abc":["1","2"]}，其中getList为getArray的变化
    //3.getString,  格式如{"abc":"def"}，其中getLong,getInt,getBoolean为getString的变化
    /********************************************************/
    
    /**
     * 获取字段对应的字符串值，支持双引号、单引号和无引号三种字符串值，如果对应的值是{或[开头则认为是对象或数组
     * TODO 可能会出现{abc:"abc}这种情况，表示值="abc，这种情况认为格式错误，应该写成{abc:\"abc}
     * 
     * @param json      JSON字符串
     * @param field     字段名
     * @return          得到的字符串值
     */
    public static String getString(String json, String field)
    {
        json = removeBlankSpace(json);
        if (Validates.isEmpty(json))
            return null;
        
        int ind = getIndex(json, field);
        if (ind == -1)
            return null;
        
        char quotation = 0;
        char firstChar = json.charAt(ind);
        switch (firstChar)
        {
        case '{':return getObject(json, field);
        case '[':return getArray(json, field);
        case '\"':quotation = '\"';break;
        case '\'':quotation = '\'';break;
        }

        //如果是字符串，则有三种情况1:双引号,2:单引号,3:无引号
        StringBuilder strb = new StringBuilder();
        
        //从第二个字符开始
        boolean isEscape = false;//是否前一字符是转义字符
        if (firstChar == '\\')
            isEscape = true;
        else
            strb.append(firstChar);//如果不是转义字符则把第一个字符放入
        
        for (int i=(ind+1);i<json.length();i++)
        {
            char c = json.charAt(i);
            if (quotation == 0)
            {//无符号字符串遇到结束符号则退出，如果前一字符为转义字符加上退出
                if (c == ',' || c == '}')
                {
                    if (isEscape)
                        strb.append("\\");
                    break;
                }
            }
                
            if (!isEscape)
            {//未转义
                if (c == '\\')
                    isEscape = true;//如果是\，且前面没有\，表示转义开始，该字符暂不写入，在下一字符判断
                else
                {
                    strb.append(c);//其他数据直接加入
                    
                    //有引号字符串，引号成对出现则退出
                    if (c == '\"' && quotation == '\"')
                        break;//双引号成对
                    
                    if (c == '\'' && quotation == '\'')
                        break;//单引号成对
                }
            }
            else
            {//转义开始
                switch (c)
                {
                case '\"':strb.append('\"');break;//转义条件下不判断成对
                case '\'':strb.append('\'');break;//转义条件下不判断成对
                case '\\':strb.append('\\');break;
                case 'b':strb.append('\b');break;
                case 'f':strb.append('\f');break;
                case 'n':strb.append('\n');break;
                case 'r':strb.append('\r');break;
                case 't':strb.append('\t');break;
                case '/':strb.append('/');break;//TODO 该字符遇到也要求转义
                default:strb.append("\\").append(c);break;//如果未找到匹配,则返原\值
                }
                isEscape = false;//重置转义为结束
            }
        }

        //对最后得到的字符串比较，null要特别判断
        String value = strb.toString();
        if ("null".equals(value))
            return null;
        
        //其他情况，要求去除引号
        return removeStartEndQuotation(value);
    }
    
    /**
     * 获取字段对应的对象字符串值，得到结果为{}开头和结束
     * TODO 可能会出现{abc:"abc}这种情况，表示值="abc，这种情况认为格式错误，应该写成{abc:\"abc}
     * 
     * @param json      JSON字符串
     * @param field     字段名
     * @return          得到的字符串值，{}开头和结束
     */
    public static final String getObject(String json, String field)
    {
        json = removeBlankSpace(json);
        if (Validates.isEmpty(json))
            return null;
        
        int ind = getIndex(json, field);
        if (ind == -1)
            return null;
        
        //第一字符不是{
        if (json.charAt(ind) != '{')
            return null;
        
        //原理为分析{}成对出现，当和第1个成对即结束，在字符串值中的{}不计数
        StringBuilder strb = new StringBuilder();int num = 0;
        boolean isString = false;char quotation =0;char lastChar = 0;char last2Char = 0;int lastIndex = 0;
        for (int i=ind;i<json.length();i++)
        {
            char c = json.charAt(i);
            if (isStringStart(quotation, c, lastChar))
            {
                quotation = c;
                lastIndex = i;
                isString = true;
            }
            
            if (isStringEnd(i, lastIndex, quotation, c, lastChar, last2Char))
            {
                quotation = 0;
                lastIndex = 0;
                isString = false;
            }
            
            //置历史数据
            last2Char = lastChar;
            lastChar = c;
            
            if (!isString)
            {//如果不是字符串内容出现{}，计数
                switch (c)
                {
                case '{':num++;break;
                case '}':num--;break;
                }
            }
            
            strb.append(c);//暂不管是否转义，统一填充，由getString等具体方法处理转义
            if (num == 0)
                break;//成对则结束
        }
        
        return strb.toString();
    }
    
    /**
     * 获取字段对应的数组字符串值，得到结果为[]开头和结束
     * 
     * @param json      JSON字符串
     * @param field     字段名
     * @return          得到的字符串值，[]开头和结束
     */
    public static final String getArray(String json, String field)
    {
        json = removeBlankSpace(json);
        if (Validates.isEmpty(json))
            return null;
        
        int ind = getIndex(json, field);
        if (ind == -1)
            return null;
        
        //第一字符不是[
        if (json.charAt(ind) != '[')
            return null;
        
        //原理为分析[]成对出现，当和第1个成对即结束，在字符串值中的[]不计数
        StringBuilder strb = new StringBuilder();int num = 0;
        boolean isString = false;char quotation =0;char lastChar = 0;char last2Char = 0;int lastIndex = 0;
        for (int i=ind;i<json.length();i++)
        {
            char c = json.charAt(i);          
            if (isStringStart(quotation, c, lastChar))
            {
                quotation = c;
                lastIndex = i;
                isString = true;
            }
            
            if (isStringEnd(i, lastIndex, quotation, c, lastChar, last2Char))
            {
                quotation = 0;
                lastIndex = 0;
                isString = false;
            }
            
            //置历史数据
            last2Char = lastChar;
            lastChar = c;
            
            if (!isString)
            {//如果不是字符串内容出现[]，计数
                switch (c)
                {
                case '[':num++;break;
                case ']':num--;break;
                }
            }
            
            strb.append(c);//暂不管是否转义，统一填充，由getString等具体方法处理转义
            if (num == 0)
                break;//成对则结束
        }
        
        return strb.toString();
    }
    
    /** 获取字符串数组列表 */
    public static List<String> getList(String json, String field)
    {
        String arrJson = getArray(json, field);
        return ArrayParser.toFieldList(arrJson);
    }
    
    /** 获取long,如果未取到或不是long，则取缺省值 */
    public static final long getLong(String json, String field, long defaultValue)
    {
        String num = getString(json, field);
        if (!Validates.isInteger(num))
            return defaultValue;
        
        return Long.parseLong(num);
    }
    
    /** 获取long */
    public static final long getLong(String json, String field)
    {
        String num = getString(json, field);
        if (!Validates.isInteger(num))
            return -1;
        
        return Long.parseLong(num);
    }
    
    /** 获取int,如果未取到或不是int，则取缺省值 */
    public static final int getInt(String json, String field, int defaultValue)
    {
        String num = getString(json, field);
        if (!Validates.isInteger(num))
            return defaultValue;
        
        return Integer.parseInt(num);
    }
    
    /** 获取int */
    public static final int getInt(String json, String field)
    {
        String num = getString(json, field);
        if (!Validates.isInteger(num))
            return -1;
        
        return Integer.parseInt(num);
    }
    
    /** 获取boolean */
    public static final boolean getBoolean(String json, String field)
    {
        String str = getString(json, field);
        return "true".equalsIgnoreCase(str);
    }
    
    /** 获取boolean */
    public static final boolean getBoolean(String json, String field, boolean defaultValue)
    {
        String str = getString(json, field);
        if (str == null)
            return defaultValue;
        
        return "true".equalsIgnoreCase(str);
    }
    
    /********************************************************/
    //以下根据toObject和toString的变化
    /********************************************************/
    
    /**
     * 未提供泛型，得到值为基本类型的HashMapSO，如果值是对象如显示为对象的JSON字符串，基本类型转为boolean,long和double三种
     * 
     * @param json  JSON字符串
     * @return      HashMapSO
     */
    public static HashMapSO toMapSO(String json)
    {
        return Json.json.toObject(json, HashMapSO.class, true);
    }
    
    /**
     * 未提供泛型，得到值为基本类型的HashMapSS，如果值是对象如显示为对象的JSON字符串
     * 
     * @param json  JSON字符串
     * @return      HashMapSS
     */
    public static HashMapSS toMapSS(String json)
    {
        return Json.json.toObject(json, HashMapSS.class, true);
    }
    
    /**
     * 提供泛型，得到值为对象的哈唏表
     * 
     * @param json          JSON字符串
     * @param genericsClass 泛型
     * @return              HashMapSV<V>
     */
    public static <V> HashMapSV<V> toMapSV(String json, Class<V> genericsClass)
    {
        List<String> fieldList = MapParser.toFieldList(json);
        HashMapSV<V> map = new HashMapSV<>();
        for (String field : fieldList)
        {
            int ind = field.indexOf(":");
            String key = removeStartEndQuotation(field.substring(0, ind).trim());
            String value = field.substring(ind+1).trim();
            if ("null".equals(value))
                map.put(key, null);
            else
            {
                V v = toObject(value, genericsClass);
                map.put(key, v);
            }
        }
        return map;
    }
    
    /**
     * 提供泛型，得到列表
     * 
     * @param json          JSON字符串
     * @param genericsClass 泛型
     * @return              List<T>
     */
    public static <T> List<T> toList(String json, Class<T> genericsClass)
    {
        List<String> fieldList = ArrayParser.toFieldList(json);
        List<T> list = new ArrayList<T>(fieldList.size());
        for (String field : fieldList)
        {
            T o = Jsons.toObject(field, genericsClass);
            list.add(o);
        }
        return list;
    }
    
    /**
     * 提供两个键值生成JSON字符串，对应MAP的一次put
     * 
     * @param key       键
     * @param value     值
     * @return 生成后的JSON字符串
     */
    public static String toString(String key, Object value)
    {
        MapSO map = new HashMapSO(1);
        map.put(key, value);
        return toString(map);
    }
    
    /**
     * 提供两个键值生成JSON字符串，对应MAP的两次put
     * 
     * @param key       第一个键
     * @param value     第一个值
     * @param key2      第二个键
     * @param value2    第二个值
     * @return          生成后的JSON字符串
     */
    public static String toString(String key, Object value, String key2, Object value2)
    {
        MapSO map = new HashMapSO(2);
        map.put(key, value);
        map.put(key2, value2);
        return toString(map);
    }
    
    /**
     * 从一个对象或MAP的json中增加或更新指定的一个KEY
     * 
     * @param json      对象或MAP对应的JSON字符串
     * @param key       对象的字段或MAP的键
     * @param value     值
     * @return          增加或更新KEY之后的json
     */
    public static String toStringAddOrUpdate(String json, String key, Object value)
    {
        MapSO jsonMap = toMapSO(json);
        jsonMap.put(key, value);
        
        return toString(jsonMap);
    }
    
    /**
     * 从一个对象或MAP的json中删除指定的一个KEY
     * 
     * @param json      对象或MAP对应的JSON字符串
     * @param key       对象的字段或MAP的键
     * @return          删除KEY之后的json
     */
    public static String toStringRemove(String json, String key)
    {
        MapSO jsonMap = toMapSO(json);
        jsonMap.remove(key);
        
        return toString(jsonMap);
    }
}
