/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.control;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

import org.zhiqim.kernel.Global;
import org.zhiqim.kernel.config.Group;
import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Randoms;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Threads;

/**
 * 线程池控制器，指定池最小值、最大值、初始值，允许线程空闲时长和允许缓存事件大小<br><br>
 * 1.创建时创建初始值大小的线程池，创建事件队列，和主线程<br>
 * 2.其他线程通过post传递事件可执行事件<br>
 * 3.主线程调度线程池处理事件，并检查线程池空闲线程
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class ThreaderController extends ThreadServicer
{
    private static final Log log = LogFactory.getLog(ThreaderController.class);
    
    private int poMinSize;                         //线程池最小数目
    private int poMaxSize;                         //线程池最大数目
    private int poIdleTimeMs;                      //线程池线程空闲时长(毫秒)
    private int poQueueSize;                       //线程池队列大小
    
    private ThreadGroup threadGroup;                //指定线程组
    private String threadName;                      //指定线程名
    
    private final ArrayList<Threader> theaderList;  //线程列表
    private final Queue<Runnable> queue;            //等待处理的可执行事件
    private final AtomicLong completedCount;        //总完成处理数
    
    public ThreaderController()
    {
        threadName = "Threader-" + Randoms.upperLetters(1)+Randoms.lowerLettersDigits(3);
        
        theaderList = new ArrayList<>();
        queue = new Queue<Runnable>();
        completedCount = new AtomicLong();
    }
    
    /*********************************************************************************************/
    //参数设置&创建&销毁
    /*********************************************************************************************/
    
    /**
     * 设置线程池必须的四个参数
     * 
     * @param poMinSize     线程池最小值
     * @param poMaxSize     线程池最大值
     * @param poIdleTime    线程池线程空闲时长，单位秒
     * @param poQueueSize   线程池允许缓存事件大小
     */
    public ThreaderController set(int poMinSize, int poMaxSize, int poIdleTime, int poQueueSize)
    {
        this.poMinSize = (poMinSize < 1)?1:poMinSize;
        this.poMaxSize = (poMaxSize < poMinSize)?poMinSize:poMaxSize;
        this.poIdleTimeMs = ((poIdleTime < 180)?180:poIdleTime) * 1000;
        this.poQueueSize = poQueueSize<(this.poMaxSize*3)?(this.poMaxSize*3):poQueueSize;
        
        return this;
    }
    
    public void setThreadName(String threadName)
    {
        this.threadName = Strings.toUpperFirstChar(threadName);
    }
    
    @Override
    protected ThreadGroup getThreadGroup()
    {
        return threadGroup;
    }
    
    @Override
    protected String getThreadName()
    {
        return threadName + "-Controller";
    }
    
    @Override
    public boolean openBefore()
    {
        //1.参数检查
        if (id == null)
        {//非配置启动，必须设置值
            Asserts.as((poMaxSize >= poMinSize && poMinSize > 0 && poIdleTimeMs > 0 && poQueueSize > 0)?null:"线程池服务配置参数不正确");
        }
        else
        {//配置启动，检查配置项
            Group grp = Global.getGroup(id);
            if (grp == null)
            {
                log.error("线程池服务[%s]未配置参数", id);
                return false;
            }
            
            threadName = Strings.toUpperFirstChar(id);
            set(grp.getInt("poMinSize"), grp.getInt("poMaxSize"), grp.getInt("poIdleTime"), grp.getInt("poQueueSize"));
        }
        
        //2.初始化创建线程操作者
        this.threadGroup = new ThreadGroup(threadName);
        for (int i=0;i<poMinSize;i++)
        {
            openThreader();
        }
        
        return true;
    }
    
    @Override
    public void closeAfter()
    {
        closeThreaders();
        log.info("线程池[%s]监视线程退出", threadName);
    }
    
    @Override
    public void loop()
    {
        if (theaderList.size() > poMinSize)
        {//如果线程池中线程比最小值大，则检查线程池是否有空闲线程，有则关闭
            chkThreaders();
        }
        
        //等待下次检查
        Threads.sleepIgnoreException(poIdleTimeMs);
    }
    
    /*********************************************************************************************/
    //线程池管理
    /*********************************************************************************************/
    
    /** 开启一个新的线程 */
    private void openThreader()
    {
        synchronized (theaderList)
        {
            Threader threader = new Threader(this);
            threader.open();
            theaderList.add(threader);
        }
    }
    
    /** 关闭所有线程 */
    private void closeThreaders()
    {
        synchronized (theaderList)
        {
            for (Threader threader : theaderList)
            {
                threader.close();
                threader = null;
            }
            theaderList.clear();
        }
    }
    
    /** 检查线程池 */
    private void chkThreaders()
    {
        synchronized (theaderList)
        {
            for (Iterator<Threader> it=theaderList.iterator();it.hasNext();)
            {
                if (theaderList.size() <= poMinSize)
                    break;
                
                Threader threader = it.next();
                if (threader.isExecessMaxIdleTime(poIdleTimeMs))
                {
                    it.remove();
                    threader.close();
                }
            }
        }
    }
    
    /*********************************************************************************************/
    //线程池处理，处理事件&清理事件&判断事件是否存在
    /*********************************************************************************************/
    
    public boolean execute(Runnable command)
    {
        return execute(command, false);
    }
    
    public synchronized boolean execute(Runnable command, boolean priority)
    {
        if (!isRunning() || command == null)
            return false;
        
        int threaderSize = theaderList.size();
        int activeSize = getThreaderActiveSize();
        if (activeSize < threaderSize)
        {//线程池空闲
            queue.post(command, priority);
            return true;
        }
        
        if (threaderSize < poMaxSize)
        {//线程池忙，但未到繁忙
            openThreader();
            queue.post(command, priority);
            return true;
        }
        
        if (queue.size() < poMaxSize)
        {//线程池繁忙，缓存一个批次
            queue.post(command, priority);
            return true;
        }
        
        if (queue.size() < poQueueSize)
        {//线程池繁忙，接近最大处理能力
            queue.post(command, priority);
            return true;
        }
        
        log.fatal("线程池[%s]繁忙超限，事件对队已满，无法接收处理", threadName);
        return false;
    }
    
    /*********************************************************************************************/
    //线程操作者调用，接受一个事件&完成事件设置数目
    /*********************************************************************************************/
    
    Runnable accept()
    {
        return queue.accept(5 * 1000);
    }
    
    void addCompletedCount()
    {
        completedCount.incrementAndGet();
    }
    
    /*********************************************************************************************/
    //线程池中对列信息&操作
    /*********************************************************************************************/
    
    public void clear()
    {
        queue.clear();
    }
    
    public boolean contains(Runnable command)
    {
        return queue.contains(command);
    }
    
    public int size()
    {
        return queue.size();
    }
    
    public boolean isEmpty()
    {
        return queue.isEmpty();
    }
    
    public Queue<Runnable> getQueue()
    {
        return queue;
    }
    
    public long getCompletedCount()
    {
        return completedCount.get();
    }
    
    /*********************************************************************************************/
    //获取线程池运行时参数
    /*********************************************************************************************/
    
    public List<Threader> getThreaderList()
    {
        return Collections.unmodifiableList(theaderList);
    }
    
    public int getThreaderSize()
    {
        return theaderList.size();
    }

    public int getThreaderActiveSize()
    {
        int count = 0;
        for (Threader threader : theaderList)
        {
            if (threader.isActive())
                count++;
        }
        return count;
    }
    
    /*********************************************************************************************/
    //获取线程池配置参数
    /*********************************************************************************************/
    
    public int getPoMinSize()
    {
        return poMinSize;
    }

    public int getPoMaxSize()
    {
        return poMaxSize;
    }
    
    public int getPoIdleTime()
    {
        return poIdleTimeMs / 1000;
    }
    
    public int getPoQueueSize()
    {
        return poQueueSize;
    }
}
