/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[欢迎加盟知启蒙，一起邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_project.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.project;

import java.util.List;

import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.manager.ZmrSessionUser;
import org.zhiqim.manager.dao.ZmrOperatorDao;
import org.zhiqim.manager.dbo.ZmrOperator;
import org.zhiqim.orm.ORM;
import org.zhiqim.orm.ZTable;
import org.zhiqim.orm.ZView;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.project.dbo.ZpmMember;
import org.zhiqim.project.dbo.ZpmMemberProject;
import org.zhiqim.project.dbo.ZpmProject;

@AnAlias("ZpmProjectDao")
public class ZpmProjectDao implements ZpmConstants
{
    
    /**
     * 获取操作员所有可见项目列表
     * 
     * @param operatorCode  操作员编码
     * @return              操作员可见项目列表
     * @throws Exception
     */
    public static List<? extends ZpmProject> getProjectList(HttpRequest request, String operatorCode) throws Exception
    {
        ZmrOperator operator = ORM.get(ZTable.class, request).item(ZmrOperator.class, operatorCode);
        
        if (operator.getOperatorType() <= 1)
        {//超级管理员或管理员，查看所有工程列表
            return ORM.get(ZTable.class, request).list(ZpmProject.class, new Selector("projectStatus", 0).addOrderbyAsc("projectSeq"));
        }
        else
        {//操作员查看所属的工程列表
            return ORM.get(ZView.class, request).list(ZpmMemberProject.class, new Selector("operatorCode", operatorCode).addMust("projectStatus", 0).addOrderbyAsc("projectSeq"));
        }
    }
    
    /**
     * 选择当前工程
     * 
     * @param sessionUser   用户会话对象
     * @param projectId     项目编号
     * @throws Exception    异常
     */
    public static void setProjectId(HttpRequest request, ZmrSessionUser sessionUser, long projectId) throws Exception
    {
        ZmrOperatorDao.addOrUpdateOperatorParam(request, sessionUser.getOperatorCode(), FPM_PROJECT_ID_KEY, projectId);
        ZmrOperator operator = ORM.get(ZTable.class, request).item(ZmrOperator.class, sessionUser.getOperatorCode());
        sessionUser.setOperator(operator);
        
        setProjectMember(request,sessionUser, projectId);
    }
    
    /**
     * 从用户会话中读取默认项目编号
     * 
     * @param request       请求对象
     * @return              项目编号
     * @throws Exception    异常
     */
    public static long getProjectId(HttpRequest request) throws Exception
    {
        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        return sessionUser == null?-1:getProjectId(sessionUser);
    }
    
    /**
     * 从用户会话中读取默认项目编号
     * 
     * @param sessionUser   用户会话对象
     * @return              项目编号
     * @throws Exception    异常
     */
    public static long getProjectId(ZmrSessionUser sessionUser) throws Exception
    {
        return sessionUser.getOperatorParamLong(FPM_PROJECT_ID_KEY);
    }
    
    /**
     * 设置默认项目成员到会话中
     * 
     * @param sessionUser   用户会话对象
     * @param projectId     项目编号
     * @param operatorCode  操作员编码
     * @throws Exception    异常
     */
    public static void setProjectMember(HttpRequest request, ZmrSessionUser sessionUser, long projectId) throws Exception
    {
        Selector selector = new Selector("projectId", projectId);
        selector.addMust("operatorCode", sessionUser.getOperatorCode());
        
        ZpmMember member = ORM.get(ZTable.class, request).item(ZpmMember.class, selector);
        sessionUser.setValue(FPM_MEMBER_KEY, member);
    }
    
    /**
     * 删除用户默认项目
     * 
     * @param sessionUser   用户会话对象
     * @param projectId     项目编号
     * @throws Exception    异常
     */
    public static void removeProjectId(HttpRequest request, ZmrSessionUser sessionUser, long projectId) throws Exception
    {
        ZmrOperatorDao.deleteOperatorParam(request, sessionUser.getOperatorCode(), FPM_PROJECT_ID_KEY);
        ZmrOperator operator = ORM.get(ZTable.class, request).item(ZmrOperator.class, sessionUser.getOperatorCode());
        sessionUser.setOperator(operator);
        sessionUser.removeValue(FPM_MEMBER_KEY);
    }
    
    /**
     * 判断是否切换了默认项目
     * 
     * @param request       请求对象
     * @param projectId     传入的项目编号
     * @return              =false表示切换了
     * @throws Exception    异常
     */
    public static boolean chkProjectId(HttpRequest request, long projectId) throws Exception
    {
        if (projectId != getProjectId(request))
        {
            if (request.getParameterBoolean(_PARAM_DIALOG_FRAME_))
                request.returnCloseDialog("您切换了项目或登录超时，请重新选择");
            else if (request.isXMLHttpRequest())
                request.setResponseError("您切换了项目或登录超时，请重新选择");
            else
                request.returnHistory("您切换了项目或登录超时，请重新选择");
            return false;
        }
        
        return true;
    }
    
    /**
     * 激活操作员默认项目
     * 
     * @param sessionUser   用户会话对象
     * @throws Exception    异常
     */
    public static void doActiveProject(HttpRequest request, ZmrSessionUser sessionUser) throws Exception
    {
        doActiveProject(request, getProjectList(request, sessionUser.getOperatorCode()), sessionUser);
    }
    
    /**
     * 激活操作员默认项目
     * 
     * @param sessionUser   用户会话对象
     * @param projectList   项目列表
     * @throws Exception    异常
     */
    public static void doActiveProject(HttpRequest request, List<? extends ZpmProject> projectList, ZmrSessionUser sessionUser) throws Exception
    {
        long zpmProjectId = sessionUser.getOperatorParamLong(FPM_PROJECT_ID_KEY);
        
        if (projectList.isEmpty())
        {//如果操作员没有一个项目，以前的默认的要取消 -->
            if (zpmProjectId != -1){
                removeProjectId(request, sessionUser, zpmProjectId);
            }
        }
        else
        {//如果操作员有项目
            boolean exist = false;
            if (zpmProjectId != -1)
            {
                for (ZpmProject proj : projectList)
                {
                    if (proj.getProjectId() != zpmProjectId)
                        continue;
                    
                    setProjectMember(request,sessionUser, zpmProjectId);
                    exist = true;
                    break;
                }
            }
            
            if (!exist)
            {//不存在或未选中，默认选中第一个
                setProjectId(request, sessionUser, projectList.get(0).getProjectId());
            }
        }
    }
}
