/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[欢迎加盟知启蒙，一起邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_manager.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.manager.dao;

import java.util.ArrayList;
import java.util.List;

import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.util.Arrays;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.manager.ZmrBootstrap;
import org.zhiqim.manager.ZmrSessionUser;
import org.zhiqim.manager.dbo.ZmrMenu;
import org.zhiqim.manager.dbo.ZmrOrg;
import org.zhiqim.manager.dbo.ZmrOrgRule;
import org.zhiqim.orm.dbo.Selector;

@AnAlias("ZmrOrgDao")
public class ZmrOrgDao
{
    /**
     * 获取组织列表
     * 
     * @param request       请求对象
     * @return              组织列表
     * @throws Exception    可能的异常
     */
    public static List<ZmrOrg> list(HttpRequest request) throws Exception
    {
        return ZmrBootstrap.tableOpr(request).list(ZmrOrg.class, new Selector().addOrderbyAsc("orgLevel,orgSeq,orgId"));
    }
    
    /**
     * 获取可切换的组织列表
     * 
     * @param request       请求对象
     * @return              组织列表
     * @throws Exception    可能的异常
     */
    public static List<ZmrOrg> getOrgList(HttpRequest request, String operatorCode) throws Exception
    {
        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        long orgId = sessionUser.getOperatorOrgId();
        ZmrOrg org = ZmrBootstrap.tableOpr(request).item(ZmrOrg.class, new Selector("orgId",orgId));
        List<ZmrOrg> list = new ArrayList<ZmrOrg>();
        if(org.getOrgLevel() == 0)
            list = list(request);
        
        if(org.getOrgLevel() == 1)
            list.add(org);
        
        return list;
    }
    
    /**
     * 获取组织列表，根据级别
     * 
     * @param request       请求对象
     * @param orgLevel      组织级别
     * @return              组织列表
     * @throws Exception    可能的异常
     */
    public static List<ZmrOrg> levelList(HttpRequest request, int orgLevel) throws Exception
    {
        return ZmrBootstrap.tableOpr(request).list(ZmrOrg.class, new Selector().addMust("orgLevel", orgLevel).addOrderbyAsc("orgSeq,orgId"));
    }
    
    /**
     * 获取根组织编号
     * 
     * @param request       请求对象
     * @return              根组织编号
     * @throws Exception    可能的异常
     */
    public static long getOrgRootId(HttpRequest request) throws Exception
    {
        return ZmrBootstrap.tableOpr(request).item(ZmrOrg.class, new Selector("orgLevel", 0)).getOrgId();
    }
    
    /**
     * 获取组织名称
     * 
     * @param request       请求对象
     * @param orgId         组织编号
     * @return              组织名称
     * @throws Exception    可能的异常
     */
    public static String getOrgName(HttpRequest request, long orgId) throws Exception
    {
        ZmrOrg org = ZmrBootstrap.tableOpr(request).item(ZmrOrg.class, orgId);
        return Validates.isEmpty(org) ? "" : org.getOrgName();
    }
    
    /**
     * 是否是组织负责人
     * 
     * @param request       请求对象
     * @param operatorCode  操作员编号
     * @return              是否组织负责人
     * @throws Exception    可能的异常
     */
    public static boolean isOrgManager(HttpRequest request, String operatorCode) throws Exception
    {
        return !ZmrBootstrap.tableOpr(request).list(ZmrOrg.class, new Selector("orgId", "orgManager", operatorCode)).isEmpty();
    }
    
    /**
     * 设置顶级组织权限和菜单权限保持一致
     * 
     * @param request       请求对象
     * @param orgId         组织编号
     * @throws Exception    可能的异常
     */
    public static void setTopOrgRule(HttpRequest request, long orgId) throws Exception
    {
        List<ZmrMenu> menuList = ZmrBootstrap.table(request).list(ZmrMenu.class, new Selector());
        List<ZmrOrgRule> orgRuleList = ZmrBootstrap.table(request).list(ZmrOrgRule.class, new Selector("orgId", orgId));
        
        String allMenu ="";
        for (ZmrMenu menu : menuList)
        {
            allMenu += menu.getMenuCode()+",";
        }
        String orgMenu = "";
        for (ZmrOrgRule rule : orgRuleList)
        {
            orgMenu += rule.getMenuCode()+",";
        }
        
        if(!allMenu.equals(orgMenu))
        {
            ZmrBootstrap.table(request).delete(ZmrOrgRule.class, new Selector("orgId", orgId));
            List<ZmrOrgRule> newRuleList = new ArrayList<ZmrOrgRule>();
            for (ZmrMenu menu : menuList)
            {
                ZmrOrgRule rule = new ZmrOrgRule();
                rule.setOrgId(orgId);
                rule.setMenuCode(menu.getMenuCode());
                newRuleList.add(rule);
            }
            ZmrBootstrap.table(request).insertBatch(newRuleList);
            
            //刷新所有在线用户会话
            ZmrOperatorDao.refresh(request);
        }
    }
    
    /**
     * 获取部门、角色、操作员所属组织的权限
     * 
     * @param request       请求对象
     * @param orgId         组织编号
     * @return              组织权限
     * @throws Exception    可能的异常
     */
    public static List<ZmrMenu> getMenuList(HttpRequest request, long orgId) throws Exception
    {
        ZmrOrg org = ZmrBootstrap.tableOpr(request).item(ZmrOrg.class,orgId);
        if(org != null && org.getOrgLevel() == 0)
            setTopOrgRule(request, orgId);
        
        List<ZmrOrgRule> orgRuleList = ZmrBootstrap.table(request).list(ZmrOrgRule.class, new Selector("orgId",orgId));
        String codeS = "";
        for (ZmrOrgRule rule : orgRuleList)
        {
            if (Validates.isEmptyBlank(codeS))
                codeS = rule.getMenuCode();
            else
                codeS = codeS +","+ rule.getMenuCode();
        }
        
        List<ZmrMenu> menuList = ZmrBootstrap.table(request).list(ZmrMenu.class, new Selector().addMustIn("menuCode", Arrays.toStringArray(codeS)));
        
        return menuList;
    }
    
    /**
     * 指定组织编号获取组织
     * 
     * @param request       请求对象
     * @param orgId         组织编号
     * @return              组织对象
     * @throws Exception    可能的异常
     */
    public static ZmrOrg getOrg(HttpRequest request, long orgId) throws Exception
    {
        return ZmrBootstrap.tableOpr(request).item(ZmrOrg.class,orgId);
    }
    
}
