/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[欢迎加盟知启蒙，一起邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_manager.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.manager.action;

import java.util.List;
import java.util.Random;

import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.httpd.context.extend.StdSwitchAction;
import org.zhiqim.httpd.validate.ones.IsNotEmpty;
import org.zhiqim.httpd.validate.onex.IsEmail;
import org.zhiqim.httpd.validate.onex.IsIP;
import org.zhiqim.httpd.validate.onex.IsMobile11;
import org.zhiqim.httpd.validate.onex.IsUserCode;
import org.zhiqim.httpd.validate.onex.IsUserPass;
import org.zhiqim.kernel.annotation.AnTransaction;
import org.zhiqim.kernel.paging.PageResult;
import org.zhiqim.kernel.util.Arrays;
import org.zhiqim.kernel.util.DateTimes;
import org.zhiqim.kernel.util.Randoms;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.manager.ZmrBootstrap;
import org.zhiqim.manager.ZmrConstants;
import org.zhiqim.manager.ZmrPassworder;
import org.zhiqim.manager.ZmrSessionUser;
import org.zhiqim.manager.dao.ZmrDeptDao;
import org.zhiqim.manager.dbo.ZmrAvatar;
import org.zhiqim.manager.dbo.ZmrDept;
import org.zhiqim.manager.dbo.ZmrOperator;
import org.zhiqim.manager.dbo.ZmrOperatorLog;
import org.zhiqim.manager.dbo.ZmrOperatorRule;
import org.zhiqim.manager.dbo.ZmrRole;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.orm.dbo.Updater;

/**
 * 操作员/管理员管理
 * 
 * @version v1.0.0 @author zouzhigang 2015-5-28 新建与整理
 */
public class OperatorAction extends StdSwitchAction implements ZmrConstants
{
    @Override
    protected void validateId(HttpRequest request)
    {
        request.addValidate(new IsUserCode("operatorCode", "操作员编码由2-16位字母数字汉字开头，字母汉字数字特殊符号组成"));
    }

    @Override
    protected void validateForm(HttpRequest request)
    {
        request.addValidate(new IsUserCode("operatorCode", "操作员编码由2-16位字母数字汉字开头，字母汉字数字特殊符号组成"));
        request.addValidate(new IsNotEmpty("operatorName", "操作员名称不能为空"));
        request.addValidate(new IsNotEmpty("operatorStatus", "操作员状态不能为空"));
        request.addValidate(new IsIP("operatorIp", "请输入正确的IP地址", true));
        request.addValidate(new IsMobile11("operatorMobile", "请输入正确的手机号", true));
        request.addValidate(new IsEmail("operatorEmail", "请输入正确的邮箱", true));
    }

    @Override
    protected void list(HttpRequest request) throws Exception
    {
        int page = request.getParameterInt("page");
        int pageSize = request.getContextAttributeInt(ZMR_PAGE_SIZE, 20);
        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        long orgId = sessionUser.getSelectedOrgId();

        List<ZmrDept> deptList = ZmrDeptDao.getOrgDeptList(request, orgId);
        List<ZmrRole> roleList = ZmrBootstrap.table(request).list(ZmrRole.class);

        Selector selector = new Selector();
        if (sessionUser.isAdmin())
        {// 管理员可以看到管理员和操作员
            selector.addMustThenG("operatorType", 0);
        }
        else
        {// 操作员只能查操作员
            selector.addMustThenG("operatorType", 1);
        }

        String operatorKey = request.getParameter("operatorKey");
        if (Validates.isNotEmpty(operatorKey))
        {// 条件
            selector.addOr(new Selector().addMaybeLike("operatorCode", operatorKey).addMaybeLike("operatorName", operatorKey));
        }
        selector.addMaybe("operatorStatus", request.getParameterInt("operatorStatus"));
        selector.addOrderbyAsc("operatorType");
        selector.addOrderbyAsc("operatorStatus");
        selector.addOrderbyAsc("operatorCreated");
        selector.addMust("orgId", orgId);

        PageResult<ZmrOperator> result = ZmrBootstrap.tableOpr(request).page(ZmrOperator.class, page, pageSize, selector);
        result.addConditionMap(request.getParameterMap());

        for (ZmrOperator operator : result.list())
        {
            toDeptName(deptList, operator);
            toRoleName(roleList, operator);
        }

        request.setAttribute("result", result);
    }

    @Override
    protected void add(HttpRequest request) throws Exception
    {
        request.addValidate(new IsUserPass("operatorPass", "操作员密码不合法，要求6-16位（大小写字母数字和特殊字符必须四选三）"));
    }

    @Override
    protected void insert(HttpRequest request) throws Exception
    {
        String dateTime = DateTimes.getDateTimeString();
        long orgId = request.getSessionUser(ZmrSessionUser.class).getSelectedOrgId();
        String operatorCode = request.getParameter("operatorCode");
        String operatorPass = request.getParameter("operatorPass");
        int operatorStatus = request.getParameterInt("operatorStatus");
        int operatorType = request.getParameterInt("operatorType", 2);// 默认为操作员
        String operatorName = request.getParameter("operatorName");
        String operatorIp = request.getParameter("operatorIp");
        String operatorMobile = request.getParameter("operatorMobile");
        String operatorEmail = request.getParameter("operatorEmail");
        String operatorWangwang = request.getParameter("operatorWangwang");

        if (ZmrBootstrap.tableOpr(request).count(ZmrOperator.class, operatorCode) > 0)
        {
            request.returnHistory("该操作员编码已经存在，请重新输入");
            return;
        }

        if (!Validates.isUserPass(operatorPass))
        {
            request.returnHistory("请输入6-16位操作员密码（大小写字母数字和特殊字符必须四选三）");
            return;
        }

        long operatorAvatar = 0;
        PageResult<ZmrAvatar> result = ZmrBootstrap.table(request).page(ZmrAvatar.class, 1, 10, new Selector("avatarId", "avatarType", 0));
        if (result.size() > 0)
        {// 随机取一个系统头像
            int ind = new Random().nextInt(result.size());
            operatorAvatar = result.list().get(ind).getAvatarId();
        }

        ZmrPassworder passworder = request.getContextAttribute(ZmrBootstrap.class).getPassworder();
        String operatorPassSalt = Randoms.lettersDigitsSecure(64);
        operatorPass = passworder.encode(operatorCode, operatorPass, operatorPassSalt);

        ZmrOperator operator = new ZmrOperator();
        operator.setOrgId(orgId);
        operator.setOperatorCode(operatorCode);
        operator.setOperatorPass(operatorPass);
        operator.setOperatorPassSalt(operatorPassSalt);
        operator.setOperatorStatus(operatorStatus);
        operator.setOperatorType(operatorType);
        operator.setOperatorName(operatorName);
        operator.setOperatorIp(operatorIp);
        operator.setOperatorAvatar(operatorAvatar);
        operator.setOperatorMobile(operatorMobile);
        operator.setOperatorEmail(operatorEmail);
        operator.setOperatorCreated(dateTime);
        operator.setOperatorModified(dateTime);
        operator.setOperatorWangwang(operatorWangwang);
        ZmrBootstrap.table(request).insert(operator);
    }

    @Override
    protected void modify(HttpRequest request) throws Exception
    {
        request.addValidate(new IsUserPass("operatorPass", "操作员密码不合法，要求6-16位（大小写字母数字和特殊字符必须四选三）", true));
        String operatorCode = request.getParameter("operatorCode");

        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
        {
            request.returnHistory("该操作员不存在，请重新选择");
            return;
        }

        request.setAttribute("operator", operator);
    }

    @Override
    protected void update(HttpRequest request) throws Exception
    {
        String dateTime = DateTimes.getDateTimeString();
        String operatorCode = request.getParameter("operatorCode");
        String operatorName = request.getParameter("operatorName");
        String operatorPass = request.getParameter("operatorPass");
        String operatorIp = request.getParameter("operatorIp");
        String operatorMobile = request.getParameter("operatorMobile");
        String operatorEmail = request.getParameter("operatorEmail");
        int operatorStatus = request.getParameterInt("operatorStatus");
        int operatorType = request.getParameterInt("operatorType");
        String operatorWangwang = request.getParameter("operatorWangwang");

        Updater updater = new Updater();
        updater.addMaybe("operatorCode", operatorCode);
        updater.addField("operatorName", operatorName);
        if (Validates.isNotEmpty(operatorPass))
        {
            if (!Validates.isUserPass(operatorPass))
            {
                request.returnHistory("请输入6-16位操作员密码（大小写字母数字和特殊字符必须四选三）");
                return;
            }

            ZmrPassworder passworder = request.getContextAttribute(ZmrBootstrap.class).getPassworder();
            String operatorPassSalt = Randoms.lettersDigitsSecure(64);

            operatorPass = passworder.encode(operatorCode, operatorPass, operatorPassSalt);
            updater.addField("operatorPass", operatorPass);
            updater.addField("operatorPassSalt", operatorPassSalt);
        }

        updater.addField("operatorStatus", operatorStatus);
        updater.addField("operatorIp", operatorIp);
        updater.addField("operatorMobile", operatorMobile);
        updater.addField("operatorEmail", operatorEmail);
        updater.addField("operatorModified", dateTime);
        updater.addField("operatorType", operatorType);
        updater.addField("operatorWangwang", operatorWangwang);

        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
    }

    @Override
    @AnTransaction
    protected void delete(HttpRequest request) throws Exception
    {
        String operatorCode = request.getParameter("operatorCode");
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null)
        {
            request.returnHistory("该操作员不存在");
            return;
        }

        Selector selector = new Selector("operatorCode", operatorCode);
        if (ZmrBootstrap.table(request).count(ZmrOperatorLog.class, selector) > 0)
        {
            request.returnHistory("该操作员已有操作日志，不允许删除，只能停用");
            return;
        }

        ZmrAvatar avatar = ZmrBootstrap.table(request).item(ZmrAvatar.class, operator.getOperatorAvatar());
        if (avatar != null && avatar.getAvatarType() == 1)
        {// 删除操作员头像
            ZmrBootstrap.table(request).delete(ZmrAvatar.class, operator.getOperatorAvatar());
        }

        // 删除操作员&权限
        ZmrBootstrap.tableOpr(request).delete(ZmrOperator.class, operatorCode);
        ZmrBootstrap.table(request).delete(ZmrOperatorRule.class, selector);
    }

    /********************************************************************************************/
    // 私有方法
    /********************************************************************************************/

    private void toDeptName(List<ZmrDept> deptList, ZmrOperator operator)
    {
        operator.setOperatorCreated("");

        long[] deptIds = Arrays.toLongArray(operator.getOperatorDept());
        for (long deptId : deptIds)
        {
            for (ZmrDept dept : deptList)
            {
                if (dept.getDeptId() == deptId)
                {
                    if (Validates.isEmpty(operator.getOperatorCreated()))
                    {
                        operator.setOperatorCreated(dept.getDeptName());
                    }
                    else
                    {
                        operator.setOperatorCreated(operator.getOperatorCreated() + "," + dept.getDeptName());
                    }
                    break;
                }
            }
        }
    }

    private void toRoleName(List<ZmrRole> roleList, ZmrOperator operator)
    {
        operator.setOperatorPass("");

        long[] roleIds = Arrays.toLongArray(operator.getOperatorRole());
        for (long roleId : roleIds)
        {
            for (ZmrRole role : roleList)
            {
                if (role.getRoleId() == roleId)
                {
                    if (Validates.isEmpty(operator.getOperatorPass()))
                    {
                        operator.setOperatorPass(role.getRoleName());
                    }
                    else
                    {
                        operator.setOperatorPass(operator.getOperatorPass() + "," + role.getRoleName());
                    }
                    break;
                }
            }
        }
    }
}
