/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util;

import java.io.FileNotFoundException;
import java.util.Collection;

import org.zhiqim.kernel.ZhiqimI18n;
import org.zhiqim.kernel.annotation.AnAlias;

/**
 * 断言相关工具类
 *
 * @version v1.1.2 @author zouzhigang 2015-10-4 新建与整理
 */
@AnAlias("Asserts")
public class Asserts
{
    /*****************************************************************************/
    //断言， 三目运算判断exception!=null时抛出IllegalArgumentException
    /*****************************************************************************/
    
    /**
     * 断言，三目运算判断，为exception=null时表示正常，否则抛出非法的参数异常
     * 
     * @param exception     异常信息
     */
    public static void as(String exception)
    {
        if (exception == null)
            return;
        
        throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
    }
    
    /**
     * 断言，三目运算判断，为exception=null时表示正常，否则抛出非法的参数异常
     * 
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void as(String exception, Object...arguments)
    {
        if (exception == null)
            return;
        
        exception = Strings.formatTwo(exception, arguments);
        throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
    }
    
    /**
     * 断言，传入要断言的判断，为false时抛出非法的状态异常
     * 
     * @param param         断言为false时抛出非法的参数异常
     * @param exception     异常信息
     */
    public static void asState(String exception)
    {
        if (exception == null)
            return;
        
        throw new IllegalStateException(Threads.getInvokerAbbr() + exception);
    }
    
    /**
     * 断言，传入要断言的判断，为false时抛出非法的状态异常
     * 
     * @param param         断言为false时抛出非法的参数异常
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void asState(String exception, Object... arguments)
    {
        if (exception == null)
            return;

        exception = Strings.formatTwo(exception, arguments);
        throw new IllegalStateException(Threads.getInvokerAbbr() + exception);
    }
    
    /**
     * 断言对象不为null，如果为null则抛出非法的参数异常
     * 
     * @param param         参数
     */
    public static void notNull(Object param)
    {
        if (param == null){
            throw new NullPointerException(Threads.getInvokerAbbr() + ZhiqimI18n.parameterCannotBeNull);
        }
    }
    
    /**
     * 断言对象不为null，如果为null则抛出非法的参数异常
     * 
     * @param param         参数
     * @param paramName     字段名
     */
    public static void notNull(Object param, String paramName)
    {
        if (param == null){
            String exception = Strings.formatTwo(ZhiqimI18n.parameterCannotBeNull1, paramName);
            throw new NullPointerException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         参数
     */
    public static void notEmpty(String param)
    {
        if (Validates.isEmpty(param)){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + ZhiqimI18n.parameterCannotBeNullOrEmpty);
        }
    }
    
    /**
     * 断言字符串不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         参数
     * @param paramName     字段名
     */
    public static void notEmpty(String param, String paramName)
    {
        if (Validates.isEmpty(param)){
            String exception = Strings.formatTwo(ZhiqimI18n.parameterCannotBeNullOrEmpty1, paramName);
            throw new NullPointerException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         参数
     */
    public static void notEmpty(Object param)
    {
        if (Validates.isEmpty(param)){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + ZhiqimI18n.parameterCannotBeNullOrEmpty);
        }
    }
    
    /**
     * 断言字符串不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         参数
     * @param paramName     字段名
     */
    public static void notEmpty(Object param, String paramName)
    {
        if (Validates.isEmpty(param)){
            String exception = Strings.formatTwo(ZhiqimI18n.parameterCannotBeNullOrEmpty1, paramName);
            throw new NullPointerException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         参数
     */
    public static void notEmptyBlank(String param)
    {
        if (Validates.isEmptyBlank(param)){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + ZhiqimI18n.parameterCannotBeNullOrEmptyBlank);
        }
    }
    
    /**
     * 断言字符串不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         参数
     * @param paramName     字段名
     */
    public static void notEmptyBlank(String param, String paramName)
    {
        if (Validates.isEmptyBlank(param)){
            String exception = Strings.formatTwo(ZhiqimI18n.parameterCannotBeNullOrEmptyBlank1, paramName);
            throw new NullPointerException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串是整型，支持前缀+-，如果不是整型则抛出非法的参数异常
     * 
     * @param param         参数
     */
    public static void asInteger(String param)
    {
        if (!Validates.isInteger(param)){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + ZhiqimI18n.parameterMustBeInteger);
        }
    }
    
    /**
     * 断言字符串是整型，支持前缀+-，如果不是整型则抛出非法的参数异常
     * 
     * @param param         参数
     * @param paramName     字段名
     */
    public static void asInteger(String param, String paramName)
    {
        if (!Validates.isInteger(param)){
            String exception = Strings.formatTwo(ZhiqimI18n.parameterMustBeInteger, paramName);
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /*****************************************************************************/
    //断言， 判断结果=false时抛出IllegalArgumentException
    /*****************************************************************************/
    
    /**
     * 断言，传入要断言的判断，为false时抛出非法的参数异常
     * 
     * @param param         断言为false时抛出非法的参数异常
     * @param exception     异常信息
     */
    public static void asserts(boolean param, String exception)
    {
        if (!param){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言，传入要断言的判断，为false时抛出非法的参数异常
     * 
     * @param param         断言为false时抛出非法的参数异常
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void asserts(boolean param, String exception, Object... arguments)
    {
        if (!param){
            exception = Strings.formatTwo(exception, arguments);
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言，传入要断言的判断，为false时抛出非法的状态异常
     * 
     * @param param         断言为false时抛出非法的参数异常
     * @param exception     异常信息
     */
    public static void assertState(boolean param, String exception)
    {
        if (!param){
            throw new IllegalStateException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言，传入要断言的判断，为false时抛出非法的状态异常
     * 
     * @param param         断言为false时抛出非法的参数异常
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void assertState(boolean param, String exception, Object... arguments)
    {
        if (!param){
            exception = Strings.formatTwo(exception, arguments);
            throw new IllegalStateException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言对象不为null，如果为null则抛出非法的参数异常
     * 
     * @param param         参数
     * @param exception     异常信息
     */
    public static void assertNotNull(Object param, String exception)
    {
        if (param == null){
            throw new NullPointerException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言对象不为null，如果为null则抛出非法的参数异常
     * 
     * @param param         参数
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void assertNotNull(Object param, String exception, Object... arguments)
    {
        if (param == null){
            exception = Strings.formatTwo(exception, arguments);
            throw new NullPointerException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         参数
     * @param exception     异常信息
     */
    public static void assertNotEmpty(String param, String exception)
    {
        if (Validates.isEmpty(param)){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         参数
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void assertNotEmpty(String param, String exception, Object... arguments)
    {
        if (Validates.isEmpty(param)){
            exception = Strings.formatTwo(exception, arguments);
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言集合不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         集合参数
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void assertNotEmpty(Collection<?> param, String exception, Object... arguments)
    {
        if (Validates.isEmpty(param)){
            exception = Strings.formatTwo(exception, arguments);
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言数组不为空，如果为空则抛出非法的参数异常
     * 
     * @param param         数组参数
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void assertNotEmpty(Object[] param, String exception, Object... arguments)
    {
        if (Validates.isEmpty(param)){
            exception = Strings.formatTwo(exception, arguments);
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串不为空白，如果为空白则抛出非法的参数异常
     * 
     * @param param         参数
     * @param exception     异常信息
     */
    public static void assertNotEmptyBlank(String param, String exception)
    {
        if (Validates.isEmptyBlank(param)){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串不为空白，如果为空白则抛出非法的参数异常
     * 
     * @param param         参数
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void assertNotEmptyBlank(String param, String exception, Object... arguments)
    {
        if (Validates.isEmptyBlank(param)){
            exception = Strings.formatTwo(exception, arguments);
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串是整型，支持前缀+-，如果不是整型则抛出非法的参数异常
     * 
     * @param param         参数
     * @param exception     异常信息
     */
    public static void assertInteger(String param, String exception)
    {
        if (!Validates.isInteger(param)){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串是整型，支持前缀+-，如果不是整型则抛出非法的参数异常
     * 
     * @param param         参数
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void assertInteger(String param, String exception, Object... arguments)
    {
        if (!Validates.isInteger(param)){
            exception = Strings.formatTwo(exception, arguments);
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串前缀匹配
     * 
     * @param str           字符串
     * @param prefix        前缀
     * @param exception     异常信息
     */
    public static void assertStartsWith(String str, String prefix, String exception)
    {
        if (!Stringx.startsWith(str, prefix)){
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /**
     * 断言字符串前缀匹配
     * 
     * @param str           字符串
     * @param prefix        前缀
     * @param exception     异常信息
     * @param arguments     异常信息中的格式化参数表
     */
    public static void assertStartsWith(String str, String prefix, String exception, Object... arguments)
    {
        if (!Stringx.startsWith(str, prefix)){
            exception = Strings.formatTwo(exception, arguments);
            throw new IllegalArgumentException(Threads.getInvokerAbbr() + exception);
        }
    }
    
    /*****************************************************************************/
    //返回断言异常，IllegalArgumentException
    /*****************************************************************************/
    
    /**
     * 断言，直接抛出异常，信息会在原信息上加上类和方法名
     * 
     * @param message       异常信息
     * @return              IllegalArgumentException
     */
    public static IllegalArgumentException exception(String message)
    {
        return new IllegalArgumentException(Threads.getInvokerAbbr() + message);
    }
    
    
    /**
     * 断言，直接抛出异常，信息会在原信息上加上类和方法名
     * 
     * @param exception     异常信息
     * @return              IllegalArgumentException
     */
    public static IllegalArgumentException exception(Exception e)
    {
        return new IllegalArgumentException(Threads.getInvokerAbbr() + e.getMessage(), e);
    }
    
    /**
     * 断言，直接抛出异常，信息会在原信息上加上类和方法名
     * 
     * @param message       异常信息
     * @param arguments     异常信息中的格式化参数表
     * @return              IllegalArgumentException
     */
    public static IllegalArgumentException exception(String message, Object... arguments)
    {
        message = Strings.formatTwo(message, arguments);
        return new IllegalArgumentException(Threads.getInvokerAbbr() + message);
    }

    
    /**
     * 断言，直接抛出异常，信息会在原信息上加上类和方法名
     * 
     * @param message       异常信息
     * @param e             异常类
     * @return              IllegalArgumentException
     */
    public static IllegalArgumentException exception(String message, Exception e)
    {
        return new IllegalArgumentException(Threads.getInvokerAbbr() + message, e);
    }
    
    /**
     * 断言，直接抛出异常，信息会在原信息上加上类和方法名
     * 
     * @param message       异常信息
     * @param e             异常类
     * @param arguments     异常信息中的格式化参数表
     * @return              IllegalArgumentException
     */
    public static IllegalArgumentException exception(String message, Exception e, Object... arguments)
    {
        message = Strings.formatTwo(message, arguments);
        return new IllegalArgumentException(Threads.getInvokerAbbr() + message, e);
    }
    
    /**
     * 断言，直接抛出文件未找到异常
     * 
     * @param message       异常信息
     * @return              FileNotFoundException
     */
    public static FileNotFoundException notFound(String message)
    {
        return new FileNotFoundException(Threads.getInvokerAbbr() + message);
    }
    
    /**
     * 断言，直接抛出文件未找到异常
     * 
     * @param message       异常信息
     * @param arguments     异常信息中的格式化参数表
     * @return              FileNotFoundException
     */
    public static FileNotFoundException notFound(String message, Object... arguments)
    {
        message = Strings.formatTwo(message, arguments);
        return new FileNotFoundException(Threads.getInvokerAbbr() + message);
    }
}
