/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;

import org.zhiqim.kernel.annotation.AnAlias;

/**
 * 数组相关工具类
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Arrays")
public class Arrays
{
    /**
     * 返回一个新的数组，包含从 start到尾部或首部(start<0时)的中的元素。
     * 
     * @param src   原数组
     * @param start 起始位置，负位表示从尾部算起,-1表示倒数第一个(包含)
     * @return      新的数组
     */
    public static <T> T[] slice(T[] src, int start)
    {
        Asserts.notNull(src, "原数组");
        
        int len = start<0?src.length+start+1:src.length-start;
        @SuppressWarnings("unchecked")
        T[] dest = (T[])Array.newInstance(src.getClass().getComponentType(), len);
        
        int index = 0;
        if (start < 0)
        {
            for (int i=src.length+start;i>=0;i--){
                dest[index++] = src[i];
            }
        }
        else
        {
            for (int i=start;i<src.length;i++){
                dest[index++] = src[i];
            }
        }
        
        return dest;
    }
    
    /**
     * 返回一个新的数组，包含从 start到 end(不包括该元素)的中的元素。设数组length=5：
     * 1.双正数：start=1,end=4，表示读取个数len=4-1=3，数据为数组的下标第1,2,3个。
     * 2.双负数：start=-1,end=-3，表示读取个数len=start-end=-1-(-3)=2个，数据为数组下标第4,3
     * 4.单负数，start=-1,end=2，先把end置为负数，end=2-5=-3，表示读取个数len=start-end=-1+3=2，数据为数组下标第4,3
     * 5.单正数，start=1,end=-2，先把end置为正数，end=-2+5=3，表示读取个数len=3-1=2，数据为数组下标第1,2
     * 
     * @param src  原数组
     * @param start 起始位置，负位表示从尾部算起,-1表示倒数第一个开始
     * @param end   结束位置，负数表示从尾部算起,-2表示倒数第二个结束
     * @return      新的数组
     */
    public static <T> T[] slice(T[] src, int start, int end)
    {
        Asserts.notNull(src);
        
        if (start < 0 && end >= 0)
        {//如果起始小于0表示从后面读取，则当end>0时要计算出<0的位置
            end = end - src.length;
        }
        
        if (start >= 0 && end < 0)
        {//如果从起始大于等于0表示从前面读取，当end<0时要计算出前面相对位置
            end = end + src.length;
        }
        
        int len = start<0?start-end:end-start;
        @SuppressWarnings("unchecked")
        T[] dest = (T[])Array.newInstance(src.getClass().getComponentType(), len);
            
        int index = 0;
        if (start < 0)
        {
            for (int i=src.length-start;i>end;i--){
                dest[index++] = src[i];
            }
        }
        else
        {
            for (int i=start;i<end;i++){
                dest[index++] = src[i];
            }
        }
        
        return dest;
    }
    
    /***
     * 获取对象在数组中的下标
     * 
     * @param objs 对象数组
     * @param obj  对象
     * @return 下标0到objs.length-1之间
     */
    public static int indexOf(Object[] objs, Object obj)
    {
        if (objs == null)
            return -1;
        
        if (obj == null)
        {
            for (int i=0;i<objs.length;i++)
            {
                if (objs[i] == null)
                    return i;
            }
        }
        else
        {
            for (int i=0;i<objs.length;i++)
            {
                if (obj.equals(objs[i]))
                    return i;
            }
        }
        
        return -1;
    }
    
    /*****************************************************/
    //对象或列表，转换成数组
    /*****************************************************/
    
    /**
     * 对一个未知的数组对象，查看数组长度
     * 
     * @param obj 未知的数组对象
     * @return 数组长度
     */
    public static int length(Object obj)
    {
        if (!Types.isArray(obj))
            return 0;
        
        if (Types.isIntArray(obj))
            return ((int[])obj).length;
        else if (Types.isLongArray(obj))
            return ((long[])obj).length;
        else if (Types.isBooleanArray(obj))
            return ((boolean[])obj).length;
        else if (Types.isByteArray(obj))
            return ((byte[])obj).length;
        else if (Types.isCharArray(obj))
            return ((char[])obj).length;
        else if (Types.isShortArray(obj))
            return ((short[])obj).length;
        else if (Types.isFloatArray(obj))
            return ((float[])obj).length;
        else if (Types.isDoubleArray(obj))
            return ((double[])obj).length;
        else
            return ((Object[])obj).length;
    }
    
    /**
     * 对一个未知的数组对象，查看数组是否为空
     * 
     * @param obj 未知的数组对象
     * @return 数组是否为空
     */
    public static boolean isEmpty(Object obj)
    {
        return length(obj) == 0;
    }
    
    /**
     * 对一个未知的数组对象，转化为一个已知的数组对象，其中对基本类型进行转化
     * 
     * @param obj 未知的数组对象
     * @return 已知的数组对象
     */
    public static Object[] toArray(Object obj)
    {
        Asserts.as(Types.isArray(obj)?null:"传入的参数不是数组类型");
        
        if (Types.isIntArray(obj))
        {
            int[] os = (int[])obj;
            Integer[] arr = new Integer[os.length];
            for (int i=0;i<os.length;i++)
                arr[i] = os[i];
            return arr;
        }
        else if (Types.isLongArray(obj))
        {
            long[] os = (long[])obj;
            Long[] arr = new Long[os.length];
            for (int i=0;i<os.length;i++)
                arr[i] = os[i];
            return arr;
        }
        else if (Types.isBooleanArray(obj))
        {
            boolean[] os = (boolean[])obj;
            Boolean[] arr = new Boolean[os.length];
            for (int i=0;i<os.length;i++)
                arr[i] = os[i];
            return arr;
        }
        else if (Types.isByteArray(obj))
        {
            byte[] os = (byte[])obj;
            Byte[] arr = new Byte[os.length];
            for (int i=0;i<os.length;i++)
                arr[i] = os[i];
            return arr;
        }
        else if (Types.isCharArray(obj))
        {
            char[] os = (char[])obj;
            Character[] arr = new Character[os.length];
            for (int i=0;i<os.length;i++)
                arr[i] = os[i];
            return arr;
        }
        else if (Types.isShortArray(obj))
        {
            short[] os = (short[])obj;
            Short[] arr = new Short[os.length];
            for (int i=0;i<os.length;i++)
                arr[i] = os[i];
            return arr;
        }
        else if (Types.isFloatArray(obj))
        {
            float[] os = (float[])obj;
            Float[] arr = new Float[os.length];
            for (int i=0;i<os.length;i++)
                arr[i] = os[i];
            return arr;
        }
        else if (Types.isDoubleArray(obj))
        {
            double[] os = (double[])obj;
            Double[] arr = new Double[os.length];
            for (int i=0;i<os.length;i++)
                arr[i] = os[i];
            return arr;
        }
        else
        {
            return (Object[])obj;
        }
    }
    
    /**
     * 列表转数组，如果list=null，则返回null
     * 
     * @param list  列表
     * @param clazz 类型
     * @return      指定类型的对象数组
     */
    @SuppressWarnings("unchecked")
    public static <T> T[] toArray(List<T> list, Class<T> clazz)
    {
        if (list == null)
            return null;
        
        //可能的基本类型转为对象类型，如int.class转为Integer.class
        clazz = (Class<T>)Types.toObjectClass(clazz);
        return list.toArray((T[])Array.newInstance(clazz, 0));
    }
    
    /**
     * 整型列表转为int[0]，而不是Integer[]，list=null返回int[0]
     * 
     * @param list  列表
     * @return      int[]
     */
    public static int[] toArrayInt(List<Integer> list)
    {
        if (list == null) 
            return new int[0];
        
        return Arrays.convert(list.toArray(new Integer[0]));
    }
    
    /**
     * 整型列表转为long[]数组，而不是Long[]，list=null返回long[0]
     * 
     * @param list  列表
     * @return      long[]
     */
    public static long[] toArrayLong(List<Long> list)
    {
        if (list == null) 
            return new long[0];
        
        return Arrays.convert(list.toArray(new Long[0]));
    }
    
    /**
     * 整型列表转为byte[]数组，而不是Byte[]，list=null返回byte[0]
     * 
     * @param list  列表
     * @return      long[]
     */
    public static byte[] toArrayByte(List<Byte> list)
    {
        if (list == null) 
            return new byte[0];
        
        return Arrays.convert(list.toArray(new Byte[0]));
    }
    
    /**
     * 整型列表转为short[]数组，而不是Short[]，list=null返回short[0]
     * 
     * @param list  列表
     * @return      short[]
     */
    public static short[] toArrayShort(List<Short> list)
    {
        if (list == null) 
            return new short[0];
        
        return Arrays.convert(list.toArray(new Short[0]));
    }
    
    /**
     * 整型列表转为boolean[]数组，而不是Boolean[]，list=null返回boolean[0]
     * 
     * @param list  列表
     * @return      boolean[]
     */
    public static boolean[] toArrayBoolean(List<Boolean> list)
    {
        if (list == null) 
            return new boolean[0];
        
        return Arrays.convert(list.toArray(new Boolean[0]));
    }
    
    /**
     * 整型列表转为char[]数组，而不是Character[]，list=null返回char[0]
     * 
     * @param list  列表
     * @return      char[]
     */
    public static char[] toArrayChar(List<Character> list)
    {
        if (list == null) 
            return new char[0];
        
        return Arrays.convert(list.toArray(new Character[0]));
    }
    
    /**
     * 整型列表转为float[]数组，而不是Float[]，list=null返回float[0]
     * 
     * @param list  列表
     * @return      float[]
     */
    public static float[] toArrayFloat(List<Float> list)
    {
        if (list == null) 
            return new float[0];
        
        return Arrays.convert(list.toArray(new Float[0]));
    }
    
    /**
     * 整型列表转为double[]数组，而不是Double[]，list=null返回double[0]
     * 
     * @param list  列表
     * @return      double[]
     */
    public static double[] toArrayDouble(List<Double> list)
    {
        if (list == null) 
            return new double[0];
        
        return Arrays.convert(list.toArray(new Double[0]));
    }
    
    /*****************************************************/
    //基本类型数组和基本对象类型数组相互转换
    /*****************************************************/
    
    /**
     * 布尔型对象相互转换，由Boolean[]转为boolean[]
     * 
     * @param values    Boolean[]
     * @return          boolean[]
     */
    public static boolean[] convert(Boolean[] values)
    {
        if (values == null)
            return null;
        
        boolean[] results = new boolean[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 布尔型对象相互转换，由boolean[]转为Boolean[]
     * 
     * @param values    boolean[]
     * @return          Boolean[]
     */
    public static Boolean[] convert(boolean[] values)
    {
        if (values == null)
            return null;
        
        Boolean[] results = new Boolean[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 字节型对象相互转换，由Byte[]转为byte[]
     * 
     * @param values    Byte[]
     * @return          byte[]
     */
    public static byte[] convert(Byte[] values)
    {
        if (values == null)
            return null;
        
        byte[] results = new byte[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 字节型对象相互转换，由byte[]转为Byte[]
     * 
     * @param values    byte[]
     * @return          Byte[]
     */
    public static Byte[] convert(byte[] values)
    {
        if (values == null)
            return null;
        
        Byte[] results = new Byte[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 字符型对象相互转换，由Character[]转为char[]
     * 
     * @param values    Character[]
     * @return          char[]
     */
    public static char[] convert(Character[] values)
    {
        if (values == null)
            return null;
        
        char[] results = new char[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 字节型对象相互转换，由char[]转为Character[]
     * 
     * @param values    char[]
     * @return          Character[]
     */
    public static Character[] convert(char[] values)
    {
        if (values == null)
            return null;
        
        Character[] results = new Character[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 短整型对象相互转换，由Short[]转为short[]
     * 
     * @param values    Short[]
     * @return          short[]
     */
    public static short[] convert(Short[] values)
    {
        if (values == null)
            return null;
        
        short[] results = new short[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 短整型对象相互转换，由short[]转为Short[]
     * 
     * @param values    short[]
     * @return          Short[]
     */
    public static Short[] convert(short[] values)
    {
        if (values == null)
            return null;
        
        Short[] results = new Short[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 整型对象相互转换，由Integer[]转为int[]
     * 
     * @param values    Integer[]
     * @return          int[]
     */
    public static int[] convert(Integer[] values)
    {
        if (values == null)
            return null;
        
        int[] results = new int[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 整型对象相互转换，由int[]转为Integer[]
     * 
     * @param values    int[]
     * @return          Integer[]
     */
    public static Integer[] convert(int[] values)
    {
        if (values == null)
            return null;
        
        Integer[] results = new Integer[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 长整型对象相互转换，由Long[]转为long[]
     * 
     * @param values    Long[]
     * @return          long[]
     */
    public static long[] convert(Long[] values)
    {
        if (values == null)
            return null;
        
        long[] results = new long[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 长整型对象相互转换，由long[]转为Long[]
     * 
     * @param values    long[]
     * @return          Long[]
     */
    public static Long[] convert(long[] values)
    {
        if (values == null)
            return null;
        
        Long[] results = new Long[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 浮点型对象相互转换，由Float[]转为float[]
     * 
     * @param values    Float[]
     * @return          float[]
     */
    public static float[] convert(Float[] values)
    {
        if (values == null)
            return null;
        
        float[] results = new float[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 浮点型对象相互转换，由float[]转为Float[]
     * 
     * @param values    float[]
     * @return          Float[]
     */
    public static Float[] convert(float[] values)
    {
        if (values == null)
            return null;
        
        Float[] results = new Float[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 双浮点型对象相互转换，由Double[]转为double[]
     * 
     * @param values    Double[]
     * @return          double[]
     */
    public static double[] convert(Double[] values)
    {
        if (values == null)
            return null;
        
        double[] results = new double[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /**
     * 双浮点型对象相互转换，由double[]转为Double[]
     * 
     * @param values    double[]
     * @return          Double[]
     */
    public static Double[] convert(double[] values)
    {
        if (values == null)
            return null;
        
        Double[] results = new Double[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = values[i];
        
        return results;
    }
    
    /*******************************************/
    //字符串转换为基本类型数组
    /*******************************************/
    
    /**
     * 把字符串按split分隔转换成字符串数组，注意每个字符串都将trim()操作去除前后空字符串(空格,\r,\n,\t,\b等)
     * 
     * @param strs      字符串
     * @param separator 分隔符
     * @return          String[] 转换后的去除前后空格的字符串数组(空格,\r,\n,\t,\b等)
     */
    public static String[] toStringArray(String strs, String separator)
    {
        if (Validates.isEmpty(strs))
            return new String[0];
        
        String[] values = strs.split(separator);
        for (int i = 0; i < values.length; i++)
            values[i] = values[i].trim();
        return values;
    }
    
    /**
     * 把字符串按[逗号]分隔转换成字符串数组，注意每个字符串都将trim()操作去除前后空字符串(空格,\r,\n,\t,\b等)
     * 
     * @param strs      字符串
     * @return          String[] 转换后的去除前后空格的字符串数组(空格,\r,\n,\t,\b等)
     */
    public static String[] toStringArray(String strs)
    {
        return toStringArray(strs, ",");
    }
    
    /**
     * 把对象数组转为字符串数组，对内部所有元素作String.valueOf()操作
     * 
     * @param objs      对象数组
     * @return          String[] 转换后的去除前后空格的字符串数组(空格,\r,\n,\t,\b等)
     */
    public static String[] toStringArray(Object[] objs)
    {
        if (objs == null)
            return new String[0];
        
        String[] strs = new String[objs.length];
        for (int i=0;i<objs.length;i++)
        {
            if (objs[i] == null)
                strs[i] = null;
            else
                strs[i] = String.valueOf(objs[i]).trim();
        }
        
        return strs;
    }
    
    /**
     * 把字符串按[逗号]分隔转换成布尔型数组
     * 
     * @param strs      字符串
     * @return          boolean[] 转换后的如果字符串内容为true的则boolean值为true，否则为false
     */
    public static boolean[] toBooleanArray(String strs)
    {
        String[] values = toStringArray(strs, ",");
        return toBooleanArray(values);
    }
    
    /**
     * 把字符串按分隔符转换成布尔型数组
     * 
     * @param strs      字符串
     * @param separator 分隔符
     * @return          boolean[] 转换后的如果字符串内容为true的则boolean值为true，否则为false
     */
    public static boolean[] toBooleanArray(String strs, String separator)
    {
        String[] values = toStringArray(strs, separator);
        return toBooleanArray(values);
    }
    
    /**
     * 字符串数组转换成布尔型数组
     * 
     * @param values    字符串数组
     * @return          boolean[] 转换后的如果字符串内容为true的则boolean值为true，否则为false
     */
    public static boolean[] toBooleanArray(String[] values)
    {
        boolean[] results = new boolean[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = "true".equals(values[i].trim());
        
        return results;
    }
    
    /**
     * 把字符串按分隔符转换成布尔型数组，支持值为1时也为true
     * 
     * @param strs      字符串
     * @param separator 分隔符
     * @return          boolean[] 转换后的如果字符串内容为true或1的则boolean值为true，否则为false
     */
    public static boolean[] toBooleanArrayForm(String strs, String separator)
    {
        String[] values = toStringArray(strs, separator);
        boolean[] results = new boolean[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = "true".equals(values[i].trim()) || "1".equals(values[i].trim());
        
        return results;
    }
    
    /**
     * 把字符串按[逗号]分隔转换成短整型数组
     * 
     * @param strs      字符串
     * @return          short[]
     */
    public static short[] toShortArray(String strs)
    {
        String[] values = toStringArray(strs, ",");
        return toShortArray(values);
    }

    /**
     * 把字符串按分隔符转换成短整型数组
     * 
     * @param strs      字符串
     * @param separator 分隔符
     * @return          short[]
     */
    public static short[] toShortArray(String strs, String separator)
    {
        String[] values = toStringArray(strs, separator);
        return toShortArray(values);
    }
    
    /**
     * 字符串数组转换成短整型数组
     * 
     * @param values    字符串数组
     * @return short[]
     */
    public static short[] toShortArray(String[] values)
    {
        short[] results = new short[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = Short.parseShort(values[i].trim());
        
        return results;
    }
    
    /**
     * 把字符串按[逗号]分隔转换成整型数组
     * 
     * @param strs      字符串
     * @return          int[] 转换后的整数数组
     */
    public static int[] toIntArray(String strs)
    {
        String[] values = toStringArray(strs, ",");
        return toIntArray(values);
    }
    
    /**
     * 把字符串按分隔符分隔转换成整型数组
     * 
     * @param strs      字符串
     * @param separator 分隔符
     * @return int[]    转换后的整数数组
     */
    public static int[] toIntArray(String strs, String separator)
    {
        String[] values = toStringArray(strs, separator);
        return toIntArray(values);
    }
    
    /**
     * 把字符串数组传换成整型数组
     * 
     * @param values    字符串数组
     * @return          int[] 转换后的整型数组
     */
    public static int[] toIntArray(String[] values)
    {
        int[] results = new int[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = Integer.parseInt(values[i].trim());
        return results;
    }

    /**
     * 把字符串按[逗号]分隔转换成长整型数组
     * 
     * @param strs      字符串
     * @return long[]   转换后的整数数组
     */
    public static long[] toLongArray(String strs)
    {
        String[] values = toStringArray(strs, ",");
        return toLongArray(values);
    }
    
    /**
     * 把字符串按分隔符分隔转换成长整型数组
     * 
     * @param strs      字符串
     * @param separator 分隔符
     * @return long[]   转换后的整数数组
     */
    public static long[] toLongArray(String strs, String separator)
    {
        String[] values = toStringArray(strs, separator);
        return toLongArray(values);
    }
    
    /**
     * 把字符串数组传换成长整型数组
     * 
     * @param values    字符串数组
     * @return long[]   转换后的长整型数组
     */
    public static long[] toLongArray(String[] values)
    {
        long[] results = new long[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = Long.parseLong(values[i].trim());
        return results;
    }
    
    /**
     * 把字符串按[逗号]分隔转换成浮点型数组
     * 
     * @param strs      字符串
     * @return float[]  转换后的浮点型数组
     */
    public static float[] toFloatArray(String strs)
    {
        String[] values = toStringArray(strs, ",");
        return toFloatArray(values);
    }
    
    /**
     * 把字符串按分隔符分隔转换成浮点型数组
     * 
     * @param strs      字符串
     * @param separator 分隔符
     * @return float[]  转换后的浮点型数组
     */
    public static float[] toFloatArray(String strs, String separator)
    {
        String[] values = toStringArray(strs, separator);
        return toFloatArray(values);
    }
    
    /**
     * 把字符串数组传换成浮点型数组
     * 
     * @param values    字符串数组
     * @return float[]  转换后的浮点型数组
     */
    public static float[] toFloatArray(String[] values)
    {
        float[] results = new float[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = Float.parseFloat(values[i].trim());
        return results;
    }
    
    /**
     * 把字符串按[逗号]分隔转换成双浮点型数组
     * 
     * @param strs      字符串
     * @return double[] 转换后的双浮点型数组
     */
    public static double[] toDoubleArray(String strs)
    {
        String[] values = toStringArray(strs, ",");
        return toDoubleArray(values);
    }
    
    /**
     * 把字符串按分隔符分隔转换成双浮点型数组
     * 
     * @param strs      字符串
     * @param separator 分隔符
     * @return double[] 转换后的双浮点型数组
     */
    public static double[] toDoubleArray(String strs, String separator)
    {
        String[] values = toStringArray(strs, separator);
        return toDoubleArray(values);
    }
    
    /**
     * 把字符串数组传换成双浮点型数组
     * 
     * @param values    字符串数组
     * @return double[] 转换后的双浮点型数组
     */
    public static double[] toDoubleArray(String[] values)
    {
        double[] results = new double[values.length];
        for (int i=0;i<values.length;i++)
            results[i] = Double.parseDouble(values[i].trim());
        return results;
    }
   
    /*******************************************/
    //基本类型数组转换为字符串
    /*******************************************/
    
    /**
     * 数组转换成分隔符分隔的字符串
     * 
     * @param values    字符串数组
     * @param separator 分隔符
     * @return          分隔符分隔的字符串,如 abc,def,pdf
     */
    public static String toString(String[] values, String separator)
    {
        if (values == null || values.length == 0)
            return "";
        
        StringBuilder strb = new StringBuilder(values[0]);
        for (int i=1;i<values.length;i++)
        {
            strb.append(separator).append(values[i]);
        }

        return strb.toString();
    }
    
    /**
     * 数组转换成[逗号]分隔的字符串
     * 
     * @param values    字符串数组
     * @return          分隔符分隔的字符串,如 abc,def,pdf
     */
    public static String toString(String[] values)
    {
        return toString(values, ",");
    }
    
    /**
     * 布尔型数组转换成分隔符分隔的字符串
     * 
     * @param values    布尔型数组
     * @param separator 分隔符
     * @return          分隔符分隔的字符串,如 12,1,34
     */
    public static String toString(boolean[] values, String separator)
    {
        if (values == null || values.length == 0)
            return "";
        
        StringBuilder strb = new StringBuilder().append(values[0]);
        for (int i=1;i<values.length;i++)
        {
            strb.append(separator).append(values[i]);
        }
        return strb.toString();
    }
    
    /**
     * 布尔型数组转换成[逗号]分隔的字符串
     * 
     * @param values    布尔型数组
     * @return          分隔符分隔的字符串,如 12,1,34
     */
    public static String toString(boolean[] values)
    {
        return toString(values, ",");
    }
    
    /**
     * 短整型数组转换成分隔符分隔的字符串
     * 
     * @param values    短整型数组
     * @param separator 分隔符
     * @return          分隔符分隔的字符串,如 12,1,34
     */
    public static String toString(short[] values, String separator)
    {
        if (values == null || values.length == 0)
            return "";
        
        StringBuilder strb = new StringBuilder().append(values[0]);
        for (int i=1;i<values.length;i++)
        {
            strb.append(separator).append(values[i]);
        }
        return strb.toString();
    }
    
    /**
     * 短整型数组转换成[逗号]分隔的字符串
     * 
     * @param values    短整型数组
     * @return          分隔符分隔的字符串,如 12,1,34
     */
    public static String toString(short[] values)
    {
        return toString(values, ",");
    }
    
    /**
     * 数组转换成分隔符分隔的字符串
     * 
     * @param values    整型数组
     * @param separator 分隔符
     * @return          分隔符分隔的字符串,如 12,1,34
     */
    public static String toString(int[] values, String separator)
    {
        if (values == null || values.length == 0)
            return "";
        
        StringBuilder strb = new StringBuilder().append(values[0]);
        for (int i=1;i<values.length;i++)
        {
            strb.append(separator).append(values[i]);
        }
        return strb.toString();
    }
    
    /**
     * 整型数组转换成分隔符分隔的字符串
     * 
     * @param values    整型数组
     * @return          分隔符分隔的字符串,如 12,1,34
     */
    public static String toString(int[] values)
    {
        return toString(values, ",");
    }
    
    /**
     * 长整型数组转换成分隔符分隔的字符串
     * 
     * @param values    长整型数组
     * @param separator 分隔符
     * @return          分隔符分隔的字符串,如 1222233,232323,2346777
     */
    public static String toString(long[] values, String separator)
    {
        if (values == null || values.length == 0)
            return "";
        
        StringBuilder strb = new StringBuilder().append(values[0]);
        for (int i=1;i<values.length;i++)
        {
            strb.append(separator).append(values[i]);
        }
        return strb.toString();
    }

    /**
     * 长整型数组转换成[逗号]分隔的字符串
     * 
     * @param values    长整型数组
     * @return          分隔符分隔的字符串,如 1222233,232323,2346777
     */
    public static String toString(long[] values)
    {
        return toString(values, ",");
    }
    
    /**
     * 浮点型数组转换成分隔符分隔的字符串
     * 
     * @param values    浮点型数组
     * @param separator 分隔符
     * @return          分隔符分隔的字符串,如 12222.33,232.323,23.46777
     */
    public static String toString(float[] values, String separator)
    {
        if (values == null || values.length == 0)
            return "";
        
        StringBuilder strb = new StringBuilder().append(values[0]);
        for (int i=1;i<values.length;i++)
        {
            strb.append(separator).append(values[i]);
        }
        return strb.toString();
    }

    /**
     * 浮点型数组转换成[逗号]分隔的字符串
     * 
     * @param values    浮点型数组
     * @return          分隔符分隔的字符串,如 12222.33,232.323,23.46777
     */
    public static String toString(float[] values)
    {
        return toString(values, ",");
    }
    
    /**
     * 双浮点型数组转换成分隔符分隔的字符串
     * 
     * @param values    双浮点型数组
     * @param separator 分隔符
     * @return          分隔符分隔的字符串,如 1222.233,23.2323,2.346777
     */
    public static String toString(double[] values, String separator)
    {
        if (values == null || values.length == 0)
            return "";
        
        StringBuilder strb = new StringBuilder().append(values[0]);
        for (int i=1;i<values.length;i++)
        {
            strb.append(separator).append(values[i]);
        }
        return strb.toString();
    }

    /**
     * 双浮点型数组转换成[逗号]分隔的字符串
     * 
     * @param values    双浮点型数组
     * @return          分隔符分隔的字符串,如 1222.233,23.2323,2.346777
     */
    public static String toString(double[] values)
    {
        return toString(values, ",");
    }
    
    /**
     * 数组转换成分隔符分隔的字符串，每个子项用''处理成SQL支持的格式，子项有'的进行''转义
     * 
     * @param values    字符串数组String[]{"abc", "de''f", "p'df"}
     * @param separator 分隔符
     * @return          分隔符分隔的字符串,转化为str = "'abc','de''''f','p''df'"
     */
    public static String toStringSql(String[] values, String separator)
    {
        if (values == null || values.length == 0)
            return "";
        
        String[] strSqlArr = new String[values.length];
        for (int i=0;i<values.length;i++)
        {
            strSqlArr[i] = Sqls.formatQuote(values[i]);
        }
        
        StringBuilder strb = new StringBuilder();
        strb.append("'").append(strSqlArr[0]).append("'");
        for (int i=1;i<values.length;i++)
        {
            strb.append(separator).append("'").append(values[i]).append("'");
        }

        return strb.toString();
    }
    
    /**
     * 过滤相同的字段
     * 
     * @param values    数组
     * @return          过滤后的数组
     */
    public static String[] toFilterSame(String[] values)
    {
        HashSet<String> strSet = toFilterSameSet(values);
        if (strSet.isEmpty())
            return new String[0];
        
        String[] dStrArr = new String[strSet.size()];
        int i = 0;
        for (String str : strSet){
            dStrArr[i++] = str;
        }
        
        return dStrArr;
    }
    
    /**
     * 过滤相同的字段，返回separator隔开的字符串
     * 
     * @param strs      逗号分隔的字符串,如 1222.233,23.2323,2.346777
     * @return          过滤后separator隔开的字符串
     */
    public static String toFilterSameStr(String strs)
    {
        return toFilterSameStr(toStringArray(strs, ","), ",");
    }
    
    /**
     * 过滤相同的字段，返回separator隔开的字符串
     * 
     * @param strs      分隔符分隔的字符串,如 1222.233,23.2323,2.346777
     * @param separator 分隔符
     * @return          过滤后separator隔开的字符串
     */
    public static String toFilterSameStr(String strs, String separator)
    {
        return toFilterSameStr(toStringArray(strs, separator), separator);
    }
    
    /**
     * 过滤相同的字段，返回separator隔开的字符串
     * 
     * @param values    数组
     * @param separator 分隔符
     * @return          返回separator隔开的字符串
     */
    public static String toFilterSameStr(String[] values, String separator)
    {
        List<String> strList = toFilterSameList(values);
        if (strList.isEmpty())
            return "";
        
        StringBuilder strb = new StringBuilder().append(strList.get(0));
        for (int i=1;i<strList.size();i++){
            strb.append(separator).append(strList.get(i));
        }
        return strb.toString();
    }
    
    /**
     * 过滤相同的字段，返回逗号隔开的字符串
     * 
     * @param values    数组
     * @return          返回逗号隔开的字符串
     */
    public static String toFilterSameStr(String[] values)
    {
        return toFilterSameStr(values, ",");
    }
    
    /**
     * 过滤相同的字段
     * 
     * @param values    数组
     * @return          过滤后的集合
     */
    public static HashSet<String> toFilterSameSet(String[] values)
    {
        if (values == null || values.length == 0)
            return new HashSet<String>(0);
        
        HashSet<String> strSet = new HashSet<String>();
        for (String str : values){
            strSet.add(str.trim());
        }
        
        return strSet;
    }
    
    /**
     * 过滤相同的字段
     * 
     * @param values    数组
     * @return          过滤后的列表
     */
    public static String[] toFilterSameList(List<String> values)
    {
        if (values == null || values.isEmpty())
            return new String[0];
        
        LinkedHashSet<String> strSet = new LinkedHashSet<String>(values.size());
        for (String str : values)
        {
            strSet.add(str.trim());
        }
        
        return strSet.toArray(new String[0]);
    }
    
    /**
     * 过滤相同的字段
     * 
     * @param values    数组
     * @return          过滤后的列表
     */
    public static List<String> toFilterSameList(String[] values)
    {
        if (values == null || values.length == 0)
            return new ArrayList<String>(0);
        
        List<String> strList = new ArrayList<String>();
        for (String str : values)
        {
            str = str.trim();
            if (!strList.contains(str))
                strList.add(str);
        }
        
        return strList;
    }
}
