/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.paging;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.annotation.AnGlobal;
import org.zhiqim.kernel.util.Validates;

/**
 * 标准分页显示，使用按钮实现，支持显示如下：<br>
 * 1、总条数<br>
 * 2、首页<br>
 * 3、上一页<br>
 * 4、显示每5页，如（1－5，6－10）<br>
 * 5、下一页<br>
 * 6、末页<br>
 *
 * @version v1.0.0 @author zouzhigang 2016-9-3 新建与整理
 * @version v1.0.1 @author zouzhigang 2017-11-16 修改为实现Paging接口，采用@AnGlobal
 */
@AnAlias("PagingStd")
@AnGlobal
public class PagingStd implements Paging
{
    /**
     * 提供传入的是URL链接的分页显示
     * 
     * @param result        结果集
     * @return              分页显示HTML
     */
    public String toHtmlLink(PageResult<?> result)
    {
        return toHtmlLink(result, null);
    }
    
    /**
     * 提供传入的是URL链接的分页显示
     * 
     * @param result        结果集
     * @param link          URL链接
     * @return              分页显示HTML
     */
    public String toHtmlLink(PageResult<?> result, String link)
    {
        if (result == null)
            result = PageBuilder.newResult(20);
        
        if (link == null)
            link = "";
        
        String condition = result.getConditionUrl();
        if (Validates.isNotEmpty(condition))
            link += condition + "&";
        else
            link += "?";
        
        StringBuilder strb = new StringBuilder();
        strb.append("<div class=\"z-relative-center z-text-center\">");
        
        strb.append("<span class=\"z-button z-mg-r5 z-readonly\">总&nbsp;<span class=\"z-bold z-text-red\">").append(result.total()).append("</span>&nbsp;条</span>");
        
        if (result.hasFirst())
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\"location.href='" + link + "page=1'\">首页</span>");
        else
            strb.append("<span class=\"z-button z-mg-r3 z-readonly\">首页</span>");
    
        if (result.hasPrevious())
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\"location.href='" + link + "page=" + (result.pageNo() - 1) + "'\">上一页</span>");
        else
            strb.append("<span class=\"z-button z-mg-r3 z-readonly\">上一页</span>");
    
        int lBound = result.getLSegmentNo(5);
        int rBound = result.getRSegmentNo(5);
        
        if (result.hasLSegment(5))
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\"location.href='"+link+"page="+(lBound - 1)+"'\"><<</span>");
            
        for (int i=lBound; i<rBound; i++)
        {
            if (result.pageNo() != i)
                strb.append("<span class=\"z-button z-mg-r3\" onclick=\"location.href='"+link+"page="+ i +"'\">"+i+"</span>");
            else
                strb.append("<span class=\"z-button z-mg-r3 z-red\" onclick=\"location.href='"+link+"page="+ i +"'\">"+i+"</span>");
        }
        
        if (result.hasRSegment(5))
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\"location.href='"+link+"page="+ rBound +"'\">>></span>");
   
        if (result.hasNext())
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\"location.href='" + link + "page=" + (result.pageNo() + 1) + "'\">下一页</span>");
        else
            strb.append("<span class=\"z-button z-mg-r3 z-readonly\">下一页</span>");
    
        if (result.hasLast())
            strb.append("<span class=\"z-button\" onclick=\"location.href='" + link + "page="+ result.totalPages() + "'\">末页</span>");
        else
            strb.append("<span class=\"z-button z-readonly\">末页</span>");
        
        strb.append("<span class=\"z-button z-mg-l5 z-readonly\">").append(result.pageNo()).append("/").append(result.totalPages()).append("(").append(result.pageSize()).append(")").append("</span>");
        strb.append("</div>");
        return strb.toString();
    }
    
    /**
     * 提供传入的是Js函数名的分页显示
     * 
     * @param result        结果集
     * @param jsMethod      Js函数
     * @return              分页显示HTML
     */
    public String toHtmlClick(PageResult<?> result, String jsMethod)
    {
        if (result == null)
            result = PageBuilder.newResult(20);
        
        StringBuilder strb = new StringBuilder();
        strb.append("<div class=\"z-relative-center z-text-center\">");
        
        strb.append("<span class=\"z-button z-mg-r5 z-readonly\">总&nbsp;<span class=\"z-bold z-text-red\">").append(result.total()).append("</span>&nbsp;条</span>");
        
        if (result.hasFirst())
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\""+jsMethod+"('1');\">首页</span>");
        else
            strb.append("<span class=\"z-button z-mg-r3 z-readonly\">首页</span>");
    
        if (result.hasPrevious())
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\""+jsMethod+"('"+(result.pageNo() - 1) + "');\">上一页</span>");
        else
            strb.append("<span class=\"z-button z-mg-r3 z-readonly\">上一页</span>");
    
        int lBound = result.getLSegmentNo(5);
        int rBound = result.getRSegmentNo(5);
        
        if (result.hasLSegment(5))
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\""+jsMethod+"('"+(lBound - 1)+"');\">&lt;&lt;</span>");
            
        for (int i=lBound; i<rBound; i++)
        {
            strb.append("<span class=\"z-button z-mg-r3");
            if (result.pageNo() == i)
                strb.append(" z-red");
            strb.append("\" onclick=\""+jsMethod+"('"+ i +"');\">"+i+"</span>");
        }
        
        if (result.hasRSegment(5))
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\""+jsMethod+"('"+ rBound +"');\">&gt;&gt;</span>");
   
        if (result.hasNext())
            strb.append("<span class=\"z-button z-mg-r3\" onclick=\""+jsMethod+"('"+(result.pageNo() + 1) + "');\">下一页</span>");
        else
            strb.append("<span class=\"z-button z-mg-r3 z-readonly\">下一页</span>");
    
        if (result.hasLast())
            strb.append("<span class=\"z-button\" onclick=\""+jsMethod+"('"+ result.totalPages() + "');\">末页</span>");
        else
            strb.append("<span class=\"z-button z-readonly\">末页</span>");
        
        strb.append("<span class=\"z-button z-mg-l5 z-readonly\">").append(result.pageNo()).append("/").append(result.totalPages()).append("(").append(result.pageSize()).append(")").append("</span>");
        strb.append("</div>");
        return strb.toString();
    }
}
