/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.json.parser;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;

import org.zhiqim.kernel.json.Json;
import org.zhiqim.kernel.json.JsonParser;
import org.zhiqim.kernel.json.Jsons;
import org.zhiqim.kernel.util.Arrays;
import org.zhiqim.kernel.util.Ints;
import org.zhiqim.kernel.util.Longs;
import org.zhiqim.kernel.util.Validates;

/**
 * 数组解析器,8种基本类型数组、字符串数组和对象数组
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class ArrayParser implements JsonParser
{
    private Json jsonMain;
    
    @Override
    public JsonParser set(Json jsonMain)
    {
        this.jsonMain = jsonMain;
        return this;
    }
    
    @Override
    public String toString(Object obj)
    {
        Object[] objs = Arrays.toArray(obj);
        
        StringBuilder strb = new StringBuilder("[");
        for (int i=0;i<objs.length;i++)
        {
            Object o = objs[i];
            if (i > 0)
                strb.append(",").append(jsonMain.toString(o));
            else
                strb.append(jsonMain.toString(o));
        }
        strb.append("]");
        return strb.toString();
    }

    @Override
    public Object toObject(String json, Class<?> cls)
    {
        List<String> jList = toFieldList(json);
        
        if (cls == boolean[].class)       //1(0|1)
            return fromJsonBoolean(jList);
        else if (cls == Boolean[].class)
            return fromJsonBooleanObj(jList);
        else if (cls == byte[].class)     //1
            return fromJsonByte(jList);
        else if (cls == Byte[].class)
            return fromJsonByteObj(jList);
        else if (cls == char[].class)     //1,2,3
            return fromJsonChar(jList);
        else if (cls == Character[].class)
            return fromJsonCharObj(jList);
        else if (cls == short[].class)    //2
            return fromJsonShort(jList);
        else if (cls == Short[].class)
            return fromJsonShortObj(jList);
        else if (cls == int[].class)      //4
            return fromJsonInt(jList);
        else if (cls == Integer[].class)
            return fromJsonIntObj(jList);
        else if (cls == long[].class)     //8
            return fromJsonLong(jList);
        else if (cls == Long[].class)
            return fromJsonLongObj(jList);
        else if (cls == float[].class)    //4
            return fromJsonFloat(jList);
        else if (cls == Float[].class)
            return fromJsonFloatObj(jList);
        else if (cls == double[].class)   //8
            return fromJsonDouble(jList);
        else if (cls == Double[].class)
            return fromJsonDoubleObj(jList);
        else if (cls == String[].class)   //字符串
            return fromJsonString(jList);
        else
        {
            //非基本类型，则使用对象
            Class<?> type = cls.getComponentType();
            Object[] objs = (Object[])Array.newInstance(type, jList.size());
            for (int i=0;i<jList.size();i++)
            {
                objs[i] = (jsonMain.toObject(jList.get(i), type, false));
            }
            
            return objs;
        }
    }

    public static boolean[] fromJsonBoolean(List<String> jList)
    {
        boolean[] val = new boolean[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Boolean.parseBoolean(v);
        }
        
        return val;
    }
    
    public static Boolean[] fromJsonBooleanObj(List<String> jList)
    {
        Boolean[] val = new Boolean[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Boolean.parseBoolean(v);
        }
        
        return val;
    }
    
    public static byte[] fromJsonByte(List<String> jList)
    {
        byte[] val = new byte[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Byte.parseByte(v);
        }
        
        return val;
    }
    
    public static Byte[] fromJsonByteObj(List<String> jList)
    {
        Byte[] val = new Byte[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Byte.parseByte(v);
        }
        
        return val;
    }
    
    public static char[] fromJsonChar(List<String> jList)
    {
        char[] val = new char[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = v.charAt(0);
        }
        
        return val;
    }
    
    public static Character[] fromJsonCharObj(List<String> jList)
    {
        Character[] val = new Character[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = v.charAt(0);
        }
        
        return val;
    }
    
    public static short[] fromJsonShort(List<String> jList)
    {
        short[] val = new short[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Short.parseShort(v);
        }
        
        return val;
    }
    
    public static Short[] fromJsonShortObj(List<String> jList)
    {
        Short[] val = new Short[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Short.parseShort(v);
        }
        
        return val;
    }
    
    public static int[] fromJsonInt(List<String> jList)
    {
        int[] val = new int[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Ints.toInt(v);
        }
        
        return val;
    }
    
    public static Integer[] fromJsonIntObj(List<String> jList)
    {
        Integer[] val = new Integer[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Ints.toInt(v);
        }
        
        return val;
    }
    
    public static long[] fromJsonLong(List<String> jList)
    {
        long[] val = new long[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Longs.toLong(v);
        }
        
        return val;
    }
    
    public static Long[] fromJsonLongObj(List<String> jList)
    {
        Long[] val = new Long[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Longs.toLong(v);
        }
        
        return val;
    }
    
    public static float[] fromJsonFloat(List<String> jList)
    {
        float[] val = new float[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Float.parseFloat(v);
        }
        
        return val;
    }
    
    public static Float[] fromJsonFloatObj(List<String> jList)
    {
        Float[] val = new Float[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Float.parseFloat(v);
        }
        
        return val;
    }
    
    public static double[] fromJsonDouble(List<String> jList)
    {
        double[] val = new double[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Double.parseDouble(v);
        }
        
        return val;
    }
    
    public static Double[] fromJsonDoubleObj(List<String> jList)
    {
        Double[] val = new Double[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            val[i] = Double.parseDouble(v);
        }
        
        return val;
    }
    
    public static String[] fromJsonString(List<String> jList)
    {
        String[] val = new String[jList.size()];
        for (int i=0;i<val.length;i++)
        {
            String v = Jsons.removeStartEndQuotation(jList.get(i));
            //字符串情况，要对转义作删除操作
            val[i] = Jsons.removeEscapeChar(v);
        }
        
        return val;
    }
    
    /** 折分成字符串列表 */
    public static List<String> toFieldList(String json)
    {
        if (Validates.isEmptyBlank(json))
            return new ArrayList<String>();
        
        if (!json.startsWith("[") || !json.endsWith("]"))
            return new ArrayList<String>();
        
        //去除[]
        json = json.substring(1, json.length()-1);
        return Jsons.getFieldList(json);
    }
}
