/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[欢迎加盟知启蒙，一起邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_manager.htm
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zhiqim.manager.presenter;

import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.httpd.context.ZmlContexts;
import org.zhiqim.httpd.context.annotation.AnIntercept;
import org.zhiqim.httpd.validate.ones.IsByteLen;
import org.zhiqim.httpd.validate.onex.IsUserPass;
import org.zhiqim.httpd.validate.two.IsEqual;
import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.constants.CodeConstants;
import org.zhiqim.kernel.extend.HashMapSO;
import org.zhiqim.kernel.extend.MapSO;
import org.zhiqim.kernel.paging.PageResult;
import org.zhiqim.kernel.util.DateTimes;
import org.zhiqim.kernel.util.Ids;
import org.zhiqim.kernel.util.Images;
import org.zhiqim.kernel.util.Randoms;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.kernel.util.codes.Base64;
import org.zhiqim.manager.ZmrBootstrap;
import org.zhiqim.manager.ZmrPassworder;
import org.zhiqim.manager.ZmrSessionUser;
import org.zhiqim.manager.dao.ZmrOperatorDao;
import org.zhiqim.manager.dbo.ZmrAvatar;
import org.zhiqim.manager.dbo.ZmrOperator;
import org.zhiqim.manager.dbo.ZmrParamOperator;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.orm.dbo.Updater;

/**
 * 管理台个人中心控制器
 * 
 * @version v1.0.0 @author zouzhigang 2017-6-5 新建与整理
 */
@AnAlias("ZmrProfilePresenter")
@AnIntercept("chkZmrLogin")
public class ZmrProfilePresenter implements CodeConstants
{
    /** 修改密码 */
    public static void doModifyPass(HttpRequest request) throws Exception
    {
        request.addValidate(new IsByteLen("oldPassword", "密码为6-16位，请输入正确的旧密码", 6, 16));
        request.addValidate(new IsUserPass("newPassword", "新密码不合法，要求6-16位（大小写字母数字和特殊字符必须四选三）"));
        request.addValidate(new IsUserPass("newPassword2", "新密码确认不合法，要求6-16位（大小写字母数字和特殊字符必须四选三）"));
        request.addValidate(new IsEqual("newPassword", "newPassword2", "新密码和新密码确认不一致"));

        if (!request.chkValidate())
        {
            request.setResponseError(request.getAlertMsg());
            return;
        }

        String oldPassword = request.getParameter("oldPassword");
        String newPassword = request.getParameter("newPassword");

        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        String operatorCode = sessionUser.getOperatorCode();

        ZmrPassworder passworder = request.getContextAttribute(ZmrBootstrap.class).getPassworder();
        String oldPassEncode = passworder.encode(operatorCode, oldPassword, sessionUser.getOperatorPassSalt());

        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, operatorCode);
        if (operator == null || !oldPassEncode.equals(operator.getOperatorPass()))
        {
            request.setResponseError("旧密码不正确");
            return;
        }

        String newPassSalt = Randoms.lettersDigitsSecure(64);
        newPassword = passworder.encode(operatorCode, newPassword, newPassSalt);

        Updater updater = new Updater();
        updater.addMaybe("operatorCode", operatorCode);
        updater.addField("operatorPass", newPassword);
        updater.addField("operatorPassSalt", newPassSalt);
        updater.addField("operatorModified", DateTimes.getDateTimeString());
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);

        sessionUser.setOperator(ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, sessionUser.getOperatorCode()));
    }

    /** 更换成手机号 */
    public static void doModifyMobile(HttpRequest request, String mobile) throws Exception
    {
        if (!Validates.isMobile11(mobile, false))
        {
            request.setResponseError("请输入正确的手机号");
            return;
        }

        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);

        Updater updater = new Updater();
        updater.addMust("operatorCode", sessionUser.getOperatorCode());
        updater.addField("operatorMobile", mobile);
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);

        sessionUser.setOperator(ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, sessionUser.getOperatorCode()));
    }

    /** 更换电子邮箱 */
    public static void doModifyEmail(HttpRequest request, String email) throws Exception
    {
        if (!Validates.isEmail(email, false))
        {
            request.setResponseError("请输入正确的邮箱");
            return;
        }

        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        Updater updater = new Updater();
        updater.addMust("operatorCode", sessionUser.getOperatorCode());
        updater.addField("operatorEmail", email);
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);

        sessionUser.setOperator(ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, sessionUser.getOperatorCode()));
    }

    /** 更换旺旺 */
    public static void doModifyWangwang(HttpRequest request, String wagnwang) throws Exception
    {
        if (Validates.isEmptyBlank(wagnwang))
        {
            request.setResponseError("请输入正确的旺旺");
            return;
        }

        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        Updater updater = new Updater();
        updater.addMust("operatorCode", sessionUser.getOperatorCode());
        updater.addField("operatorWangwang", wagnwang);
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);

        sessionUser.setOperator(ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, sessionUser.getOperatorCode()));
    }

    /** 查询系统头像 */
    public static String doQuerySysAvatar(HttpRequest request, int pageNo) throws Exception
    {
        int pageSize = 18;
        if (pageNo < 1)
        {
            pageNo = 1;
        }

        PageResult<ZmrAvatar> result = ZmrBootstrap.table(request).page(ZmrAvatar.class, pageNo, pageSize,
                new Selector("avatarType", 0).addOrderbyAsc("avatarId"));

        MapSO attrMap = new HashMapSO();
        attrMap.put("result", result);
        attrMap.put("pageNo", pageNo);

        return ZmlContexts.parseZmlPath(request, "/zview/zhiqim_manager/presenter/selSysAvatarInfo.zml", attrMap);
    }

    /** 更换成系统头像 */
    public static void doModifySysAvatar(HttpRequest request, long avatarId) throws Exception
    {
        if (ZmrBootstrap.table(request).item(ZmrAvatar.class, avatarId) == null)
        {
            request.setResponseError("选择的头像不存在，请重新选择");
            return;
        }

        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, sessionUser.getOperatorCode());
        if (operator.getOperatorAvatar() != 0)
        {// 不为0，检查是否用户自定义的
            ZmrAvatar avatar = ZmrBootstrap.table(request).item(ZmrAvatar.class, operator.getOperatorAvatar());
            if (avatar.getAvatarType() == 1)
            {// 如果是，要删除自定义的数据
                ZmrBootstrap.table(request).delete(ZmrAvatar.class, operator.getOperatorAvatar());
            }
        }

        Updater updater = new Updater();
        updater.addMust("operatorCode", sessionUser.getOperatorCode());
        updater.addField("operatorAvatar", avatarId);
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);

        sessionUser.setOperator(ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, sessionUser.getOperatorCode()));
    }

    /** 更换成自定义头像 */
    public static void doModifySelfAvatar(HttpRequest request, String data) throws Exception
    {
        // 1.读取头像图片
        int idx = data.indexOf(",");
        byte[] imageData = Base64.decode(data.substring(idx + 1));

        byte[] bytes50 = Images.resizePNG(imageData, 50, 50);
        if (bytes50 == null)
        {
            request.setResponseError("图片错误");
            return;
        }

        byte[] bytes100 = Images.resizePNG(imageData, 100, 100);
        if (bytes100 == null)
        {
            request.setResponseError("图片错误");
            return;
        }

        byte[] bytes150 = Images.resizePNG(imageData, 150, 150);
        if (bytes150 == null)
        {
            request.setResponseError("图片错误");
            return;
        }

        // 2.以数据库数据为准，查操作员信息，防止账号多浏览器登录
        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        ZmrOperator operator = ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, sessionUser.getOperatorCode());

        if (operator.getOperatorAvatar() != 0)
        {
            ZmrAvatar avatar = ZmrBootstrap.table(request).item(ZmrAvatar.class, operator.getOperatorAvatar());
            if (avatar.getAvatarType() == 1)
            {// 操作员自定义的，要删除头像图片
                ZmrBootstrap.table(request).delete(ZmrAvatar.class, operator.getOperatorAvatar());
            }
        }

        // 3.新的头像插入到头像表
        long avatarId = Ids.longId13();
        ZmrAvatar avatar = new ZmrAvatar();
        avatar.setAvatarId(avatarId);
        avatar.setAvatarType(1);// 自定义头像
        avatar.setAvatarTime(System.currentTimeMillis());
        avatar.setAvatar50(bytes50);
        avatar.setAvatar100(bytes100);
        avatar.setAvatar150(bytes150);
        ZmrBootstrap.table(request).insert(avatar);

        // 4.更新操作员头像Id号
        Updater updater = new Updater();
        updater.addMust("operatorCode", sessionUser.getOperatorCode());
        updater.addField("operatorAvatar", avatarId);
        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);

        // 5.更新会话为新的头像Id号
        operator.setOperatorAvatar(avatarId);
        sessionUser.setOperator(operator);
    }

    /** 关闭超级管理员权限 */
    public static void doCloseSuperAdmin(HttpRequest request) throws Exception
    {
        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);
        if (!sessionUser.isSuperAdmin())
        {
            request.setResponseError("你没有关闭超级管理员权限");
            return;
        }

        if (ZmrBootstrap.tableOpr(request).count(ZmrOperator.class, new Selector("operatorType", 1)) == 0)
        {
            request.setResponseError("还没有设置管理员，不能关闭超级管理员");
            return;
        }

        Updater updater = new Updater();
        updater.addField("operatorStatus", 1);
        updater.addMust("operatorType", 0);

        ZmrBootstrap.tableOpr(request).update(ZmrOperator.class, updater);
    }

    /** 修改操作员参数 */
    public static void doModifyOperatorParam(HttpRequest request) throws Exception
    {
        String paramKey = request.getParameter("paramKey");
        String paramValue = request.getParameter("paramValue");
        ZmrSessionUser sessionUser = request.getSessionUser(ZmrSessionUser.class);

        if (ZmrBootstrap.tableOpr(request).count(ZmrParamOperator.class, paramKey) <= 0)
        {
            request.setResponseError("该操作员参数键未定义，请重新输入");
            return;
        }

        ZmrOperatorDao.addOrUpdateOperatorParam(request, sessionUser.getOperatorCode(), paramKey, paramValue);
        sessionUser.setOperator(ZmrBootstrap.tableOpr(request).item(ZmrOperator.class, sessionUser.getOperatorCode()));
    }
}
