/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util.seqs;

import org.zhiqim.kernel.util.Ints;
import org.zhiqim.kernel.util.Longs;
import org.zhiqim.kernel.util.Strings;

/**
 * 整型序列值，默认从1从0x7FFFFFFF(最大值)<br><br>
 * 1、支持指定len/max两种方式来指定最大值，注意max时使用长整型，如999L，999表示长度<br>
 * 2、注意：本类不会验证min/max/len/seq四值在初始化时的值问题，要谨慎使用
 * 
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class Sequence
{
    //最小最大值必须构造即指定，不支持中途修改
    private final long min;
    private final long max;
    private final int len;
    
    //子类可见，用于子类重写next等
    protected long sequence;
    
    /******************************************************************************/
    //构造函数&设置长度方法
    /******************************************************************************/
    
    /** 从1到0x7FFFFFFF */
    public Sequence()
    {
        this(1, Integer.MAX_VALUE, 10, 1);
    }
    
    /** 从1到len个9 */
    public Sequence(int len)
    {
        this(1, Ints.getMaxInt(len), len, 1);
    }
    
    /** 从min到max */
    public Sequence(long min, long max)
    {
        this(min, max, Longs.length(max), min);
    }
    
    /** 从min到max，指定初始值 */
    public Sequence(long min, long max, long seq)
    {
        this(min, max, Longs.length(max), seq);
    }
    
    /** 绝大多数的情况下不会设置错误，所以没有作验证，防止不明所以的NoClassDefFoundError */
    protected Sequence(long min, long max, int len, long seq)
    {
        this.min = min;
        this.max = max;
        this.len = len;
        
        this.sequence = seq;
    }
    
    /******************************************************************************/
    //公开的next方法
    /******************************************************************************/
    
    /** 下一个长整数 */
    public long next()
    {
        synchronized (this)
        {
            chkSequence();
            return sequence++;
        }
    }
    
    /** 下一个整数 */
    public int nextInt()
    {
        return (int)next();
    }
    
    /** 下一个整数前补0的定长字符串 */
    public String nextString()
    {
        return Strings.prefixZero(next(), len);
    }
    
    /******************************************************************************/
    //当前参数值
    /******************************************************************************/
    
    public long getMin()
    {
        return min;
    }
    
    public long getMax()
    {
        return max;
    }

    public int getLen()
    {
        return len;
    }
    
    public long getSequence()
    {
        return sequence;
    }
    
    /******************************************************************************/
    //子类调用的方法
    /******************************************************************************/
    
    /** 检查序列号是否回归，注意：子类需同步锁 */
    protected void chkSequence()
    {
        if (sequence <= min || sequence > max)
        {//0x7FFFFFFF+1=负数<0，取sequence<=min，其他取sequence>max
            sequence = min;
        }
    }
    
    /** 重置序列号为最小值，注意：子类需同步锁 */
    protected void resetSequence()
    {
        sequence = min;
    }
    
    /** 设置序列号值，一般用于初始化时 */
    protected void setSequence(long seq)
    {
        if (seq < min || seq > max)
            return;
        
        sequence = seq;
    }
}
