/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util.codes;

import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import org.zhiqim.kernel.constants.CodeConstants;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Hexs;

/**
 * HMAC算法<br><br>
 * 1.注意编码类型支持HmacMD5 | HmacSHA1 | HmacSHA256 | HmacSHA384 | HmacSHA512<br>
 * 2.对结果支持Base64 | HEX | MD5进行编译成字符串<br>
 * 使用时注意和对方协商后
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class HMAC implements CodeConstants
{
    /**
     * 生成签名数据，指定UTF-8和SHA1编码，以及对结果进行Base64编码
     * 
     * @param data  待加密的数据
     * @param key   加密使用的key
     * @return      生成签名值的字节数组
     */
    public static String signatureUTF8_SHA1_BASE64(String data, String key)
    {
        try
        {
            return signature(data.getBytes(_UTF_8_), key.getBytes(_UTF_8_), _HMAC_SHA1_, _BASE64_);
        }
        catch (Exception e)
        {
            return null;
        }
    }
    
    /**
     * 生成签名数据，指定UTF-8和SHA1编码
     * 
     * @param data  待加密的数据
     * @param key   加密使用的key
     * @return      生成签名值的字节数组
     */
    public static byte[] signatureUTF8_SHA1(String data, String key)
    {
        try
        {
            return signature(data.getBytes(_UTF_8_), key.getBytes(_UTF_8_), _HMAC_SHA1_);
        }
        catch (Exception e)
        {
            return null;
        }
    }
    
    /**
     * 生成签名数据
     * 
     * @param data                      待加密的数据
     * @param key                       加密使用的key
     * @param keyCoding                 加密算法 HmacMD5 | HmacSHA1 | HmacSHA256 | HmacSHA384 | HmacSHA512
     * @param retCoding                 签名编码 Base64 | HEX | MD5
     * @return                          生成签名值
     * @throws InvalidKeyException      
     * @throws NoSuchAlgorithmException
     */
    public static String signature(byte[] data, byte[] key, String keyCoding, String retCoding) throws InvalidKeyException, NoSuchAlgorithmException
    {
        byte[] bytes = signature(data, key, keyCoding);
        if (_HEX_.equals(retCoding))
            return Hexs.toHexString(bytes);
        else if (_BASE64_.equals(retCoding))
            return Base64.encode(bytes);
        else if (_MD5_.equals(retCoding))
            return MD5.encode(bytes);
        else
            throw Asserts.exception("不支持的retCoding");
    }

    /**
     * 生成签名数据
     * 
     * @param data 待加密的数据
     * @param key 加密使用的key
     * @param keyCoding 加密算法 HmacMD5 | HmacSHA1 | HmacSHA256 | HmacSHA384 | HmacSHA512
     * @return 生成签名值
     * @throws InvalidKeyException
     * @throws NoSuchAlgorithmException
     */
    public static byte[] signature(byte[] data, byte[] key, String keyCoding) throws InvalidKeyException, NoSuchAlgorithmException
    {
        SecretKeySpec secretKey = new SecretKeySpec(key, keyCoding);
        Mac mac = Mac.getInstance(secretKey.getAlgorithm());
        mac.init(secretKey);
        return mac.doFinal(data);
    }
}
