/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.util;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.constants.EnumConstants;

/**
 * 基础日期工具类，包括功能如下：<br><br>
 * 1、日历和时间字符串转换<br>
 * 2、日历Long和时间字符串转换<br>
 * 3、日期Date和时间字符串转换<br>
 * 4、年月向上下取值，月最大天数<br>
 * 5、比较两个日期相差值<br>
 * 
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("DateTimes")
public class DateTimes implements EnumConstants
{
    /************************************************************************/
    //日期常量定义，用得比较多的几个定义成常量
    /************************************************************************/
    
    public static final String YYYY_MM_DD              = "yyyy-MM-dd";
    public static final String YYYY_MM_DD_HH_MM_SS     = "yyyy-MM-dd HH:mm:ss";
    
    public static final String YYYYMMDDHHMMSS          = "yyyyMMddHHmmss";
    public static final String MMDDHHMMSS              = "MMddHHmmss";
    public static final String MMDDHHMM                = "MMddHHmm";
    
    /************************************************************************/
    //日历转换，String和Calendar之间转换
    /************************************************************************/
    
    /** 返回当前时间的Date */
    public static Date nowDate()
    {
        return new Date();
    }
    
    /**
     * 字符串转为日历,字符串符合标准格式:yyyy-MM-dd HH:mm:ss
     * 
     * @param datetime  标准时间格式 "yyyy-MM-dd HH:mm:ss"
     * @return          Calendar
     */
    public static Calendar toCalendar(String datetime)
    {
        int index = datetime.indexOf(_SPACE_);
        String date = datetime.substring(0, index);
        String time = datetime.substring(index + 1);

        return toCalendar(date, time);
    }
    
    /**
     * 字符串转为日历,字符串符合标准日期格式yyyy-MM-dd,和标准时间格式HH:mm:ss
     * 
     * @param date      标准日期格式yyyy-MM-dd
     * @param time      标准时间格式HH:mm:ss
     * @return          Calendar
     */
    public static Calendar toCalendar(String date, String time)
    {
        Asserts.notEmptyBlank(date, "date");
        Asserts.notEmptyBlank(time, "time");

        int dateSlash1 = date.indexOf(_MINUS_);
        int dateSlash2 = date.lastIndexOf(_MINUS_);

        Asserts.as(dateSlash1 > 0?null:"非法的参数date["+date+"]");
        Asserts.as(dateSlash2 > dateSlash1?null:"非法的参数date["+date+"]");
        
        String year = date.substring(0, dateSlash1);
        String month = date.substring(dateSlash1 + 1, dateSlash2);
        String day = date.substring(dateSlash2 + 1);

        return toCalendar(year, month, day, time);
    }
    
    /**
     * 通过标准时间输入,年,月,日,时,分,秒,生成Calendar，默认毫秒为0
     * 
     * @param year      年
     * @param month     月
     * @param day       日
     * @param time      时间
     * @return          Calendar
     */
    public static Calendar toCalendar(String year, String month, String day, String time)
    {
        int timeColon1 = time.indexOf(_COLON_);
        int timeColon2 = time.lastIndexOf(_COLON_);

        Asserts.as(timeColon1 > 0?null:"非法的参数time["+time+"]");
        Asserts.as(timeColon2 > timeColon1?null:"非法的参数time["+time+"]");
        
        String hour = time.substring(0, timeColon1);
        String minute = time.substring(timeColon1 + 1, timeColon2);
        String second = time.substring(timeColon2 + 1);
        
        return toCalendar(year, month, day, hour, minute, second);
    }
    
    /**
     * 通过标准时间输入,年,月,日,时,分,秒,生成Calendar，默认毫秒为0
     * 
     * @param year      年
     * @param month     月
     * @param day       日
     * @param hour      时
     * @param minute    分
     * @param second    秒
     * @return          Calendar
     */
    public static Calendar toCalendar(String year, String month, String day, String hour, String minute, String second)
    {
        int yyyy = Integer.parseInt(year);
        int MM = Integer.parseInt(month);
        int dd = Integer.parseInt(day);
        int HH = Integer.parseInt(hour);
        int mm = Integer.parseInt(minute);
        int ss = Integer.parseInt(second);

        return toCalendar(yyyy, MM, dd, HH, mm, ss);
    }
    
    /**
     * 通过标准时间输入,年,月,日,时,分,秒,生成Calendar，默认毫秒为0
     * 
     * @param year 年
     * @param month 月
     * @param day 日
     * @param hour 时
     * @param minute 分
     * @param second 秒
     * @return Calendar
     */
    public static Calendar toCalendar(int year,int month, int day, int hour, int minute, int second)
    {
        Calendar c = Calendar.getInstance();
        c.set(Calendar.YEAR, year);
        c.set(Calendar.MONTH, month-1);
        c.set(Calendar.DATE, day);
        c.set(Calendar.HOUR_OF_DAY, hour);
        c.set(Calendar.MINUTE, minute);
        c.set(Calendar.SECOND, second);
        c.set(Calendar.MILLISECOND, 0);

        return c;
    }
    
    /**
     * HTTP时间格式转化为日历时间格式
     * 
     * @param datetime HTTP时间格式"Wed, 31 Dec 2014 04:06:10 GMT"对应为北京时间 2014-12-31 12:06:10
     * @return Calendar
     */
    public static Calendar toCalendarHttp(String datetime)
    {
        Asserts.as(Validates.isLen(datetime, 29, 29)?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        String week = datetime.substring(0, 3);
        Asserts.as(Validates.isContain(_WEEKS_, week)?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        char comma = datetime.charAt(3);
        Asserts.as(comma == _COMMA_?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        char space = datetime.charAt(4);
        Asserts.as(space == _SPACE_?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        String day = datetime.substring(5, 7);
        Asserts.as(Validates.isDay(day)?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        space = datetime.charAt(7);
        Asserts.as(space == _SPACE_?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        String month = datetime.substring(8, 11);
        Asserts.as(Validates.isContain(_MONTHS_, month)?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        space = datetime.charAt(11);
        Asserts.as(space == _SPACE_?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        String year = datetime.substring(12, 16);
        Asserts.as(Validates.isYear(year)?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        space = datetime.charAt(16);
        Asserts.as(space == _SPACE_?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        String time = datetime.substring(17, 25);
        Asserts.as(Validates.isTime(time)?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        space = datetime.charAt(25);
        Asserts.as(space == _SPACE_?null:"参数["+datetime+"]不是HTTP GMT标准格式");
        
        String gmt = datetime.substring(26);
        Asserts.as("GMT".equals(gmt)?null:"参数["+datetime+"]不是HTTP GMT标准格式");

        int m = Arrays.indexOf(_MONTHS_, month) + 1;
        Calendar calendar = toCalendar(year, Strings.prefixZero(m, 2), day, time);
        calendar.setTimeZone(new SimpleTimeZone(0, "GMT"));
        
        return calendar;
    }
    
    /**
     * 日历转字符串，给定格式的日期时间字符串
     * 
     * @param calendar  日历,给定一个日历
     * @param format    格式,如yyyy-MM-dd HH:mm:ss
     * @return          取得给定日历,给定格式的日期时间字符串
     */
    public static String toDateTimeString(Calendar calendar, String format)
    {
        return new SimpleDateFormat(format).format(calendar.getTime());
    }
    
    /**
     * 日历转标准时间字符串
     * 
     * @param calendar  日历,给定一个日历
     * @return          取得默认的日期时间字符串yyyy-MM-dd HH:mm:ss
     */
    public static String toDateTimeString(Calendar calendar)
    {
        return new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS).format(calendar.getTime());
    }
    
    /**
     * 日历转标准日期字符串
     * 
     * @param calendar 日历,给定一个日历
     * @return String 取得默认的日期时间字符串yyyy-MM-dd
     */
    public static String toDateString(Calendar calendar)
    {
        return new SimpleDateFormat(YYYY_MM_DD).format(calendar.getTime());
    }
    
    /**
     * HTTP格式时间转标准时间字符串
     * 
     * @param datetime  HTTP时间格式Wed, 31 Dec 2014 04:06:10 GMT对应为北京时间 2014-12-31 12:06:10
     * @return          取得默认的日期时间字符串yyyy-MM-dd HH:mm:ss
     */
    public static String toDateTimeStringHttp(String datetime)
    {
        Calendar calendar = toCalendarHttp(datetime);
        return toDateTimeString(calendar);
    }
    
    /********************************************************/
    //时间值转换，String和long之间转换
    /********************************************************/
    
    /**
     * 取得给定时间,给定格式的日期时间字符串,标准格式:yyyy-MM-dd HH:mm:ss
     * 
     * @param datetime  日期,给定一个时间的毫秒数
     * @return          取得给定时间,给定格式的日期时间字符串
     */
    public static String toDateTimeString(long datetime)
    {
        return new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS).format(new Date(datetime));
    }
    
    /**
     * 取得给定时间,给定格式的日期时间字符串,标准格式:"yyyy-MM-dd HH:mm:ss,SSS";
     * 
     * @param datetime 日期,给定一个时间的毫秒数
     * @return String 取得给定时间,给定格式的日期时间字符串
     */
    public static String toDateTimeSSSString(long datetime)
    {
        return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss,SSS").format(new Date(datetime));
    }
    
    /**
     * 取得给定时间,给定格式的日期时间字符串
     * 
     * @param datetime  日期,给定一个时间的毫秒数
     * @param format    格式,如yyyy-MM-dd HH:mm:ss
     * @return          取得给定时间,给定格式的日期时间字符串
     */
    public static String toDateTimeString(long datetime, String format)
    {
        return new SimpleDateFormat(format).format(new Date(datetime));
    }
    
    /**
     * 通过标准时间输入,年,月,日,时,分,秒,生成long值
     * 
     * @param year 年
     * @param month 月
     * @param day 日
     * @param hour 时
     * @param minute 分
     * @param second 秒
     * @return long
     */
    public static long toLong(int year, int month, int day, int hour, int minute, int second)
    {
        return toCalendar(year, month, day, hour, minute, second).getTimeInMillis();
    }
    
    /**
     * 通过标准时间输入,年,月,日,时,分,秒,生成long值
     * 
     * @param year 年
     * @param month 月
     * @param day 日
     * @param hour 时
     * @param minute 分
     * @param second 秒
     * @return long
     */
    public static long toLong(String year,String month, String day, String hour, String minute, String second)
    {
        return toCalendar(year, month, day, hour, minute, second).getTimeInMillis();
    }
    
    /**
     * 字符串转为Long时间
     * 
     * @param datetime  标准日期格式yyyy-MM-dd HH:mm:ss
     * @return          Long时间
     */
    public static long toLong(String datetime)
    {
        return toCalendar(datetime).getTimeInMillis();
    }
    
    /**
     * 字符串转为时间,字符串符合标准日期格式:yyyy-MM-dd,和标准时间格式:HH:mm:ss
     * 
     * @param date      标准日期格式yyyy-MM-dd
     * @param time      标准时间格式HH:mm:ss
     * @return          Long时间
     */
    public static long toLong(String date, String time)
    {
        return toCalendar(date, time).getTimeInMillis();
    }
    
    /**
     * HTTP时间格式转化为长整型时间格式
     * 
     * @param datetime  HTTP时间格式Wed, 31 Dec 2014 04:06:10 GMT对应为北京时间2014-12-31 12:06:10
     * @return          Long时间
     */
    public static long toLongHttp(String datetime)
    {
        Calendar calendar = toCalendarHttp(datetime);
        return calendar.getTimeInMillis();
    }
    
    /********************************************************/
    //时间转换，String和Date之间转换
    /********************************************************/
    /**
     * 生成标准日期，格式为yyyy-MM-dd
     * 
     * @param date      日期对象
     * @return          日期字符串，格式为yyyy-MM-dd
     */
    public static String toDateString(Date date)
    {
        return new SimpleDateFormat(YYYY_MM_DD).format(date);
    }
    
    /**
     * 生成指定格式字符串，如格式为yyyy-MM-dd
     * 
     * @param date      日期对象
     * @param format    格式如yyyy-MM-dd
     * @return          日期字符串
     */
    public static String toDateTimeString(Date date, String format)
    {
        return new SimpleDateFormat(format).format(date);
    }
    
    /**
     * 生成标准格式的字符串 格式为yyyy-MM-dd HH:mm:ss
     *
     * @param date      日期对象
     * @return          生成默认格式的字符串 格式为yyyy-MM-dd HH:mm:ss
     */
    public static String toDateTimeString(Date date)
    {
        return new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS).format(date);
    }
    
    /**
     * 字符串转为时间,字符串符合标准格式yyyy-MM-dd HH:mm:ss
     * 
     * @param datetime  标准时间格式yyyy-MM-dd HH:mm:ss
     * @return          日期对象
     */
    public static Date toDate(String datetime)
    {
        return toCalendar(datetime).getTime();
    }
    
    /**
     * 字符串转为时间,字符串符合标准日期格式yyyy-MM-dd，和标准时间格式:HH:mm:ss
     * 
     * @param date      标准日期格式yyyy-MM-dd
     * @param time      标准时间格式HH:mm:ss
     * @return          日期对象
     */
    public static Date toDate(String date, String time)
    {
        return toCalendar(date, time).getTime();
    }
    
    /**
     * 通过标准时间输入,年,月,日,时,分,秒,生成Date,毫秒位为000
     * 
     * @param year      年
     * @param month     月
     * @param day       日
     * @param hour      时
     * @param minute    分
     * @param second    秒
     * @return          日期对象
     */
    public static Date toDate(String year,String month, String day, String hour, String minute, String second)
    {
        return toCalendar(year, month, day, hour, minute, second).getTime();
    }

    /**
     * 通过标准时间输入,年,月,日,时,分,秒,生成Date,毫秒位为000
     * 
     * @param year      年
     * @param month     月
     * @param day       日
     * @param hour      时
     * @param minute    分
     * @param second    秒
     * @return          Date对象
     */
    public static Date toDate(int year, int month, int day, int hour, int minute, int second)
    {
        return toCalendar(year, month, day, hour, minute, second).getTime();
    }
    
    /**
     * 根据输入的时,分,秒,生成时间格式 HH:mm:ss
     * 
     * @param hour      时
     * @param minute    分
     * @param second    秒
     * @return          生成时间格式为HH:mm:ss
     */
    public static String toTimeString(int hour, int minute, int second)
    {
        String h = "" + hour;
        String m = "" + minute;
        String s = "" + second;

        if (hour < 10)
            h = "0" + hour;
        
        if (minute < 10)
            m = "0" + minute;

        if (second < 10)
            s = "0" + second;

        return h + ":" + m + ":" + s;
    }
    
    /**
     * 根据输入的时间,生成时间格式 HH:mm:ss
     * 
     * @param date      Date对象
     * @return          生成时间格式为HH:mm:ss
     */
    public static String toTimeString(Date date)
    {
        return new SimpleDateFormat("HH:mm:ss").format(date);
    }
    
    /** 
     * 获取当前日期的转化，格式yyyy-MM-dd或yyyy/MM/dd，转为yyyyMMdd 
     * 
     * @param date      日期字符串，yyyy-MM-dd或yyyy/MM/dd
     * @return          日期整型，yyyyMMdd
     */
    public static int toDateInt(String date)
    {
        //去掉中间的符号
        StringBuilder strb = new StringBuilder();
        for (int i=0;i<date.length();i++)
        {
            char c = date.charAt(i);
            if (Validates.isDigit(c))
                strb.append(c);
        }
        
        //转为整型
        return Ints.toInt(strb.toString());
    }
    
    /********************************************************/
    //相关的日期计算方法
    /********************************************************/
    
    /**
     * 获取指定月天数,格式：yyyyMM
     * 
     * @param yearMonth     格式：yyyyMM
     * @return              2月28/29，1月31
     */
    public static int getMonthDays(String yearMonth)
    {
        int year = Integer.parseInt(yearMonth.substring(0, 4));
        int month = Integer.parseInt(yearMonth.substring(4));
        return getMonthDays(year, month);
    }
    
    /**
     * 获取指定月天数
     * 
     * @param yearMonth     年月份201305
     * @return              2月28/29，1月31
     */
    public static int getMonthDays(int yearMonth)
    {
        int year = yearMonth / 100;
        int month = yearMonth % 100;
        return getMonthDays(year, month);
    }
    
    /**
     * 获取指定月天数
     * 
     * @param year 年份
     * @param month 月份
     * @return 2月28/29，1月31
     */
    public static int getMonthDays(int year, int month)
    {
        switch (month)
        {
        case 1:
        case 3:
        case 5:
        case 7:
        case 8:
        case 10:
        case 12:
            return 31;
        case 4:
        case 6:
        case 9:
        case 11:
            return 30;
        default://2月
            return Validates.isLeapYear(year)?29:28;
        }
    }

    /**
     * 获取指定年月的下一个年月，格式:yyyyMM
     * 
     * @param yearMonth 指定的年月，格式:yyyyMM
     * @return yyyyMM
     */
    public static int getNextYearMonth(int yearMonth)
    {
        int year = yearMonth / 100;
        int month = yearMonth % 100;
        if (month < 12)
            month += 1;
        else
        {
            year += 1;
            month = 1;
        }
        
        return year * 100 + month;
    }
    
    /**
     * 获取指定年月的下一个年月，格式:yyyyMM
     * 
     * @param yearMonth 指定的年月，格式:yyyyMM
     * @return yyyyMM
     */
    public static String getNextYearMonth(String yearMonth)
    {
        int curYearMonth = Ints.toInt(yearMonth);
        return String.valueOf(getNextYearMonth(curYearMonth));
    }
    
    /** 
     * 获到下一个月份yyyy-MM, curYearMonth格式yyyyMM或yyyy-MM 
     * 
     * @param yearMonth 指定的年月，格式:yyyyMM或yyyy-MM 
     * @return yyyy-MM
     */
    public static String getNextYearMonthSpe(String yearMonth)
    {
        yearMonth = yearMonth.replace("-", "");
        String nextYearMonth = getNextYearMonth(yearMonth);
        return nextYearMonth.substring(0, 4) + "-" + nextYearMonth.substring(4);
    }
    
    /**
     * 获取下一日的值, yyyyMMdd
     * 
     * @param date  yyyyMMdd格式的int型值
     * @return      yyyyMMdd格式的int型值
     */
    public static int getNextDateInt(int date)
    {
        int yearMonth = date / 100;
        int day = date % 100;
        
        int monthDays = getMonthDays(yearMonth);
        if (day < monthDays)
            day += 1;
        else
        {
            yearMonth = getNextYearMonth(yearMonth);
            day = 1;
        }
        
        return yearMonth * 100 + day;
    }
    
    /**
     * 获取下一小时的值, yyyyMMddHH
     * 
     * @param curHour yyyyMMddHH格式的long型值
     * @return yyyyMMddHH格式的long型值
     */
    public static long getNextHour(long curHour)
    {
        long yearMonthDay = curHour / 100;
        long hour = curHour % 100;
        
        if (hour < 23)
            hour += 1;
        else
        {
            yearMonthDay = getNextDateInt((int)yearMonthDay);
            hour = 0;
        }

        return yearMonthDay * 100 + hour;
    }
    
    /**
     * 获取下一分钟的值, yyyyMMddHHmm
     * 
     * @param curMinute yyyyMMddHHmm格式的long型值
     * @return yyyyMMddHHmm格式的long型值
     */
    public static long getNextMinute(long curMinute)
    {
        long yearMonthDayHour = curMinute / 100;
        long minute = curMinute % 100;
        
        if (minute < 59)
            minute += 1;
        else
        {
            yearMonthDayHour = getNextHour((int)yearMonthDayHour);
            minute = 0;
        }

        return yearMonthDayHour * 100 + minute;
    }
    
    /**
     * 获取指定年月的上一个年月，格式:yyyyMM
     * 
     * @param yearMonth 指定的年月，格式:yyyyMM
     * @return yyyyMM
     */
    public static String getPreviousYearMonth(String yearMonth)
    {
        int year = Integer.parseInt(yearMonth.substring(0, 4));
        int month = Integer.parseInt(yearMonth.substring(4));
        if (month == 1)
        {
            year -= 1;
            month = 12;
        }
        else
        {
            month -= 1;
        }
        
        StringBuilder strb = new StringBuilder().append(year);
        if (month > 9)
            strb.append(month);
        else
            strb.append("0").append(month);
        return strb.toString();
    }
    
    /** 
     * 获到上一个月份yyyy-MM, curYearMonth格式yyyyMM或yyyy-MM 
     * 
     * @param yearMonth 指定的年月，格式:yyyyMM或yyyy-MM 
     * @return yyyy-MM
     */
    public static String getPreviousYearMonthSpe(String yearMonth)
    {
        yearMonth = yearMonth.replace("-", "");
        String prevYearMonth = getPreviousYearMonth(yearMonth);
        return prevYearMonth.substring(0, 4) + "-" + prevYearMonth.substring(4);
    }
    
    /**
     * 获取指定日期格式为20170101得到20161231格式的前一天
     * 
     * @param date  指定日期值
     * @return      前一天日期值
     */
    public static int getPreviousDateInt(int date)
    {
        int year = date / 10000;
        int month = date / 100 % 100;
        int day = date % 100;
        if (day == 1)
        {
            year = month==1?year-1:year;
            month = month==1?12:month-1;
            day = DateTimes.getMonthDays(year, month);
            return year * 10000 + month * 100 + day;
        }
        else
        {
            day--;
            return year * 10000 + month * 100 + day;
        }
    }
    
    /**
     * 获到指定月最后一天yyyy-MM-dd
     * 
     * @param yearMonth yyyyMM
     * @return 如2014-02-28,2014-01-31
     */
    public static String getLastMonthDay(String yearMonth)
    {
        String yearStr = yearMonth.substring(0, 4);
        String monthStr = yearMonth.substring(4);
        int year = Integer.parseInt(yearStr);
        int month = Integer.parseInt(monthStr);
        switch (month)
        {
        case 1:
        case 3:
        case 5:
        case 7:
        case 8:
        case 10:
        case 12:
            return yearStr + "-" + monthStr + "-31";
        case 4:
        case 6:
        case 9:
        case 11:
            return yearStr + "-" + monthStr + "-30";
        case 2:
            int day = Validates.isLeapYear(year)?29:28;
            return yearStr + "-" + monthStr + "-" + day;
        }
        
        throw Asserts.exception("请求参数["+yearMonth+"]格式不对或数据不对");
    }
    
    /********************************************************/
    //时间比较
    /********************************************************/
    
    /**
     * 获取两个日期对象相差年数，不管月日，如2014-02-19比2013-12-22年份差值为1
     * 
     * @param date1 日期对象
     * @param date2 日期对象
     * @return int 年份差值
     */
    public static int compareYear(Date date1, Date date2)
    {
        Asserts.notNull(date1, "date1");
        Asserts.notNull(date2, "date2");
        
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date1);
        int year1 = calendar.get(Calendar.YEAR);
        
        calendar.setTime(date2);
        int year2 = calendar.get(Calendar.YEAR);

        return year1 - year2;
    }

    /**
     * 获取两个日期对象相差月数，年月一起比，如2014-02比2013-12差值为12*(2014-2013)+(2-12)=2，日期不处理
     * 
     * @param date1 日期对象
     * @param date2 日期对象
     * @return int 月份差值
     */
    public static int compareMonth(Date date1, Date date2)
    {
        if (date1 == null || date2 == null)
            return 0;
        
        int year = compareYear(date1, date2);

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date1);
        int month1 = calendar.get(Calendar.MONTH);
        
        calendar.setTime(date2);
        int month2 = calendar.get(Calendar.MONTH);

        /* 进行比较 */
        return 12 * year + (month1 - month2);

    }

    /**
     * 获取两个日期对象相差月数,大一整月才算大一个月
     * 
     * @param date1 字符串对象
     * @param date2 字符串对象
     * @return int 月份差值
     */
    public static int compareMonth(String date1, String date2)
    {
        if (date1 == null || date2 == null)
            return 0;
        
        int year1 = Integer.parseInt(date1.substring(0, 4));
        int year2 = Integer.parseInt(date2.substring(0, 4));
        int month1 = Integer.parseInt(date1.substring(5, 7));
        int month2 = Integer.parseInt(date2.substring(5, 7));
        int day1 = Integer.parseInt(date1.substring(8, 10));
        int day2 = Integer.parseInt(date2.substring(8, 10));
        
        int value = (year1 - year2) * 12 + (month1 - month2);
        if (day1 < day2)
            value--;

        return value;
    }

    /**
     * 获取两个日期对象相差天数
     * 
     * @param date1str String yyyy-MM-dd
     * @param date2str String yyyy-MM-dd
     * @return int 日差值
     */
    public static int compareDay(String date1str, String date2str)
    {
        if (date1str == null || date2str == null)
            return 0;

        Date date1 = toDate(date1str, "00:00:00");
        Date date2 = toDate(date2str, "00:00:00");
        
        return compareDay(date1, date2);
    }
    
    /**
     * 获取两个日期对象相差天数
     * 
     * @param date1 日期对象
     * @param date2 日期对象
     * @return int 日差值
     */
    public static int compareDay(Date date1, Date date2)
    {
        if (date1 == null || date2 == null)
            return 0;

        long time1 = date1.getTime();
        long time2 = date2.getTime();

        long margin = time1 - time2;

        /* 转化成天数 */
        int ret = (int)Math.floor((double)margin / (1000 * 60 * 60 * 24));

        return ret;
    }

    /**
     * 获取两个日期对象相差的小时数
     * 
     * @param date1str String yyyy-MM-dd hh:mm:ss
     * @param date2str String yyyy-MM-dd hh:mm:ss
     * @return int 相差小时数
     */
    public static int compareHour(String date1str, String date2str)
    {
        if (date1str == null || date2str == null)
            return 0;
        
        Date date1 = toDate(date1str);
        Date date2 = toDate(date2str);

        return compareHour(date1, date2);
    }
  
    /**
     * 获取两个日期对象相差的小时数
     * 
     * @param date1 日期对象
     * @param date2 日期对象
     * @return int 相差小时数
     */
    public static int compareHour(Date date1, Date date2)
    {
        if (date1 == null || date2 == null)
            return 0;

        long time1 = date1.getTime();
        long time2 = date2.getTime();

        long margin = time1 - time2;

        int ret = (int)Math.floor((double)margin / (1000 * 60 * 60));

        return ret;
    }
    
    /**
     * 获取两个日期对象相差的分钟数
     * 
     * @param date1str String yyyy-MM-dd hh:mm:ss
     * @param date2str String yyyy-MM-dd hh:mm:ss
     * @return int 相差分钟数
     */
    public static int compareMinute(String date1str, String date2str)
    {
        if (date1str == null || date2str == null)
            return 0;
        
        Date date1 = toDate(date1str);
        Date date2 = toDate(date2str);

        return compareMinute(date1, date2);
    }
    
    /**
     * 获取两个日期对象相差的分钟数
     * 
     * @param date1 日期对象
     * @param date2 日期对象
     * @return int 相差分钟数
     */
    public static int compareMinute(Date date1, Date date2)
    {
        if (date1 == null || date2 == null)
            return 0;

        long time1 = date1.getTime();
        long time2 = date2.getTime();

        long margin = time1 - time2;

        int ret = (int)Math.floor((double)margin / (1000 * 60));

        return ret;
    }

    /**
     * 获取两个日期对象相差秒数
     * 
     * @param date1str String yyyy-MM-dd hh:mm:ss
     * @param date2str String yyyy-MM-dd hh:mm:ss
     * @return int 相差秒数
     */
    public static int compareSecond(String date1str, String date2str)
    {
        if (date1str == null || date2str == null)
            return 0;
        
        Date date1 = toDate(date1str);
        Date date2 = toDate(date2str);

        return compareSecond(date1, date2);
    }
    
    /**
     * 获取两个日期对象相差秒数
     * 
     * @param date1 日期对象
     * @param date2 日期对象
     * @return int 相差秒数
     */
    public static int compareSecond(Date date1, Date date2)
    {
        if (date1 == null || date2 == null)
            return 0;

        long time1 = date1.getTime();
        long time2 = date2.getTime();

        long margin = time1 - time2;

        Long longValue = new Long(margin / (1000));

        return longValue.intValue();
    }
    
    /**
     * 计算指定时间是当前时间的中文差值（1秒前，1分钟前，1小时前，1天前，1月前，1年前）
     * 
     * @param date          日期格式
     * @return              x秒前，x分钟前，x小时前，x天前，x月前，x年前
     */
    public static String compareDateTime(Date date)
    {
        long dateTimeMs = date.getTime();
        return compareDateTime(dateTimeMs);
    }
    
    /**
     * 计算指定时间是当前时间的中文差值（1秒前，1分钟前，1小时前，1天前，1月前，1年前）
     * 
     * @param datetime      毫秒的时间（yyyy-MM-dd HH:mm:ss）格式
     * @return              x秒前，x分钟前，x小时前，x天前，x月前，x年前
     */
    public static String compareDateTime(String datetime)
    {
        long dateTimeMs = toLong(datetime);
        return compareDateTime(dateTimeMs);
    }
    
    /**
     * 计算指定时间是当前时间的中文差值（1秒前，1分钟前，1小时前，1天前，1月前，1年前）
     * 
     * @param dateTimeMs    毫秒的时间（1970-01-01 00：00：00）计算
     * @return              x秒前，x分钟前，x小时前，x天前，x月前，x年前
     */
    public static String compareDateTime(long dateTimeMs)
    {
        long curTime = System.currentTimeMillis() / 1000;
        long cmpTime = dateTimeMs / 1000;
        
        if (curTime <= cmpTime)
            return "1秒前";
        
        long des = curTime - cmpTime;
        long year = des / _SECOND_IN_TIME_[0];
        if (year > 0)
            return year + "年前";
        
        long month = des / _SECOND_IN_TIME_[1];
        if (month > 0)
            return month + "月前";
        
        long day = des / _SECOND_IN_TIME_[2];
        if (day > 0)
            return day + "天前";
        
        long hour = des / _SECOND_IN_TIME_[3];
        if (hour > 0)
            return hour + "小时前";
        
        long minute = des / _SECOND_IN_TIME_[4];
        if (minute > 0)
            return minute + "分钟前";
        
        return des + "秒前";
    }
    
    /********************************************************/
    //时间转换，String和Date之间转换
    /********************************************************/
    
    /**
     * 取得当前时间的HTTP要求格式
     * 
     * @return format: Wed, 25 08 2011 12:11:35 GMT
     */
    public static String getDateTimeHttp()
    {
        Calendar calendar = Calendar.getInstance(new SimpleTimeZone(0, "GMT"));
        return getDateTimeHttp(calendar);
    }
    
    /**
     * 取得指定时间的HTTP要求格式
     * 
     * @return format: Wed, 25 08 2011 12:11:35 GMT
     */
    public static String getDateTimeHttp(long millis)
    {
        Calendar calendar = Calendar.getInstance(new SimpleTimeZone(0, "GMT"));
        calendar.setTimeInMillis(millis);
        return getDateTimeHttp(calendar);
    }
    
    /**
     * 取得指定时间的HTTP要求格式
     * 
     * @return format: Wed, 25 08 2011 12:11:35 GMT
     */
    public static String getDateTimeHttpNextHour()
    {
        Calendar calendar = Calendar.getInstance(new SimpleTimeZone(0, "GMT"));
        calendar.add(Calendar.HOUR_OF_DAY, 1);
        return getDateTimeHttp(calendar);
    }
    
    /**
     * 取得指定Calendar时间的HTTP要求格式
     * 
     * @return format: Wed, 25 08 2011 12:11:35 GMT
     */
    public static String getDateTimeHttp(Calendar calendar)
    {
        StringBuilder buf = new StringBuilder();
        int week = calendar.get(Calendar.DAY_OF_WEEK) - 1;
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        int month = calendar.get(Calendar.MONTH);
        int year = calendar.get(Calendar.YEAR);

        int epoch = (int) ((calendar.getTimeInMillis() / 1000) % (60 * 60 * 24));
        int seconds = epoch % 60;
        epoch = epoch / 60;
        int minutes = epoch % 60;
        int hours = epoch / 60;

        buf.append(_WEEKS_[week]);
        buf.append(',');
        buf.append(' ');
        buf.append(Strings.prefixZero(day, 2));

        buf.append(' ');
        buf.append(_MONTHS_[month]);
        buf.append(' ');
        buf.append(Strings.prefixZero(year, 4));

        buf.append(' ');
        buf.append(Strings.prefixZero(hours, 2));
        buf.append(':');
        buf.append(Strings.prefixZero(minutes, 2));
        buf.append(':');
        buf.append(Strings.prefixZero(seconds, 2));
        buf.append(" GMT");
        
        return buf.toString();
    }
    
    /********************************************************/
    //求星期取值
    /********************************************************/
    
    /**
     * 获取指定时间的星期数:星期日=0;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6;
     * 
     * @return 周数值
     */
    public static int getDateWeek(int date)
    {
        int year = date / 10000;
        int month = date / 100 % 100;
        int day = date % 100;
        
        //采用日期第一秒，标准星期日=1，星期一=2，这里减1处理
        return toCalendar(year, month, day, 0, 0, 1).get(Calendar.DAY_OF_WEEK) - 1;
    }
    
    /**
     * 获取指定时间的星期数:星期日=7;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6;
     * 
     * @param date  日期，格式yyyyMMdd
     * @return      周数值，星期日=7，有利于算周末
     */
    public static int getDateWeek7(int date)
    {
        int week = getDateWeek(date);
        return (week==0)?7:week;
    }
    
    /**
     * 获取中文的星期几
     * 
     * @param date  日期，格式yyyyMMdd
     * @return      星期X
     */
    public static String getDateWeekString(int date)
    {
        int week = getDateWeek(date);
        return _WEEKS_ZH_[week];
    }
    
    /**
     * 获取指定时间的星期数:星期日=0;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6;
     * 
     * @param year  年，格式yyyy
     * @param month 月，格式MM
     * @param day   日，格式dd
     * @return      周数值
     */
    public static int getDateWeek(int year, int month, int day)
    {
        //采用日期第一秒，标准星期日=1，星期一=2，这里减1处理
        return toCalendar(year, month, day, 0, 0, 1).get(Calendar.DAY_OF_WEEK) - 1;
    }
    
    /**
     * 获取指定时间的星期数:星期日=7;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6;
     * 
     * @param year  年，格式yyyy
     * @param month 月，格式MM
     * @param day   日，格式dd
     * @return      周数值
     */
    public static int getDateWeek7(int year, int month, int day)
    {
        int week = getDateWeek(year, month, day);
        return (week==0)?7:week;
    }
    
    /**
     * 获取中文的星期几
     * 
     * @param year  年，格式yyyy
     * @param month 月，格式MM
     * @param day   日，格式dd
     * @return      星期X
     */
    public static String getDateWeekString(int year, int month, int day)
    {
        int week = getDateWeek(year, month, day);
        return _WEEKS_ZH_[week];
    }
    
    /**
     * 获取指定时间的星期数:星期日=0;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6;
     * 
     * @param date  日期，格式yyyy-MM-dd
     * @return 周数值
     */
    public static int getDateWeek(String date)
    {
        return getDateTimeWeek(date + " 00:00:01");
    }
    
    /**
     * 获取指定时间的星期数:星期日=7;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6;
     * 
     * @param date  日期，格式yyyy-MM-dd
     * @return 周数值
     */
    public static int getDateWeek7(String date)
    {
        int week = getDateTimeWeek(date + " 00:00:01");
        return (week==0)?7:week;
    }
    
    /**
     * 获取中文的星期几
     * 
     * @param year  年，格式yyyy
     * @param month 月，格式MM
     * @param day   日，格式dd
     * @return      星期X
     */
    public static String getDateWeekString(String date)
    {
        int week = getDateWeek(date);
        return _WEEKS_ZH_[week];
    }
    
    /**
     * 获取指定时间的星期数:星期日=0;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6;
     * 
     * @param datetime  时间格式yyyy-MM-dd HH:mm:ss
     * @return          周数值
     */
    public static int getDateTimeWeek(String datetime)
    {
        return toCalendar(datetime).get(Calendar.DAY_OF_WEEK) - 1;
    }
    
    /**
     * 获取指定时间的星期数:星期日=7;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6;
     * 
     * @param datetime  时间格式yyyy-MM-dd HH:mm:ss
     * @return          周数值
     */
    public static int getDateTimeWeek7(String datetime)
    {
        int week = getDateTimeWeek(datetime);
        return (week==0)?7:week;
    }
    
    /**
     * 获取中文的星期几
     * 
     * @param datetime  时间格式yyyy-MM-dd HH:mm:ss
     * @return          星期X
     */
    public static String getDateTimeWeekString(String datetime)
    {
        int week = getDateTimeWeek(datetime);
        return _WEEKS_ZH_[week];
    }
    
    /*************************************************************/
    //获取当前时间值，年、月、日期、小时、分钟、秒、星期
    /*************************************************************/
    
    /**
     * @return int[10]<br>
     * int[0] = year        4位数字<br>
     * int[1] = month       1-12<br>
     * int[2] = day         1-31<br>
     * int[3] = hour        0-23<br>
     * int[4] = min         0-59<br>
     * int[5] = sec         0-59<br>
     * int[6] = msec         0-59<br>
     * int[7] = week        1-7<br>
     * int[8] = yearMonth   201401<br>
     * int[9] = monthDays   28-31，当月天数
     */
    public static int[] getCurrent()
    {
        Calendar calendar = Calendar.getInstance();
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        int min = calendar.get(Calendar.MINUTE);
        int sec = calendar.get(Calendar.SECOND);
        int msec = calendar.get(Calendar.MILLISECOND);
        int week = calendar.get(Calendar.DAY_OF_WEEK);
        week = week -1;if (week == 0) week = 7;//变成周一至周七分别为1-7
        
        int yearMonth = year * 100 + month;
        int monthDays = getMonthDays(year, month);//月天数
        
        return new int[]{year, month, day, hour, min, sec, msec, week, yearMonth, monthDays};
    }
    
    /** 获取当前日期，格式yyyyMMdd */
    public static int getCurrentDate()
    {
        Calendar calendar = Calendar.getInstance();
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        
        return year * 10000 + month * 100 + day;
    }
    
    /** 获取当前年份 */
    public static int getCurrentYear()
    {
        return Calendar.getInstance().get(Calendar.YEAR);
    }

    /** 获取当前月份 */
    public static int getCurrentMonth()
    {
        return Calendar.getInstance().get(Calendar.MONTH) + 1;
    }
    
    /** 获取当前月天数 */
    public static int getCurrentMonthDays()
    {
        Calendar calendar = Calendar.getInstance();
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH) + 1;
        return getMonthDays(year, month);
    }
    
    /** 获取当前月内的日期 */
    public static int getCurrentDay()
    {
        return Calendar.getInstance().get(Calendar.DAY_OF_MONTH);
    }

    /** 获取当前时 */
    public static int getCurrentHour()
    {
        return Calendar.getInstance().get(Calendar.HOUR_OF_DAY);
    }

    /** 获取当前分 */
    public static int getCurrentMinute()
    {
        return Calendar.getInstance().get(Calendar.MINUTE);
    }
    
    /** 获取当前秒 */
    public static int getCurrentSecond()
    {
        return Calendar.getInstance().get(Calendar.SECOND);
    }
    
    /** 获取当前毫秒 */
    public static int getCurrentMillis()
    {
        return Calendar.getInstance().get(Calendar.MILLISECOND);
    }
    
    /** 获取当前时间的星期数:星期日=0;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6; */
    public static int getCurrentWeek()
    {
        Calendar calendar = Calendar.getInstance();
        return calendar.get(Calendar.DAY_OF_WEEK) - 1;
    }
    
    /** 获取当前时间的星期数:星期日=7;星期一=1;星期二=2;星期三=3;星期四=4;星期五=5;星期六=6; */
    public static int getCurrentWeek7()
    {
        Calendar calendar = Calendar.getInstance();
        int week = calendar.get(Calendar.DAY_OF_WEEK) - 1;
        return week == 0?7:week;
    }
    
    /** 获取当前的日期时间字符串MMddHHmmss */
    public static int getCurrentTime10()
    {
        Calendar calendar = Calendar.getInstance();
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        int min = calendar.get(Calendar.MINUTE);
        int sec = calendar.get(Calendar.SECOND);
        
        return month * 100000000 + day * 1000000 + hour * 10000 + min * 100 + sec;
    }
    
    /*************************************************************/
    //获取当前时间格式，包括标准DT,D,T,17,14,12,8,7,6等常用格式
    /*************************************************************/
    
    /**
     * 取得当前的日期时间字符串
     * 
     * @param format    格式,如yyyy-MM-dd HH:mm:ss
     * @return          取得当前的日期时间字符串
     */
    public static String getDateTimeString(String format)
    {
        return new SimpleDateFormat(format).format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyyy-MM-dd HH:mm:ss
     * 
     * @return String 取得当前的日期时间字符串yyyy-MM-dd HH:mm:ss
     */
    public static String getDateTimeString()
    {
        return new SimpleDateFormat(YYYY_MM_DD_HH_MM_SS).format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyyyMMddHHmmssSSS
     * 
     * @return String 取得当前的日期时间字符串yyyyMMddHHmmssSSS
     */
    public static String getDateTime17String()
    {
        return new SimpleDateFormat("yyyyMMddHHmmssSSS").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyyyMMddHHmmss
     * 
     * @return String 取得当前的日期时间字符串yyyyMMddHHmmss
     */
    public static String getDateTime14String()
    {
        return new SimpleDateFormat(YYYYMMDDHHMMSS).format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyMMddHHmmss
     * 
     * @return String 取得当前的日期时间字符串yyMMddHHmmss
     */
    public static String getDateTime12String()
    {
        return new SimpleDateFormat("yyMMddHHmmss").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyMMddHHmmss
     * 
     * @return String 取得当前的日期时间字符串yyMMddHHmmss
     */
    public static String getDateTime10String()
    {
        return new SimpleDateFormat("yyMMddHHmm").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串MMddHHmmss
     * 
     * @return String 取得当前的日期时间字符串MMddHHmmss
     */
    public static String getDateTime10String2()
    {
        return new SimpleDateFormat(MMDDHHMMSS).format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyyyMMdd
     * 
     * @return String 取得当前的日期时间字符串
     */
    public static String getDateTime8String()
    {
        return new SimpleDateFormat("yyyyMMdd").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串MMddHHmm
     * 
     * @return String 取得当前的日期时间字符串
     */
    public static String getDateTime8String2()
    {
        return new SimpleDateFormat(MMDDHHMM).format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyyy-MM
     * 
     * @return String 取得当前的日期时间字符串
     */
    public static String getDateTime7String()
    {
        return new SimpleDateFormat("yyyy-MM").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyyyMM
     * 
     * @return String 取得当前的日期时间字符串
     */
    public static String getDateTime6String()
    {
        return new SimpleDateFormat("yyyyMM").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyyy-MM-dd
     * 
     * @return String 取得当前的日期时间字符串
     */
    public static String getDateString()
    {
        return new SimpleDateFormat(YYYY_MM_DD).format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串HH:mm:ss
     * 
     * @return String 取得当前的日期时间字符串
     */
    public static String getTimeString()
    {
        return new SimpleDateFormat("HH:mm:ss").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串yyyy-MM-dd HH:mm:ss,SSS
     * 
     * @return String 取得当前的日期时间字符串YYYY-MM-DD HH:mm:ss,SSS
     */
    public static String getDateTimeSSSString()
    {
        return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss,SSS").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串YYYY/MM/DD HH:mm:ss (移动)
     * 
     * @return String 取得当前的日期时间字符串YYYY/MM/DD HH:mm:ss
     */
    public static String getDateTimeStringSlash()
    {
        return new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(nowDate());
    }
    
    /**
     * 取得当前的日期时间字符串YYYY/MM/DD HH:mm:ss (移动)
     * 
     * @return String 取得当前的日期时间字符串YYYY/MM/DD HH:mm:ss
     */
    public static String getDateStringSlash()
    {
        return new SimpleDateFormat("yyyy/MM/dd").format(nowDate());
    }
    
    /*************************************************************/
    //获取当月月初和月末日期，上月下月月初月末，如2013-12-01,2013-12-31
    /*************************************************************/
    
    /** 取得当月第一天 */
    public static String getFirstMonthDay()
    {
        return getDateTime7String() + "-01";
    }

    /** 获到当前月最后一天yyyy-MM-dd */
    public static String getLastMonthDay()
    {
        String curYearMonth = getDateTime6String();
        return getLastMonthDay(curYearMonth);
    }
    
    /** 获取上一个月第一天 yyyy-MM-dd */
    public static String getPreviousMonthFirstDay()
    {
        String curYearMonth = getDateTime6String();
        String yearMonth = getPreviousYearMonth(curYearMonth);
        return yearMonth.substring(0, 4) + "-" + yearMonth.substring(4) + "-01";
    }
    
    /** 获到上一月最后一天yyyy-MM-dd */
    public static String getPreviousMonthLastDay()
    {
        String curYearMonth = getDateTime6String();
        String yearMonth = getPreviousYearMonth(curYearMonth);
        return getLastMonthDay(yearMonth);
    }
    
    /** 获取下一个月第一天 yyyy-MM-dd */
    public static String getNextMonthFirstDay()
    {
        String curYearMonth = getDateTime6String();
        String yearMonth = getNextYearMonth(curYearMonth);
        return yearMonth.substring(0, 4) + "-" + yearMonth.substring(4) + "-01";
    }
    
    /** 获到下一月最后一天yyyy-MM-dd */
    public static String getNextMonthLastDay()
    {
        String curYearMonth = getDateTime6String();
        String yearMonth = getNextYearMonth(curYearMonth);
        return getLastMonthDay(yearMonth);
    }
    
    /*************************************************************/
    //获取向前推进的时间格式，包括推月、日、时、秒对应的日期和时间格式
    /*************************************************************/
    
    /**
     * 获取前一天对应的当前时间,采用标准格式yyyy-MM-dd
     * 
     * @return String
     */
    public static String getPreviousDateString()
    {
        return getPreviousDateTimeString(YYYY_MM_DD);
    }
    
    /**
     * 获取前一天对应的当前时间,采用短信格式yyyy/MM/dd
     * 
     * @return String
     */
    public static String getPreviousDateStringSlash()
    {
        return getPreviousDateTimeString("yyyy/MM/dd");
    }
    
    /**
     * 获取前一天对应的当前时间
     * 
     * @param format 格式化如 yyyy-MM-dd HH:mm:ss
     * @return String
     */
    public static String getPreviousDateTimeString(String format)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, -1);
        
        return toDateTimeString(calendar, format);
    }
    
    /**
     * 获取前一天对应的当前时间,采用标准格式yyyy-MM-dd HH:mm:ss
     * 
     * @return String
     */
    public static String getPreviousDateTimeString()
    {
        return getPreviousDateTimeString(YYYY_MM_DD_HH_MM_SS);
    }
    
    /**
     * 获取前一天对应的当前时间,采用短信格式yyyy/MM/dd HH:mm:ss
     * 
     * @return String
     */
    public static String getPreviousDateTimeStringSlash()
    {
        return getPreviousDateTimeString("yyyy/MM/dd HH:mm:ss");
    }
    
    //////////////////////////////////////////////////
    //getPreviousDateString()
    /////////////////////////////////////////////////
    
    /**
     * 获取前几天对应的当前时间
     * 
     * @param days 相差天数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateString(int days)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, -days);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取前几年对应的当前时间
     * 
     * @param years 相差年数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateStringByYear(int years)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.YEAR, -years);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取前几月对应的当前时间
     * 
     * @param months 相差月数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateStringByMonth(int months)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, -months);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取前几小时对应的当前时间
     * 
     * @param hours 相差小时数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateStringByHour(int hours)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR_OF_DAY, -hours);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取前几秒对应的当前时间
     * 
     * @param seconds 秒数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateStringBySecond(int seconds)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.SECOND, -seconds);
        
        return toDateString(calendar);
    }
    
    //////////////////////////////////////////////////
    //getPreviousDateTimeString()
    /////////////////////////////////////////////////
    
    /**
     * 获取前几天对应的当前时间
     * 
     * @param days 相差天数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeString(int days)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, -days);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取前几年对应的当前时间
     * 
     * @param years 相差年数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeStringByYear(int years)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.YEAR, -years);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取前几月对应的当前时间
     * 
     * @param months 相差月数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeStringByMonth(int months)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, -months);
        
        return toDateTimeString(calendar);
    }

    
    /**
     * 获取前几小时对应的当前时间
     * 
     * @param hours 相差小时数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeStringByHour(int hours)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR_OF_DAY, -hours);
        
        return toDateTimeString(calendar);
    }

    /**
     * 获取前几秒对应的当前时间
     * 
     * @param seconds 秒数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeStringBySecond(int seconds)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.SECOND, -seconds);
        
        return toDateTimeString(calendar);
    }
    
    //////////////////////////////////////////////////
    //getPreviousDateString(date)
    /////////////////////////////////////////////////
    
    /**
     * 获取指定时间的前一天日期
     * 
     * @param date 指定日期
     * @return yyyy-MM-dd
     */
    public static String getPreviousDateString(String date)
    {
        return getPreviousDateString(date, 1);
    }
    
    /**
     * 获取指定时间前几天的时间
     * 
     * @param date 指定日期
     * @param days 相差天数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateString(String date, int days)
    {
        Calendar calendar = toCalendar(date + " 00:00:00");
        calendar.add(Calendar.DAY_OF_MONTH, -days);
        
        return toDateString(calendar);
    }

    /**
     * 获取指定时间前几年的时间
     * 
     * @param datetime 指定时间
     * @param years 相差年数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateStringByYear(String datetime, int years)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.YEAR, -years);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取指定时间前几月的时间
     * 
     * @param datetime 指定时间
     * @param months 相差月数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateStringByMonth(String datetime, int months)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.MONTH, -months);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取指定时间前几小时的时间
     * 
     * @param datetime 指定时间
     * @param hours 相差小时数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateStringByHour(String datetime, int hours)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.HOUR_OF_DAY, -hours);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取指定时间前几秒的时间
     * 
     * @param datetime 指定时间
     * @param seconds 相差秒数
     * @return String yyyy-MM-dd
     */
    public static String getPreviousDateStringBySecond(String datetime, int seconds)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.SECOND, -seconds);
        
        return toDateString(calendar);
    }
    
    //////////////////////////////////////////////////
    //getPreviousDateTimeString
    /////////////////////////////////////////////////
        
    /**
     * 获取指定时间前几天的时间
     * 
     * @param datetime 指定时间
     * @param days 相差天数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeString(String datetime, int days)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.DAY_OF_MONTH, -days);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取指定时间前几月的时间
     * 
     * @param datetime 指定时间
     * @param years 相差年数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeStringByYear(String datetime, int years)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.YEAR, -years);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取指定时间前几月的时间
     * 
     * @param datetime 指定时间
     * @param months 相差月数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeStringByMonth(String datetime, int months)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.MONTH, -months);
        
        return toDateTimeString(calendar);
    }

    /**
     * 获取指定时间前几小时的时间
     * 
     * @param datetime 指定时间
     * @param hours 相差小时数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeStringByHour(String datetime, int hours)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.HOUR_OF_DAY, -hours);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取指定时间前几秒的时间
     * 
     * @param datetime 指定时间
     * @param second 秒数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getPreviousDateTimeStringBySecond(String datetime, int second)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.SECOND, -second);
        
        return toDateTimeString(calendar);
    }
    
    /*************************************************************/
    //获取向后推进的时间格式，包括推月、日、时、秒对应的日期和时间格式
    /*************************************************************/
    
    /**
     * 获取后一天对应的当前时间,采用标准格式yyyy-MM-dd
     * 
     * @return String
     */
    public static String getNextDateString()
    {
        return getNextDateTimeString(YYYY_MM_DD);
    }
    
    /**
     * 获取后一天对应的当前时间,采用标准格式yyyy-MM-dd HH:mm:ss
     * 
     * @return String
     */
    public static String getNextDateTimeString()
    {
        return getNextDateTimeString(YYYY_MM_DD_HH_MM_SS);
    }
    
    /**
     * 获取后一天对应的当前时间
     * 
     * @param format 格式化如 yyyy-MM-dd HH:mm:ss
     * @return String
     */
    public static String getNextDateTimeString(String format)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, 1);
        
        return toDateTimeString(calendar, format);
    }
    
    /**
     * 获取指定日期的后几天日期
     * 
     * @param days 天数
     * @return Date 几天后的当前日期
     */
    public static Date getNextDate(int days)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, days);
        
        return calendar.getTime();
    }
    
    /**
     * 获取后几年对应的当前日期
     * 
     * @param years 年数
     * @return 几年后的当前日期格式
     */
    public static Date getNextDateByYeah(int years)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.YEAR, years);
        
        return calendar.getTime();
    }
    
    /**
     * 获取指定日期的后几月日期
     * 
     * @param months 月数
     * @return Date 几月后的当前日期
     */
    public static Date getNextDateByMonth(int months)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, months);
        
        return calendar.getTime();
    }
    
    /**
     * 获取指定日期的后几小时日期
     * 
     * @param hours 小时数
     * @return Date 几小时后的当前日期
     */
    public static Date getNextDateByHour(int hours)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR_OF_DAY, hours);
        
        return calendar.getTime();
    }
    
    /**
     * 获取指定日期的后几秒日期
     * 
     * @param seconds   秒数
     * @return          Date 几小时后的当前日期
     */
    public static Date getNextDateBySecond(int seconds)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.SECOND, seconds);
        
        return calendar.getTime();
    }
    
    /**
     * 获取后几天对应的当前日期
     * 
     * @param days 日期数
     * @return String 后几天对应的当前日期
     */
    public static String getNextDateString(int days)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, days);
        
        return toDateString(calendar);
    }

    /**
     * 获取后几年对应的当天日期
     * 
     * @param years 年数
     * @return String 后几年的日期yyyy-MM-dd
     */
    public static String getNextDateStringByYeah(int years)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.YEAR, years);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取后几月对应的当前日期
     * 
     * @param months 月数
     * @return String 几月后的当前日期
     */
    public static String getNextDateStringByMonth(int months)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, months);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取后几小时对应的当前日期
     * 
     * @param hours 格式化如 yyyy-MM-dd
     * @return String 几小时后对应的当前日期
     */
    public static String getNextDateStringByHour(int hours)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR_OF_DAY, hours);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取后几秒对应的当前日期
     * 
     * @param seconds 秒数
     * @return String 几秒后对应的当前日期
     */
    public static String getNextDateStringBySecond(int seconds)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.SECOND, seconds);
        
        return toDateString(calendar);
    }
    
    /**
     * 获取指定时间的后一天日期
     * 
     * @param date 指定日期
     * @return yyyy-MM-dd
     */
    public static String getNextDateString(String date)
    {
        return getNextDateString(date, 1);
    }
    
    /**
     * 获取指定日期的后几天日期
     * 
     * @param date 指定日期
     * @param days 指定天数
     * @return yyyy-MM-dd
     */
    public static String getNextDateString(String date, int days)
    {
        Calendar calendar = toCalendar(date + " 00:00:00");
        calendar.add(Calendar.DAY_OF_MONTH, days);
        return toDateString(calendar);
    }
    
    /**
     * 获取指定日期的后几年日期
     * 
     * @param date 指定日期
     * @param years 指定年数
     * @return yyyy-MM-dd
     * @throws Exception
     */
    public static String getNextDateStringByYear(String date, int years)
    {
        Calendar calendar = toCalendar(date + " 00:00:00");
        calendar.add(Calendar.YEAR, years);
        return toDateString(calendar);
    }
    
    /**
     * 获取指定日期的后几月日期
     * 
     * @param date 指定日期
     * @param months 指定月数
     * @return yyyy-MM-dd
     * @throws Exception
     */
    public static String getNextDateStringByMonth(String date, int months)
    {
        Calendar calendar = toCalendar(date + " 00:00:00");
        calendar.add(Calendar.MONTH, months);
        return toDateString(calendar);
    }
    
    /**
     * 获取指定日期的后几小时
     * 
     * @param date 指定日期
     * @param hours 指定小时数
     * @return yyyy-MM-dd
     * @throws Exception
     */
    public static String getNextDateStringByHour(String date, int hours)
    {
        Calendar calendar = toCalendar(date + " 00:00:00");
        calendar.add(Calendar.HOUR_OF_DAY, hours);
        return toDateString(calendar);
    }
    
    /**
     * 获取指定日期的后几秒
     * 
     * @param date 指定日期
     * @param seconds 指定秒数
     * @return yyyy-MM-dd
     * @throws Exception
     */
    public static String getNextDateStringBySecond(String date, int seconds)
    {
        Calendar calendar = toCalendar(date + " 00:00:00");
        calendar.add(Calendar.SECOND, seconds);
        return toDateString(calendar);
    }
    
    /**
     * 获取后几天对应的当前时间
     * 
     * @param days 天数
     * @return String 几天后对应的当前时间
     */
    public static String getNextDateTimeString(int days)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, days);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取后几年对应的当前时间
     * 
     * @param years 年数
     * @return String 几年后对应的当前时间
     */
    public static String getNextDateTimeStringByYear(int years)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.YEAR, years);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取后几月对应的当前时间
     * 
     * @param months 月数
     * @return String 几月后对应的当前时间
     */
    public static String getNextDateTimeStringByMonth(int months)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.MONTH, months);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取后几小时对应的当前时间
     * 
     * @param hours 小时数
     * @return String 几小时后对应的当前时间
     */
    public static String getNextDateTimeStringByHour(int hours)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR_OF_DAY, hours);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取后几秒对应的当前时间
     * 
     * @param seconds 秒数
     * @return String 几秒后对应的当前时间
     */
    public static String getNextDateTimeStringBySecond(int seconds)
    {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.SECOND, seconds);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取指定时间后几日的时间
     * 
     * @param datetime 指定时间
     * @param days 相差日数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getNextDateTimeString(String datetime, int days)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.DAY_OF_MONTH, days);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取指定时间后几年的时间
     * 
     * @param datetime 指定时间
     * @param years 相差年数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getNextDateTimeStringByYear(String datetime, int years)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.YEAR, years);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取指定时间后几月的时间
     * 
     * @param datetime 指定时间
     * @param months 相差月数
     * @return String yyyy-MM-dd HH:mm:ss
     */
    public static String getNextDateTimeStringByMonth(String datetime, int months)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.MONTH, months);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取指定时间后几小时对应的当前时间
     * 
     * @param datetime 格式 yyyy-MM-dd HH:mm:ss
     * @param hours 小时数
     * @return yyyy-MM-dd HH:mm:ss
     */
    public static String getNextDateTimeStringByHour(String datetime, int hours)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.HOUR_OF_DAY, hours);
        
        return toDateTimeString(calendar);
    }
    
    /**
     * 获取指定时间后几秒对应的当前时间
     * 
     * @param datetime 格式 yyyy-MM-dd HH:mm:ss
     * @param seconds 秒数
     * @return yyyy-MM-dd HH:mm:ss
     */
    public static String getNextDateTimeStringBySecond(String datetime, int seconds)
    {
        Calendar calendar = toCalendar(datetime);
        calendar.add(Calendar.SECOND, seconds);
        
        return toDateTimeString(calendar);
    }
    
    /*******************************************************************************/
    //时区
    /*******************************************************************************/
    
    /** 获取当前时区 */
    public static TimeZone getTimeZone()
    {
        return Calendar.getInstance().getTimeZone();
    }
    
    /** 获取当前时区小时数 */
    public static int getTimeZoneHour()
    {
        return Calendar.getInstance().getTimeZone().getRawOffset() / (60 * 60 * 1000);
    }
    
    /** 获取当前时区分钟数 */
    public static int getTimeZoneMinute()
    {
        return Calendar.getInstance().getTimeZone().getRawOffset() / (60 * 1000);
    }
    
    /** 获取当前时区秒数 */
    public static int getTimeZoneSecond()
    {
        return Calendar.getInstance().getTimeZone().getRawOffset() / 1000;
    }
}
