/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.schedule;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.util.DateTimes;

/**
 * 每月任务，设置天、时、分、秒后在该点运行
 * 时间格式：yyyyMM
 * 
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Month")
public final class Month extends TaskThreader
{
    private int taskDay     = 1;//指定处理的日期，=0表示月末当日
    private int taskHour    = 0;//指定处理的小时
    private int taskMinute  = 0;//指定处理的分钟
    private int taskSecond  = 0;//指定处理的秒
    
    /** 空构造函数,用于newInstance后再设置参数 */
    public Month(){}
    
    /**
     * 设置定时安排时间点，指定每月的时该任务运行
     * 
     * @param task      任务
     * @param day       每月指定的日期0-28，0表示月末，不支持29,30,31三个日期
     * @param hour      指定小时
     * @param minute    指定分钟
     * @param second    指定秒
     */
    public Month(Task task, int day, int hour, int minute, int second)
    {
        schedule(task, new int[]{day, hour, minute, second});
    }
    
    /**
     * 任务名称
     * 
     * @return          线程名称
     */
    public String getName()
    {
        return "Month";
    }
    
    /**
     * 任务时间安排
     * 
     * @return          时间安排
     */
    public String getTime()
    {
        return new StringBuilder().append(taskDay).append("/").append(taskHour).append("/").append(taskMinute).append("/").append(taskSecond).toString();
    }
    
    /**
     * 时钟安排
     * 
     * @param task      任务
     * @param time      时间数组格式
     * @return          返回本对象
     */
    protected TaskThreader schedule(Task task, int[] time)
    {
        if (task == null || time == null || time.length != 4)
            throw new RuntimeException("Month传入的参数为空或数组长度不对");
        
        if (time[0] < 0 || time[0] > 28 || time[1] < 0 || time[1] > 23 || time[2] < 0 || time[2] > 59 || time[3] < 0 || time[3] > 59)
            throw new RuntimeException("Month传时的时间格式不正确");
        
        super.task = task;
        this.taskDay = time[0];
        this.taskHour = time[1];
        this.taskMinute = time[2];
        this.taskSecond = time[3];
        return this;
    }

    @Override
    protected long calcLastTime(int curYearMonth, int curMonthMaxDay, int curDay, int curHour, int curMinute, int curSecond, int curWeek)
    {
        return curYearMonth;
    }
    
    @Override
    protected long calcLastTimeNextTime()
    {
        return DateTimes.getNextYearMonth((int)lastTime);
    }
    
    @Override
    protected boolean isArrive(int curYearMonth, int curMonthMaxDay, int curDay, int curHour, int curMinute, int curSecond, int curWeek)
    {
        //下月，判断日期，日期未达
        int taskHDay = taskDay==0?curMonthMaxDay:taskDay;//==0表示月末
        if (curDay < taskHDay)
            return false;
        
        //下月，日期已达
        if (curDay > taskHDay)
            return true;
        
        //下月，日期相等，小时未达
        if (curHour < taskHour)
            return false;
        
        //下月，日期相等，小时已达
        if (curHour > taskHour)
            return true;
        
        //下月，日期小时相等，分钟未达
        if (curMinute < taskMinute)
            return false;
        
        //下月，日期小时相等，分钟已达
        if (curMinute > taskMinute)
            return true;
        
        //下月，日期小时分钟相等，比较秒数
        return curSecond >= taskSecond;
    }
    
    public int getTaskDay()
    {
        return taskDay;
    }

    public int getTaskHour()
    {
        return taskHour;
    }

    public int getTaskMinute()
    {
        return taskMinute;
    }

    public int getTaskSecond()
    {
        return taskSecond;
    }

    public boolean isMonthLastDay()
    {
        return taskDay == 0;
    }
    
    @Override
    public String toString()
    {
        return new StringBuilder("{type:Month, ")
            .append("time:").append(taskDay).append("/").append(taskHour).append("/").append(taskMinute).append("/").append(taskSecond).append(", ")
            .append("first:").append(isFirstExecute).append("}")
            .toString();
    }
}
