/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.schedule;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.util.DateTimes;

/**
 * 每10分钟任务，设置分、秒后在该点运行
 * 时间格式：yyyyMMddHHm，0-5表示10分钟取商数
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Minute10")
public final class Minute10 extends TaskThreader
{
    private int taskMinute  = 0;                //指定处理的分钟
    private int taskSecond  = 0;                //指定处理的秒
    
    /** 空构造函数,用于newInstance后再设置参数 */
    public Minute10(){}
    
    /**
     * 设置定时安排时间点，指定每日的时该任务运行
     * 
     * @param task      任务
     * @param minute    指定0-9之间的分钟
     * @param second    指定秒
     */
    public Minute10(Task task, int minute, int second)
    {
        schedule(task, new int[]{minute, second});
    }
    
    /**
     * 任务名称
     * 
     * @return          线程名称
     */
    public String getName()
    {
        return "Minute10";
    }
    
    /**
     * 任务时间安排
     * 
     * @return          时间安排
     */
    public String getTime()
    {
        return new StringBuilder().append(taskMinute).append("/").append(taskSecond).toString();
    }
    
    /**
     * 时钟安排
     * 
     * @param task      任务
     * @param time      时间数组格式
     * @return          返回本对象
     */
    protected TaskThreader schedule(Task task, int[] time)
    {
        if (task == null || time == null || time.length != 2)
            throw new RuntimeException("Minute10传入的参数为空或数组长度不对");
        
        if (time[0] < 0 || time[0] > 9 || time[1] < 0 || time[1] > 59)
            throw new RuntimeException("Minute10传时的时间格式不正确");
        
        super.task = task;
        this.taskMinute = time[0];
        this.taskSecond = time[1];
        return this;
    }
    
    @Override
    protected long calcLastTime(int curYearMonth, int curMonthMaxDay, int curDay, int curHour, int curMinute, int curSecond, int curWeek)
    {
        return curYearMonth * 100000L + curDay * 1000 + curHour * 10 + curMinute / 10;//取10分钟的商
    }
    
    @Override
    protected long calcLastTimeNextTime()
    {
        long yearMonthDayHour = lastTime / 10;
        long curMinuteMod = lastTime % 10;
        
        if (curMinuteMod < 5)
            curMinuteMod += 1;
        else
        {
            curMinuteMod = 0;
            yearMonthDayHour = DateTimes.getNextHour(yearMonthDayHour);
        }
        
        return yearMonthDayHour * 10 + curMinuteMod;
    }
    
    @Override
    protected final boolean isArrive(int curYearMonth, int curMonthMaxDay, int curDay, int curHour, int curMinute, int curSecond, int curWeek)
    {
        int curMinuteMod = curMinute % 10;//取分钟的余
        
        //下10分钟，分钟未达
        if (curMinuteMod < taskMinute)
            return false;
            
        //下10分钟，分钟已达
        if (curMinuteMod > taskMinute)
            return true;
        
        //下10分钟，分钟相等，比较秒数
        return curSecond >= taskSecond;
    }
    
    public int getTaskMinute()
    {
        return taskMinute;
    }

    public int getTaskSecond()
    {
        return taskSecond;
    }
    
    @Override
    public String toString()
    {
        return new StringBuilder("{type:Minute10, ")
            .append("time:").append(taskMinute).append("/").append(taskSecond).append(", ")
            .append("first:").append(isFirstExecute).append("}")
            .toString();
    }
}
