/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.schedule;

import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.util.DateTimes;

/**
 * 每(0-12,12-23)两时段小时任务，设置分、秒后在每天在指定点运行两次，每12小时运行一次
 * 时间格式：yyyyMMddHH[0-1]，0-1表示12小时取商数
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
@AnAlias("Hour12")
public final class Hour12 extends TaskThreader
{
    private int taskHour    = 0;            //指定处理的小时
    private int taskMinute  = 0;            //指定处理的分钟
    private int taskSecond  = 0;            //指定处理的秒
    
    /** 空构造函数,用于newInstance后再设置参数 */
    public Hour12(){}
    
    /**
     * 设置定时安排时间点，指定每日的时该任务运行
     * 
     * @param task      任务
     * @param hour      指定0-5小时
     * @param minute    指定分钟
     * @param second    指定秒
     */
    public Hour12(Task task, int hour, int minute, int second)
    {
        schedule(task, new int[]{hour, minute, second});
    }
    
    /**
     * 任务名称
     * 
     * @return          线程名称
     */
    public String getName()
    {
        return "Hour12";
    }
    
    /**
     * 任务时间安排
     * 
     * @return          时间安排
     */
    public String getTime()
    {
        return new StringBuilder().append(taskHour).append("/").append(taskMinute).append("/").append(taskSecond).toString();
    }
    
    /**
     * 时钟安排
     * 
     * @param task      任务
     * @param time      时间数组格式
     * @return          返回本对象
     */
    protected TaskThreader schedule(Task task, int[] time)
    {
        if (task == null || time == null || time.length != 3)
            throw new RuntimeException("Hour12传入的参数为空或数组长度不对");
        
        if (time[0] < 0 || time[0] > 11 || time[1] < 0 || time[1] > 59 || time[2] < 0 || time[2] > 59)
            throw new RuntimeException("Hour12传时的时间格式不正确");
        
        super.task = task;
        this.taskHour = time[0];
        this.taskMinute = time[1];
        this.taskSecond = time[2];
        return this;
    }
    
    @Override
    protected long calcLastTime(int curYearMonth, int curMonthMaxDay, int curDay, int curHour, int curMinute, int curSecond, int curWeek)
    {
        return curYearMonth * 1000L + curDay * 10 + curHour / 12;//取12小时的商
    }
    
    @Override
    protected long calcLastTimeNextTime()
    {
        long yearMonthDay = lastTime / 10;
        long curHourMod = lastTime % 10;
        
        if (curHourMod < 1)
            curHourMod += 1;
        else
        {
            curHourMod = 0;
            yearMonthDay = DateTimes.getNextDateInt((int)yearMonthDay);
        }
        
        return yearMonthDay * 10 + curHourMod;
    }
    
    @Override
    protected final boolean isArrive(int curYearMonth, int curMonthMaxDay, int curDay, int curHour, int curMinute, int curSecond, int curWeek)
    {
        int curHourMod = curHour % 12;//取12小时的余
        
        //下半天，小时数未达
        if (curHourMod < taskHour)
            return false;
        
        //下半天，小时数已过，一般由于不及时导致，表示到达
        if (curHourMod > taskHour)
            return true;
        
        //下半天，小时相等，比较分钟，分钟未到达
        if (curMinute < taskMinute)
            return false;
        
        //下半天，小时相等，比较分钟，分钟已达
        if (curMinute > taskMinute)
            return true;
        
        //下半天，小时分钟相等，比较秒数
        return curSecond >= taskSecond;
    }

    public int getTaskHour()
    {
        return taskHour;
    }

    public int getTaskMinute()
    {
        return taskMinute;
    }

    public int getTaskSecond()
    {
        return taskSecond;
    }
}
