/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.logging;

import java.util.List;

import org.zhiqim.kernel.util.Strings;

/**
 * 日志类，由日志工厂创建
 *
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class Log implements LogConstants
{
    private LogService logService;

    private String mustName;
    private Class<?> mustClass;
    
    public Log(String mustName, LogService logService)
    {
        this.mustName = mustName;
        this.logService = logService;
    }
    
    public Log(Class<?> mustClass, LogService logService)
    {
        this.mustClass = mustClass;
        this.logService = logService;
    }
    
    public void setLogService(LogService logService)
    {
        this.logService = logService;
    }
    
    public String getMustName()
    {
        return mustName;
    }

    public Class<?> getMustClass()
    {
        return mustClass;
    }
    
    public LogService getLogService()
    {
        return logService;
    }
    
    /** 获取日志生成时对应的参数名 */
    public String getParamName()
    {
        return (mustClass != null)?mustClass.getName():mustName;
    }
    
    /******************************************************/
    //调试
    /******************************************************/
 
    public void debug(Object message)
    {
        handle(DEBUG, message);
    }
    
    public void debug(Throwable throwable)
    {
        handle(DEBUG, null, throwable);
    }
    
    public void debug(Object message, Object... arguments)
    {
        handle(DEBUG, message, arguments);
    }
    
    public void debug(Object message, Throwable throwable, Object... arguments)
    {
        handle(DEBUG, message, throwable, arguments);
    }
    
    /******************************************************/
    //信息
    /******************************************************/
    
    public void info(Object message)
    {
        handle(INFO, message);
    }
    
    public void info(Throwable throwable)
    {
        handle(INFO, null, throwable);
    }
    
    public void info(Object message, Object... arguments)
    {
        handle(INFO, message, arguments);
    }
    
    public void info(Object message, Throwable throwable, Object... arguments)
    {
        handle(INFO, message, throwable, arguments);
    }
    
    /******************************************************/
    //错误
    /******************************************************/
    
    public void error(Object message)
    {
        handle(ERROR, message);
    }
    
    public void error(Throwable throwable)
    {
        handle(ERROR, null, throwable);
    }
    
    public void error(Object message, Object... arguments)
    {
        handle(ERROR, message, arguments);
    }
    
    public void error(Object message, Throwable throwable, Object... arguments)
    {
        handle(ERROR, message, throwable, arguments);
    }
    
    /******************************************************/
    //后端警告
    /******************************************************/
    
    public void warn(Object message)
    {
        handle(WARN, message);
    }
    
    public void warn(Throwable throwable)
    {
        handle(WARN, null, throwable);
    }
    
    public void warn(Object message, Object... arguments)
    {
        handle(WARN, message, arguments);
    }
    
    public void warn(Object message, Throwable throwable, Object... arguments)
    {
        handle(WARN, message, throwable, arguments);
    }
    
    /******************************************************/
    //后台危急错误
    /******************************************************/
    
    public void fatal(Object message)
    {
        handle(FATAL, message);
    }
    
    public void fatal(Throwable throwable)
    {
        handle(FATAL, null, throwable);
    }
    
    public void fatal(Object message, Object... arguments)
    {
        handle(FATAL, message, arguments);
    }
    
    public void fatal(Object message, Throwable throwable, Object... arguments)
    {
        handle(FATAL, message, throwable, arguments);
    }
    
    /******************************************************/
    //判断是否开启
    /******************************************************/
    
    /** 调试是否开启 */
    public boolean isDebugEnabled()
    {
        return allow(DEBUG);
    }

    /** 信息是否开启 */
    public boolean isInfoEnabled()
    {
        return allow(INFO);
    }

    /** 错误是否开启 */
    public boolean isErrorEnabled()
    {
        return allow(ERROR);
    }
    
    /** 后端警告是否开启 */
    public boolean isWarnEnabled()
    {
        return allow(WARN);
    }

    /** 后端危急错误是否开启 */
    public boolean isFatalEnabled()
    {
        return allow(FATAL);
    }
    
    /**
     * 处理日志参数
     * 
     * @param level         日志级别
     * @param message       日志信息
     * @param arguments     日志信息参数
     */
    private void handle(int level, Object message, Object... arguments)
    {
        handle(level, message, null, arguments);
    }
    
    /**
     * 处理日志打印
     * 
     * @param level         日志级别
     * @param message       日志信息
     * @param throwable     日志异常
     * @param arguments     日志信息参数
     */
    private void handle(int level, Object message, Throwable throwable, Object... arguments)
    {
        List<Logger> loggerList = logService.getLoggerList(this, level);
        if (loggerList == null || loggerList.isEmpty())
        {//未开启
            return;
        }
        
        String msg = Strings.formatTwo(String.valueOf(message), arguments);
        for (Logger logger : loggerList)
        {
            logger.logger(this, level, msg, throwable);
        }
    }
    
    /**
     * 判断日志级别是否开启
     * 
     * @param level 日志级别
     * @return =true表示开启,=false表示未开启
     */
    private boolean allow(int level)
    {
        return logService.isLoggerEnabled(this, level);
    }
}
