/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * https://www.zhiqim.com/gitcan/zhiqim/zhiqim_kernel.htm
 *
 * This file is part of [zhiqim_kernel].
 * 
 * [zhiqim_kernel] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_kernel] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_kernel].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.kernel.control;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import org.zhiqim.kernel.paging.PageBuilder;
import org.zhiqim.kernel.paging.PageResult;
import org.zhiqim.kernel.util.Asserts;

/**
 * 消息队列,对LinkedList进行封装，增加对象锁对队列进行控制：<br><br>
 *  1)当列表中没有数据时阻塞到队列中读取消息的线程
 *  2)当列表中新增数据时通知所有正在阻塞的线程
 *  
 * @version v1.0.0 @author zouzhigang 2014-2-27 新建与整理
 */
public class Queue<E> implements Serializable
{
    private static final long serialVersionUID = 1L;
    
    private final ThreadLock lock = new ThreadLock();
    private final LinkedList<E> list = new LinkedList<E>();

    /** 投递对象 */
    public void post(E e)
    {
        post(e, false);
    }
    
    /** 投递对象，支持指定优化，优先的投放到前面 */
    public void post(E e, boolean priority)
    {
        if (e == null)
            return;
        
        synchronized (list)
        {
            if (priority)
                list.addFirst(e);
            else
                list.add(e);
        }
        
        lock.unlock();
    }
    
    /**
     * 接收消息队列中的对象，如果当前没有则阻塞线程等待指定时间，
     * 如果等待时间内有通知则读取，等待结束再尝试一次，未读到返回NULL
     * 
     * @param timeout 等待时间，以毫秒为单位
     * @return 如果有消息，则返回消息，如果没有则返回null
     */
    public E accept(int timeout)
    {
        synchronized (list)
        {
            if (!list.isEmpty())
                return list.removeFirst();
        }

        lock.lock(timeout);

        synchronized (list)
        {
            if (list.isEmpty())
                return null;
            return list.removeFirst();
        }
    }

    /** 对象是否存在对列中 */
    public boolean contains(E e)
    {
        return list.contains(e);
    }
    
    /** 当前对列是否为空 */
    public boolean isEmpty()
    {
        return list.isEmpty();
    }
    
    /** 清除队列消息 */
    public void clear()
    {
        synchronized (list)
        {
            list.clear();
        }
    }
    
    /** 获取当前队列长度 */
    public int size()
    {
        return list.size();
    }
    
    /** 获取消息分页列表，注意根据index读取有可能有重复，仅供参考 */
    public PageResult<E> list(int pageNo, int pageSize)
    {
        Asserts.as((pageNo > 0 && pageSize > 0)?null:"页码必须大于0");
        
        int formIndex = (pageNo-1) * pageSize;
        int toIndex = pageNo * pageSize;
        
        List<E> li = new ArrayList<>(pageSize);
        for (int i=formIndex;i<toIndex;i++)
        {
            if (i >= list.size())
                break;
            
            try{li.add(list.get(i));}catch(IndexOutOfBoundsException e){break;}
        }

        return PageBuilder.newResult(list.size(), pageNo, pageSize, Collections.unmodifiableList(li));
    }
}