/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙标记语言（zhiqim_zml）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_zml.htm
 *
 * This file is part of [zhiqim_zml].
 * 
 * [zhiqim_zml] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_zml] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_zml].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.zml;

import java.util.Iterator;

import org.zhiqim.kernel.extend.HashMapSV;
import org.zhiqim.kernel.schedule.Interval;
import org.zhiqim.kernel.schedule.Task;

/**
 * ZML缓存
 *
 * @version v1.0.0 @author zouzhigang 2017-11-21 新建与整理
 */
public class ZmlCache implements Task
{
    private final HashMapSV<Zml> cache = new HashMapSV<>();
    
    private int maxIdleTime;//最长空闲时长，单位秒
    private int maxKeepTime;//最长保持时长，单位秒
    
    public ZmlCache(int maxIdleTime, int maxKeepTime)
    {
        this.maxIdleTime = maxIdleTime;
        this.maxKeepTime = maxKeepTime;
        
        //每15分钟检查一次
        Interval.shedule(this, 15 * 60 * 1000);
    }
    
    public Zml get(String path)
    {
        synchronized (cache)
        {
            Zml zml = cache.get(path);
            if (zml != null)
                zml.setLastAccessed();
            
            return zml;
        }
    }
    
    public void put(String path, Zml zml)
    {
        synchronized (cache)
        {
            if (cache.containsKey(path))
                cache.remove(path);
            
            cache.put(path, zml);
        }
    }
    
    public void remove(String path)
    {
        synchronized (cache)
        {
            cache.remove(path);
        }
    }

    @Override
    public void execute()
    {
        boolean hasMaxIdleTime = maxIdleTime > 0;
        boolean hasMaxKeepTime = maxKeepTime > 0;
        
        if (!hasMaxIdleTime && !hasMaxKeepTime)
            return;
        
        long lastKeepTimeMs = System.currentTimeMillis() - maxKeepTime * 1000;
        long lastIdleTimeMs = System.currentTimeMillis() - maxIdleTime * 1000;
        
        synchronized (cache)
        {
            for (Iterator<Zml> it=cache.values().iterator();it.hasNext();)
            {
                Zml zml = it.next();
                long lastAccessed = zml.getLastAccessed();
                if ((hasMaxKeepTime && lastAccessed < lastKeepTimeMs) 
                  || (hasMaxIdleTime && lastAccessed < lastIdleTimeMs))
                {//超时删除
                    it.remove();
                }
            }
        }
    }
    
    public int getMaxIdleTime()
    {
        return maxIdleTime;
    }

    public int getMaxKeepTime()
    {
        return maxKeepTime;
    }

    public void setMaxIdleTime(int maxIdleTime)
    {
        this.maxIdleTime = maxIdleTime;
    }
    
    public void setMaxKeepTime(int maxKeepTime)
    {
        this.maxKeepTime = maxKeepTime;
    }
}
