/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm.executor;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import org.zhiqim.kernel.extend.HashMapSO;
import org.zhiqim.kernel.extend.LinkedMapSO;
import org.zhiqim.kernel.extend.MapSO;
import org.zhiqim.kernel.extend.MapSS;
import org.zhiqim.kernel.extend.TreeMapSO;
import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.util.Sqls;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.orm.ORMException;
import org.zhiqim.orm.ORMServer;
import org.zhiqim.orm.datasource.ZDBClose;
import org.zhiqim.orm.datasource.ZResultSet;
import org.zhiqim.orm.datasource.ZStatement;
import org.zhiqim.orm.dbo.CallParam;
import org.zhiqim.orm.dbo.CallResult;
import org.zhiqim.orm.dbo.Dbo;

public class CallExecutor
{
    public final Log log = LogFactory.getLog(QueryExecutor.class);
    public final Log cSqlLog = LogFactory.getLog("call.sql.log");
    
    /**
     * 执行executeQuery处理，基本类型不需要判断是否有通配符##,sql语句用?代替
     * 
     * @param server        ORM服务
     * @param conn          数据库连接
     * @param sql           ZSQL语句
     * @param paramList     参数表
     * @param resultList    结果集表
     * @param replaceMap    可替换表
     * @throws ORMException 映射异常
     * @throws SQLException 数据映射异常
     */
    public void execute(ORMServer server, Connection conn, String sql, List<CallParam> paramList, List<CallResult> resultList, MapSS replaceMap) throws ORMException, SQLException
    {
        CallableStatement cstmt = null;
        ResultSet rst = null;

        try
        {
            //1.替换可替换符$$
            sql = Sqls.formatReplaceMap(sql, replaceMap);
            cstmt = conn.prepareCall(sql);
          
            //设置IN/INOUT参数
            for (int i=0;i<paramList.size();i++)
            {
                CallParam param = paramList.get(i);
                if (param.isInWrite())
                    ZStatement.setParameter(server, cstmt, i+1, param.value());
                else
                    cstmt.registerOutParameter(i+1, param.typeSQL());
            }

            if (server.isQuerySqlLog())
            {//查询可以打印SQL
                cSqlLog.info(sql);
            }

            //执行
            cstmt.execute();
            
            //读取OUT/INOUT参数
            for (int i=0;i<paramList.size();i++)
            {
                CallParam param = paramList.get(i);
                if (param.isOutRead())
                    param.value(cstmt, i+1);
            }
            
            //读取结果集
            if (Validates.isNotEmpty(resultList))
            {
                int index = 0;
                
                do
                {
                    rst = cstmt.getResultSet();
                    if (rst == null)
                        break;
                    
                    CallResult callResult = resultList.get(index);
                    Class<?> resultClass = callResult.getResultClass();
                    
                    if (server.isResultBase(resultClass))
                        callResult.setResultList(ZResultSet.parseToBase(rst, resultClass));
                    else if (resultClass == Map.class || resultClass == MapSO.class || resultClass == HashMapSO.class || resultClass == LinkedMapSO.class || resultClass == TreeMapSO.class)
                        callResult.setResultList(ZResultSet.parseToMap(rst, resultClass));
                    else if (server.isDbo(resultClass))
                    {
                        Dbo resultDbo = server.getDbo(resultClass);
                        callResult.setResultList(ZResultSet.parseToDbo(rst, resultDbo));
                    }
                    else
                    {//解析不了
                        throw new ORMException("调用[函数/存储过程]时不支持的结果集类["+resultClass+"]");
                    }
                    
                    ZDBClose.close(rst);
                    rst = null;
                    index++;
                }
                while (cstmt.getMoreResults() && resultList.size() > index);
            }
        }
        catch (SQLException e)
        {
            log.error("CallExecutor[SQLException][%s][sql=%s]", e, e.getMessage(), sql);
            throw e;
        }
        catch (Exception e)
        {
            log.error("CallExecutor[Exception][%s]", e, e.getMessage());
            throw new ORMException(e.getMessage(), e);
        }
        finally
        {
            ZDBClose.close(rst, cstmt);
        }
    }
}
