/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm.dbo.defined;

import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;

import org.zhiqim.kernel.extend.HashMapSS;
import org.zhiqim.kernel.extend.LinkedMapSV;
import org.zhiqim.kernel.extend.MapSO;
import org.zhiqim.kernel.extend.MapSS;
import org.zhiqim.kernel.util.Arrays;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Lists;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.orm.ORMConstants;
import org.zhiqim.orm.ORMException;
import org.zhiqim.orm.ZTable;
import org.zhiqim.orm.dbo.Dbo;
import org.zhiqim.orm.dbo.Selector;

/**
 * 视图对象，配置格式如：<br><br>
 * <view name="com.zhiqim.dbo_ex.UserEX" table="USER, USER_EX"><br>
 *     <field name="userId" table="USER" column="USER_ID" type="long"/><br>
 *     <field name="userName" table="USER" column="USER_NAME" type="String"/><br>
 *     <field name="userDesc" table="USER_EX" column="USER_DESC" type="String"/><br>
 *     <join type="EQUAL" lTable="USER" lColumn="USER_ID" rTable="USER_EX" rColumn="USER_ID"/><br>
 * </view><br><br>
 * 自联表注意使用别名配置，如下配置格式：
 * <view name="com.zhiqim.dbo_ex.UserEX" table="USER u, USER e"><br>
 *     <field name="userId" table="u" column="USER_ID" type="long"/><br>
 *     <field name="userName" table="u" column="USER_NAME" type="String"/><br>
 *     <field name="userDesc" table="e" column="USER_DESC" type="String"/><br>
 *     <join type="EQUAL" lTable="u" lColumn="USER_ID" rTable="e" rColumn="USER_ID"/><br>
 * </view><br><br>
 * 用于ZView时，传入_View类，映射成数据库多表，和根据数据库表列组装成_View对象
 * 
 * 注：_View不支持通过column找到_ViewField
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class _View implements ORMConstants, Dbo
{
    private String name;
    private String table;
    
    private LinkedMapSV<_ViewField> fieldMap = new LinkedMapSV<>();
    private List<_ViewJoin> joinList = new ArrayList<>();
    
    public _View(String name, String table)
    {
        this.name = name;
        this.table = table;
    }
    
    public void addField(String name, String table, String column, int type)
    {
        _ViewField field = new _ViewField(name, table, column.toUpperCase(), type);
        fieldMap.put(name, field);
    }
    
    public void addJoin(String type, String lTable, String lColumn, String rTable, String rColumn, String rValue)
    {
        if (Validates.isNotEmpty(rColumn))
            rColumn = rColumn.toUpperCase();
        
        _ViewJoin join = new _ViewJoin(type, lTable, lColumn.toUpperCase(), rTable, rColumn, rValue);
        joinList.add(join);
    }
    
    public String toString()
    {
        StringBuilder strb = new StringBuilder(_FOUR_).append("<view")
            .append(" name=").append(_DOUBLE_QUOTE_).append(name).append(_DOUBLE_QUOTE_)
            .append(" table=").append(_DOUBLE_QUOTE_).append(table).append(_DOUBLE_QUOTE_)
            .append(">").append(_BR_);
        
        for (_ViewField field : fieldMap.values())
        {
            strb.append(_FOUR_).append(_FOUR_).append(field).append(_BR_);
        }
        
        strb.append(_BR_);
        for (_ViewJoin where : joinList)
        {
            strb.append(_FOUR_).append(_FOUR_).append(where).append(_BR_);
        }
        
        strb.append(_FOUR_).append("</view>");
        return strb.toString();
    }
    
    public String getName()
    {
        return name;
    }
    
    public List<_ViewJoin> getWhereList()
    {
        return joinList;
    }
    
    public String[] getTableArr()
    {
        return Arrays.toStringArray(table, ",");
    }
    
    public List<String> getTableNoAliasList()
    {
        ArrayList<String> list = new ArrayList<>();
        
        String[] tableArr = getTableArr();
        for (String table : tableArr)
        {
            String[] alias = Arrays.toStringArray(table, " ");
            if (alias.length == 2)
                list.add(alias[0]);
            else
                list.add(table);
        }
        
        return Lists.trim(list);
    }
    
    /** 可能表名有别名，再找一次 */
    public String getTableName(String tableName)
    {
        String[] tableArr = getTableArr();
        for (String table : tableArr)
        {
            String[] alias = Arrays.toStringArray(table, " ");
            if (alias.length != 2)
                continue;
            
            if (tableName.equals(alias[1]))
                return alias[0];
        }
        
        return tableName;
    }
    
    public List<_ViewField> getFieldList(String fields)
    {
        if (Validates.isEmptyBlank(fields))//为空表示全部
            return new ArrayList<_ViewField>(fieldMap.values());
        
        List<String> fieldNameList = Lists.toStringList(fields);
        List<_ViewField> fieldList = new ArrayList<_ViewField>();
        for (_ViewField field : fieldMap.values())
        {
            if (fieldNameList.contains(field.getField()))
                fieldList.add(field);
        }
        
        Asserts.as(!fieldList.isEmpty()?null:"["+fields+"]未找到一个匹配的字段");
        return fieldList;
    }
    
    /** 根据名称查字段get方法 */
    public String getGetMethod(String name)
    {
        _ViewField field = fieldMap.get(name);
        if (field == null)
            return null;
        
        return field.getGetMethod();
    }
    
    /** 根据名称查字段set方法 */
    public String getSetMethod(String name)
    {
        _ViewField field = fieldMap.get(name);
        if (field == null)
            return null;
        
        return field.getSetMethod();
    }
    
    public _ViewField getField(String field)
    {
        return fieldMap.get(field);
    }
    
    public String getColumn(String field)
    {
        _ViewField _viewField = fieldMap.get(field);
        if (_viewField == null)
            return null;
        
        return _viewField.getTable() + "." + _viewField.getColumn();
    }
    
    public boolean hasField(String field)
    {
        return fieldMap.containsKey(field);
    }
    
    /** 是否有对应的所有字段名 */
    public boolean hasAllField(String[] fieldArr)
    {
        for (String field : fieldArr)
        {
            if (!fieldMap.containsKey(field))
                return false;
        }
        return true;
    }
    
    /** 缺省字段 */
    public String getDefaultFieldSQL(String fields)
    {
        StringBuilder strb = new StringBuilder();
        
        List<_ViewField> fieldList = getFieldList(fields);
        //第一个
        _ViewField field = fieldList.get(0);
        strb.append(field.getTable()).append(".").append(field.getColumn()).append(" as ").append(field.getField());
        //剩下的前加逗号
        for (int i=1;i<fieldList.size();i++)
        {
            field = fieldList.get(i);
            strb.append(", ").append(field.getTable()).append(".").append(field.getColumn()).append(" as ").append(field.getField());
        }
        
        return strb.toString();
    }
    
    /** 是否有相等的连接条件 */
    public boolean hasEqualJoin()
    {
        for (_ViewJoin join : joinList)
        {
            if (JOIN_EQUAL.equals(join.getType()))
                return true;
        }
        
        return false;
    }
    
    /** 获取字段列表按长度排序 */
    public _ViewField[] getFieldColumnListOrderByLen()
    {
        _ViewField[] array = new _ViewField[fieldMap.size()];
        int index = 0;
        for (_ViewField field : fieldMap.values())
        {
            array[index++] = field;
        }
        
        for (int i=0;i<array.length;i++)
        {
            for (int j=i+1;j<array.length;j++)
            {
                if (array[i].getField().length() < array[j].getField().length())
                {//冒泡交换
                    _ViewField v = array[i];
                    array[i] = array[j];
                    array[j] = v;
                }
            }
        }
        
        return array;
    }
    
    /** 获取相等的连接SQL */
    public String getEqualJoinSQL()
    {
        StringBuilder strb = new StringBuilder();
        boolean isFirst = true;
        for (_ViewJoin join : joinList)
        {
            if (!JOIN_EQUAL.equals(join.getType()))
                continue;
            
            if (!isFirst)
                strb.append(" and ");
            else
            {
                strb.append(" where ");
                isFirst = false;
            }
            
            strb.append(join.getLTable()).append(".").append(join.getLColumn());
            strb.append(" = ");
            
            if (Validates.isEmptyBlank(join.getRValue()))
                strb.append(join.getRTable()).append(".").append(join.getRColumn());
            else
                strb.append("'").append(join.getRValue()).append("'");
                
        }

        return strb.toString();
    }
    
    /** 连接表 */
    public String getJoinTable(ZTable fTable, Selector selector, MapSO paramMap) throws ORMException
    {
        //去除tables 中的空格，但不能去除别名的空格 如 BUS_USER a, BUS_USER b 只能改为 BUS_USER a,BUS_USER b
        MapSS tableos = new HashMapSS();
        MapSS tables = new HashMapSS();
        MapSS tableas = new HashMapSS();
        
        String[] tableArray = getTableArr();
        for (int i=0;i<tableArray.length;i++)
        {
            String table = tableArray[i].trim();
            int ind = table.indexOf(" ");
            if (ind == -1)
            {
                tables.put(table, table);
                tableos.put(table, table);
                tableas.put(table, table);
            }
            else
            {//如果有别名，KEY=别名
                String alias = table.substring(ind+1).trim();
                tables.put(alias, table);
                tableos.put(alias, table);
                tableas.put(alias, table.substring(0, ind));
            }
        }
        
        //找到左右联的表，从dbo.getTable() 中滤除，加上 join 和 on       
        MapSS joinMap = new HashMapSS();
        
        for (int i=0;i<joinList.size();i++)
        {
            _ViewJoin join = joinList.get(i);
            if (JOIN_EQUAL.equals(join.getType()))
                continue;//去除相等

            String joinType = JOIN_LEFT.equals(join.getType())?" left join ":" right join ";
            String lTable = join.getLTable();
            String laTable = tables.get(lTable);//如果左表为别名，laTable为左表名表全，如BUS_USER a
            String rTable = join.getRTable();
            String rrTable = tableas.get(rTable);   //右表真实表名
            String raTable = tables.get(rTable);    //如果右表为别名，raTable为右表名表全，如BUS_USER b
            
            //如果有别名，传入别名，或者传入表类名，得到带条件的join表
            raTable = selector.getJoinWhereSQL(fTable, rTable, rrTable, raTable, paramMap);
            
            StringBuilder joinSql = null;
            
            if (joinMap.containsKey(lTable + "," + rTable))
            {//表示已有一个左右联，这里要求相同表联时，左右联一致
                joinSql = new StringBuilder(joinMap.get(lTable + "," + rTable));
                joinSql.append(" and ");
            }
            else
            {//新的连接
                joinSql = new StringBuilder();
                joinSql.append(laTable).append(joinType).append(raTable).append(" on ");
            }

            joinSql.append(join.getLTable()).append(".").append(join.getLColumn());
            joinSql.append(" = ");
            
            if (Validates.isEmptyBlank(join.getRValue()))
                joinSql.append(join.getRTable()).append(".").append(join.getRColumn()).append(" ");
            else
                joinSql.append("'").append(join.getRValue()).append("' ");
            
            joinMap.put(lTable + "," + rTable, joinSql.toString());
        }

        for (Entry<String, String> entry : joinMap.entrySet())
        {
            String tableKey = entry.getKey();
            String joinSql = entry.getValue();
            
            String[] tableArr = tableKey.split(",");
            String lTable = tableArr[0];
            String rTable = tableArr[1];
            
            tables.remove(rTable);//去除右表
            
            String laTable = tableos.get(lTable);
            String lTableValue = tables.get(lTable);
            if (lTableValue == null)
                throw new ORMException("暂不支持嵌套左右连接,请调整为以一张表为主表进行左右连接");
            
            lTableValue = lTableValue.replaceAll(lTable+"\\.", "T1A2B3L4E5.");
            lTableValue = lTableValue.replaceAll(laTable, joinSql);//改进左表,如果左表有别名，则连别名一起改进
            lTableValue = lTableValue.replaceAll("T1A2B3L4E5.", lTable+"\\.");
            tables.put(lTable, lTableValue);
        }

        StringBuilder strb = new StringBuilder();
        for (String tab : tables.values()){
            strb.append(tab).append(",");
        }
        if (strb.length() > 0){
            strb.setLength(strb.length()-1);
        }
        return strb.toString();
    }
}
