/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm.datasource;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;

/**
 * 数据库连接测试类 <br>
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class ZConnectionTester
{
    private static final Log log = LogFactory.getLog("database.log");
    
    public final static int CONNECTION_IS_OKAY       =  0;
    public final static int CONNECTION_IS_INVALID    = -1;
    public final static int DATABASE_IS_INVALID      = -8;
    
    private final static Set<String> INVALID_DB_STATES;
    static
    {
        Set<String> temp = new HashSet<String>();
        temp.add("08001"); //SQL State "Unable to connect to data source"
        temp.add("08007"); //SQL State "Connection failure during transaction"
        temp.add(null);    //SQL State "Io 异常: Connection reset by peer: socket write error"
        
        // MySql appently uses this state to indicate a stale, expired
        // connection when the database is fine, so we'll not presume
        // this SQL state signals an invalid database.
        temp.add("08S01"); //SQL State "Communication link failure"
    
        INVALID_DB_STATES = Collections.unmodifiableSet(temp);
    }

    private ZDataSource dataSource = null;
    private ZConnection testconnection = null;
    
    public ZConnectionTester(ZDataSource dataSource)
    {
        this.dataSource = dataSource;
    }
    
    /**
     * 判断数据库是否断开
     * 
     * @return boolean =true表示断开,=false表示正常
     */
    public boolean isDbBreak()
    {
        if (testconnection == null)
        {
            testconnection = dataSource.newProxyConnection("测试");
            if (testconnection == null)
                return true;
        }
        
        if (testconnection.isClosed() || testconnection.isOvertimeOrCompletedCount())
        {
            testconnection.shutdown();
            testconnection = null;
            testconnection = dataSource.newProxyConnection("测试");
            if (testconnection == null)
                return true;
        }  
        
        int status = isConnectionAvailable(testconnection);
        if (status == CONNECTION_IS_OKAY)
            return false;
        
        testconnection.shutdown();
        testconnection = null;
        return true;
    }
    
    /**
     * 检查数据库连接
     * 
     * @param connection 数据库连接
     * @return =0,正常,=-1,该连接异常,=-8,数据库异常
     */
    public int isConnectionAvailable(Connection connection)
    {
        if (connection == null)
            return CONNECTION_IS_INVALID;
            
        ResultSet rst = null;
        
        try
        {
            DatabaseMetaData metaData = connection.getMetaData();

            rst = metaData.getTables(null, null, "PROBABLYNOT", new String[]{"TABLE"});
            return CONNECTION_IS_OKAY;
        }
        catch (SQLException e)
        {
            String state = e.getSQLState();
            if (INVALID_DB_STATES.contains(state))
            {
                log.error("检查数据库连接是否可用时:数据库不可用");
                return DATABASE_IS_INVALID;
            }
            else
            {
                log.error("检查数据库连接是否可用时:连接不可用");
                return CONNECTION_IS_INVALID; 
            }
        }
        finally
        {
            ZDBClose.close(rst);
        }
    }
    
    /** 关闭测试连接 */
    public void shutdown()
    {
        if (testconnection != null)
        {
            testconnection.shutdown();
            testconnection = null;
        }
    }
}
