/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map.Entry;

import org.zhiqim.kernel.MultiInstancer;
import org.zhiqim.kernel.extend.HashMapSO;
import org.zhiqim.kernel.extend.LinkedMapSO;
import org.zhiqim.kernel.extend.MapSO;
import org.zhiqim.kernel.extend.MapSS;
import org.zhiqim.kernel.paging.PageBuilder;
import org.zhiqim.kernel.paging.PageResult;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Classes;
import org.zhiqim.orm.annotation.AnTable;
import org.zhiqim.orm.annotation.AnTableField;
import org.zhiqim.orm.annotation.AnView;
import org.zhiqim.orm.annotation.AnViewField;
import org.zhiqim.orm.annotation.AnViewJoin;
import org.zhiqim.orm.annotation.AnViewJoinValue;
import org.zhiqim.orm.dbo.Selector;
import org.zhiqim.orm.dbo.defined._Table;
import org.zhiqim.orm.dbo.defined._TableField;
import org.zhiqim.orm.dbo.defined._View;

/**
 * 视图类与数据库表多表映射调用，支持(count,sum,item,list,page)<br><br>
 * count,       统计视图类对应的关联表数目<br>
 * sum,         总计视图类对应的关联表数目<br>
 * item,        查询视图类对应的关联表一条数据，根据主键，或根据条件排序取第一条<br>
 * list,        查询视图类对应的关联表列表，或根据条件排序<br>
 * pag,         分页查询视图类对应的关联表列表，指定分页页码和页数目<br>
 * 
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 * @version v1.2.0 @author zouzhigang 2016-6-27 修改原类名FRHELExcutor为ZViewExecutor，正式使用视图名称
 */
class ZViewImplement extends MultiInstancer implements ZView, ORMConstants
{
    private ORMServer server;
    private ZSQL zSQL;
    private ZTabler zTabler;
    private ZTable zTable;
    private HashSet<String> existViewList;
    private HashMap<Class<?>, _View> viewMap;
    
    public ZViewImplement(ORMServer server)
    {
        this.server = server;
        this.zSQL = server.sql();
        this.zTabler = server.tabler();
        this.zTable = server.table();
        this.existViewList = new HashSet<String>();
        this.viewMap = new HashMap<>();
    }
    
    /**
     * 通过视图类获取视图对象
     * 
     * @param clazz     视图类
     * @return          视图对象
     */
    public _View getView(Class<?> clazz)
    {
        _View _view = viewMap.get(clazz);
        if (_view != null)
            return _view;
        
        AnView anView = clazz.getAnnotation(AnView.class);
        if (anView == null)
            return null;
        
        String table = anView.value();
        _view = new _View(clazz.getName(), table);
        
        //1.增加连接
        AnViewJoin join = clazz.getAnnotation(AnViewJoin.class);
        if (join != null)
        {
            AnViewJoinValue[] values = join.value();
            for (AnViewJoinValue value : values)
            {
                _view.addJoin(value.type(), value.lTable(), value.lColumn(), value.rTable(), value.rColumn(), value.rValue());
            }
        }
        //2.增加字段
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields)
        {
            if (Classes.isStaticTransientField(field))
                continue;
            
            String fieldName = field.getName();
            
            AnViewField vfield = field.getAnnotation(AnViewField.class);
            if (vfield != null)
            {//从表中找字段
                _Table _table = zTable.getTableByTableName(_view.getTableName(vfield.table()));
                Asserts.assertNotNull(_table, ORMI18n.ormViewNotFountTable2, clazz.getName(), vfield.table());
                
                _TableField _field = _table.getField(vfield.column());
                Asserts.assertNotNull(_field, ORMI18n.ormViewNotFoundField2, clazz.getName(), fieldName);
                
                _view.addField(fieldName, vfield.table(), vfield.column(), _field.getType());
            }
        }
        
        //3.增加表字段
        Class<?> superClass = clazz.getSuperclass();
        if (superClass != Object.class)
        {
            fields = superClass.getDeclaredFields();
            for (Field field : fields)
            {
                if (Classes.isStaticTransientField(field))
                    continue;
                
                String fieldName = field.getName();
                
                AnTableField tfield = field.getAnnotation(AnTableField.class);
                if (tfield != null)
                {
                    Class<?> fpclass = field.getDeclaringClass();
                    AnTable t = fpclass.getAnnotation(AnTable.class);
                    if (t != null)
                    {
                        int type = ORMType.getColumnTypeMaybeLength(tfield.type());
                        _view.addField(fieldName, t.table(), tfield.column(), type);
                    }
                }
            }
        }
        
        viewMap.put(clazz, _view);
        return _view;
    }
    
    /*****************************************************/
    //count 查询数目
    /*****************************************************/
    
    /**
     * 查询数目，查全视图
     * 
     * @param clazz         视图类
     * @return              返回数目
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int count(Class<?> clazz)throws ORMException, SQLException
    {
        return count(clazz, new Selector());
    }
    
    /**
     * 查询数目，根据条件、可替换表查询
     * 
     * @param clazz         视图类
     * @param selector      对象查询器
     * @return              返回数目
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int count(Class<?> clazz, Selector selector)throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZView[count][selector]不允许为NULL]");
        
        _View _view = getView(clazz);
        if (_view == null)
            throw new ORMException("ZView[count]["+clazz.getName()+"]未找到相应的配置]");
        
        //检查表是否存在，当前仅支持MYSQL
        createJoinTable(_view, selector.getReplaceMap());
        
        MapSO paramMap = new HashMapSO();
        StringBuilder sql = new StringBuilder("select count(*) from ");
        sql.append(_view.getJoinTable(zTable, selector, paramMap));         //连接表
        sql.append(_view.getEqualJoinSQL());                                //连接条件
        sql.append(selector.getWhereSQL(_view, paramMap));                  //自定义条件
        
        List<Integer> list = zSQL.executeQuery(sql.toString(), Integer.class, paramMap, selector.getReplaceMap());
        return list.get(0);
    }
    
    /*****************************************************/
    //sum 计算总和
    /*****************************************************/
    
    /**
     * 计算总和，全视图求和
     * 
     * @param clazz         视图类
     * @param field         视图字段
     * @return              返回总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long sum(Class<?> clazz, String field) throws ORMException, SQLException
    {
        return sum(clazz, field, new Selector());
    }

    /**
     * 计算总和，根据条件视图中求和
     * 
     * @param clazz         视图类
     * @param field         视图字段
     * @param selector      查询器
     * @return              返回总和(sum)
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long sum(Class<?> clazz, String field, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZView[sum][Selector]不允许为NULL]");
        
        _View view = getView(clazz);
        if (view == null)
            throw new ORMException("ZView[sum]["+clazz.getName()+"]未找到相应的配置]");
        
        String column = view.getColumn(field);
        if (column == null)
            throw new ORMException("ZView[sum]["+clazz.getName()+"]配置中未找到["+field+"]字段");
        
        //检查表是否存在，当前仅支持MYSQL
        createJoinTable(view, selector.getReplaceMap());
        
        MapSO paramMap = new HashMapSO();
        StringBuilder sql = new StringBuilder("select sum(").append(column).append(")");
        sql.append(" from ").append(view.getJoinTable(zTable, selector, paramMap));         //连接表
        sql.append(view.getEqualJoinSQL());                                                 //连接条件
        sql.append(selector==null?"":selector.getWhereSQL(view, paramMap));                 //自定义条件
        
        List<Long> list = zSQL.executeQuery(sql.toString(), Long.class, paramMap, selector.getReplaceMap());
        return list.isEmpty()?0:list.get(0);
    }
    
    /**
     * 计算多个总和，全视图中多个求和
     * 
     * @param clazz         视图类
     * @param fieldArr      多个视图字段
     * @return              返回计算多个总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long[] sum(Class<?> clazz, String[] fieldArr) throws ORMException, SQLException
    {
        return sum(clazz, fieldArr, new Selector());
    }
    
    /**
     * 计算多个总和，根据条件视图中多个求和
     * 
     * @param clazz         视图类
     * @param fieldArr      多个视图字段
     * @param selector      查询器
     * @return              返回计算多个总和
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public long[] sum(Class<?> clazz, String[] fieldArr, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZView[sum][selector]不允许为NULL]");
        
        if (fieldArr == null || fieldArr.length < 1)
            throw new ORMException("ZView[sum][fieldArr]不能为空]");
        
        _View view = getView(clazz);
        if (view == null)
            throw new ORMException("ZView[sum]["+clazz.getName()+"]未找到相应的配置]");
        
        if (!view.hasAllField(fieldArr))
            throw new ORMException("ZView[sum]["+Arrays.toString(fieldArr)+"]有未找到的字段");
        
        //检查表是否存在，当前仅支持MYSQL
        createJoinTable(view, selector.getReplaceMap());
        
        StringBuilder sql = new StringBuilder("select ");
        //第一个
        String column = view.getColumn(fieldArr[0]);
        sql.append(" sum(").append(column).append(") as ").append(fieldArr[0]);
        //后面的
        for (int i=1;i<fieldArr.length;i++)
        {
            column = view.getColumn(fieldArr[i]);
            sql.append(", sum(").append(column).append(") as ").append(fieldArr[i]);
        }
        
        HashMapSO paramMap = new HashMapSO();
        sql.append(" from ").append(view.getJoinTable(zTable, selector, paramMap))
            .append(view.getEqualJoinSQL())
            .append(selector==null?"":selector.getWhereSQL(view, paramMap));
        
        List<LinkedMapSO> list = zSQL.executeQuery(sql.toString(), LinkedMapSO.class, paramMap, selector==null?null:selector.getReplaceMap());
        if (list.isEmpty())
            return new long[fieldArr.length];

        LinkedMapSO item = list.get(0);
        long[] valueArr = new long[fieldArr.length];
        for (int i=0;i<fieldArr.length;i++)
        {//默认为数值型，需要转化为long
            BigDecimal value = (BigDecimal)item.get(fieldArr[i]);
            if (value == null)
                valueArr[i] = 0;
            else
                valueArr[i] = value.longValue();
        }
        return valueArr;
    }
    
    /*****************************************************/
    //item 查询一条数据
    /*****************************************************/
    
    /**
     * 查询一个视图对象,全视图中取第一个
     * 
     * @param clazz         视图类
     * @return              返回视图对象
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> clazz) throws ORMException, SQLException
    {
        return item(clazz, new Selector());
    }
    
    /**
     * 查询一个视图对象,可排序取第一个
     * 
     * @param clazz         视图类
     * @param selector      查询器
     * @return              返回视图对象
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> T item(Class<T> clazz, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZView[item][Selector]不允许为NULL]");
        
        _View _view = getView(clazz);
        if (_view == null)
            throw new ORMException("ZView[item]["+clazz.getName()+"]未找到相应的配置]");
        
        //检查表是否存在，当前仅支持MYSQL
        createJoinTable(_view, selector.getReplaceMap());
        
        //组装SQL
        MapSO paramMap = new HashMapSO();
        String whereSQL = selector.getWhereSQL(_view, paramMap);
        String orderbySQL = selector.getOrderbySQL(_view);
        String groupbySQL = selector.getGroupbySQL(_view);

        //加上一行标识,组装SQL
        StringBuilder sql = new StringBuilder();
        switch (server.getDatabaseType())
        {
        case Z_ORM_MYSQL_INT:
        case Z_ORM_SQLITE_INT:
            sql.append("select ").append(_view.getDefaultFieldSQL(selector.getFields())).append(" from ").append(_view.getJoinTable(zTable, selector, paramMap))
                .append(_view.getEqualJoinSQL()).append(whereSQL).append(orderbySQL).append(groupbySQL).append(" limit 1");
            break;
        case Z_ORM_MSSQL_INT:
        case Z_ORM_HSQL_INT:
            sql.append("select top 1 ").append(_view.getDefaultFieldSQL(selector.getFields())).append(" from ").append(_view.getJoinTable(zTable, selector, paramMap))
                .append(_view.getEqualJoinSQL()).append(whereSQL).append(orderbySQL).append(groupbySQL);
            break;
        case Z_ORM_ORACLE_INT:
            sql.append("select * from (select ").append(_view.getDefaultFieldSQL(selector.getFields())).append(" from ").append(_view.getJoinTable(zTable, selector, paramMap))
                .append(_view.getEqualJoinSQL()).append(whereSQL).append(orderbySQL).append(groupbySQL)
                .append(") where rownum = 1 ");
            break;
        default:
            throw new ORMException("不支持的数据库类型["+server.getDatabaseType()+"]");
        }
        
        List<T> list = zSQL.executeQuery(sql.toString(), clazz, paramMap);
        return list.isEmpty()?null:list.get(0);
    }
    
    /*****************************************************/
    //list 查询列表
    /*****************************************************/
    
    /**
     * 查询列表，查全视图
     * 
     * @param clazz         视图类
     * @return              视图对象列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> clazz) throws ORMException, SQLException
    {
        return list(clazz, new Selector());
    }
    
    /**
     * 查询列表，根据查询器
     * 
     * @param clazz         视图类
     * @param selector      查询器
     * @return              视图对象列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> list(Class<T> clazz, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZView[list][selector]不允许为NULL]");
        
        _View _view = getView(clazz);
        if (_view == null)
            throw new ORMException("ZView[list]["+clazz.getName()+"]未找到相应的配置]");
        
        //检查表是否存在，当前仅支持MYSQL
        createJoinTable(_view, selector.getReplaceMap());
        
        //组装SQL
        MapSO paramMap = new HashMapSO();
        StringBuilder sql = new StringBuilder("select ").append(_view.getDefaultFieldSQL(selector.getFields()));    //字段
        sql.append(" from ").append(_view.getJoinTable(zTable, selector, paramMap));                                  //连接表
        sql.append(_view.getEqualJoinSQL());                                                                        //连接条件
        sql.append(selector.getWhereSQL(_view, paramMap));                                                          //自定义条件
        sql.append(selector.getOrderbySQL(_view));                                                                  //自定义排序
        sql.append(selector.getGroupbySQL(_view));                                                                  //自定义分组
        
        return zSQL.executeQuery(sql.toString(), clazz, paramMap, selector.getReplaceMap());
    }
    
    /*****************************************************/
    //page 分页显示
    /*****************************************************/
    
    /**
     * 查询视图分页信息，查全视图
     * 
     * @param clazz         视图类
     * @param pageNo        页码
     * @param pageSize      页数
     * @return              分页信息,包括总页数,页码,页数和查询的记录
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> PageResult<T> page(Class<T> clazz, int pageNo, int pageSize) throws ORMException, SQLException
    {
        return page(clazz, pageNo, pageSize, new Selector());
    }
    
    /**
     * 查询视图分页信息
     * 
     * @param clazz         视图类
     * @param pageNo        页码
     * @param pageSize      页数
     * @param selector      查询器
     * @return              分页信息,包括总页数,页码,页数和查询的记录
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> PageResult<T> page(Class<T> clazz, int pageNo, int pageSize, Selector selector) throws ORMException, SQLException
    {
        if (selector == null)
            throw new ORMException("ZView[page][selector]不允许为NULL]");
        
        _View _view = getView(clazz);
        if (_view == null)
            throw new ORMException("ZView[page]["+clazz.getName()+"]未找到相应的配置]");
        
        //检查表是否存在，当前仅支持MYSQL
        createJoinTable(_view, selector.getReplaceMap());
        
        if (pageNo < 1) pageNo = 1;
        if (pageSize < 1) pageSize = 10;
        
        //先查数目
        int totalRecord = count(clazz, selector);
        if (totalRecord == 0)
            return PageBuilder.newResult(pageNo, pageSize);
        
        //检查where
        int minNum = (pageNo-1) * pageSize + 1;
        int maxNum = pageNo * pageSize;
        MapSO paramMap = new HashMapSO();
        paramMap.put("minNum", new Integer(minNum));
        paramMap.put("maxNum", new Integer(maxNum));
        paramMap.put("minSize", new Integer(minNum - 1));
        paramMap.put("pageSize", new Integer(pageSize));
        
        //组装SQL
        String whereSQL = selector.getWhereSQL(_view, paramMap);
        String orderbySQL = selector.getOrderbySQL(_view);
        //String orderbyReSQL = selector.getOrderbyReverseSQL(_view);MSSQL使用临时表方案暂未用反转
        String groupbySQL = selector.getGroupbySQL(_view);
        
        StringBuilder innerTableSQL = new StringBuilder(" ( select ");
        innerTableSQL.append(_view.getDefaultFieldSQL(selector.getFields()));                   //字段
        innerTableSQL.append(" from ").append(_view.getJoinTable(zTable, selector, paramMap));  //连接表
        innerTableSQL.append(_view.getEqualJoinSQL());                                          //连接条件
        innerTableSQL.append(whereSQL);                                                         //自定义条件
        innerTableSQL.append(orderbySQL);                                                       //自定义排序
        innerTableSQL.append(groupbySQL);                                                       //自定义分组
        innerTableSQL.append(" ) ");

        String sql = server.getPolicy().toPageViewSQL(innerTableSQL, maxNum, pageNo);
        
        //执行SQL
        List<T> list = zSQL.executeQuery(sql, clazz, paramMap, selector.getReplaceMap());
        return PageBuilder.newResult(totalRecord, pageNo, pageSize, list);
    }
    
    /** 检查创建表 */
    private void createJoinTable(_View _view, MapSS replaceMap) throws SQLException, ORMException
    {
        String viewName = _view.getName();
        if (replaceMap != null)
        {//如果替换表不为null则加进去
            StringBuilder strb = new StringBuilder(viewName);
            for (Entry<String, String> entry : replaceMap.entrySet())
            {
                strb.append("_").append(entry.getKey()).append("_").append(entry.getValue());
            }
            viewName = strb.toString();
        }
        
        //检查表是否存在，当前仅支持MYSQL
        if (existViewList.contains(viewName))
            return;
        
        List<String> tableList = _view.getTableNoAliasList();
        for (String table : tableList)
        {
            _Table _table = zTable.getTableByTableName(table);
            if (_table == null)
                throw new ORMException("ZView[createJoinTable]["+_view.getName()+"]未找到相应["+table+"]的配置");
            
            zTabler.create(_table, replaceMap);
        }
        
        existViewList.add(viewName);
    }
}
