/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙数据库映射（zhiqim_orm）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_orm.htm
 *
 * This file is part of [zhiqim_orm].
 * 
 * [zhiqim_orm] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_orm] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_orm].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.orm;

import java.sql.SQLException;
import java.util.List;

import org.zhiqim.kernel.MultiInstance;
import org.zhiqim.kernel.annotation.AnAlias;
import org.zhiqim.kernel.extend.LinkedMapSO;
import org.zhiqim.kernel.extend.MapSS;
import org.zhiqim.orm.dbo.CallParam;
import org.zhiqim.orm.dbo.CallResult;

/**
 * 标准SQL调用，支持(executeUpdate, executeBatch, executeQuery)<br><br>
 * executeUpdate,       执行更新SQL语句，支持insert,replace,update,delete等<br>
 * executeBatch,        批量执行更新SQL语句，支持insert,replace,update,delete等<br>
 * executeQuery,        执行查询SQL语句，<br>
 *                      返回结果可以是boolean,byte,short,int,long,String,Timestamp,double,byte[]等9种类型<br>
 *                      
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
@AnAlias("ZSQL")
public interface ZSQL extends MultiInstance
{
    /************************************************************************************************/
    //select @@IDENTITY 查自增ID
    /************************************************************************************************/
    
    /**
     * 查询当前自增ID值
     * 
     * @return              整型
     * @throws SQLException 数据库异常
     */
    public int identityInt() throws SQLException;
    
    /**
     * 查询当前自增ID值
     * 
     * @return              长整型
     * @throws SQLException 数据库异常
     */
    public long identityLong() throws SQLException;
    
    /************************************************************************************************/
    //exist 查询是否存在表或字段
    /************************************************************************************************/
    
    /**
     * 查询表是否存在
     * 
     * @param table         表名
     * @return              boolean存在是否
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public boolean existTable(String table) throws ORMException, SQLException;
    
    /************************************************************************************************/
    //execute
    /************************************************************************************************/
    
    /**
     * 执行标准SQL语句，不带参数，如DDL,DML语句
     * 
     * @param sql           SQL语句 
     * @throws SQLException 数据库异常
     */
    public void execute(String sql) throws SQLException;
    
    /**
     * 批执行标准SQL语句，不带参数，如DDL,DML语句
     * 
     * @param sqlList       SQL语句列表
     * @throws SQLException 数据库异常
     */
    public void execute(List<String> sqlList) throws SQLException;
    
    /************************************************************************************************/
    //executeUpdate
    /************************************************************************************************/
    
    /**
     * 无参数executeUpdate执行
     * 
     * @param id                SQL配置编号
     * @return int              执行条数
     * @throws ORMException     映射异常
     * @throws SQLException     数据库异常
     */
    public int executeUpdateId(String id) throws ORMException, SQLException;
    
    /**
     * executeUpdate执行
     * 
     * @param id                SQL配置编号
     * @param param             参数,需和executeUpdate配置对应，否则将报错
     * @return int              执行条数
     * @throws ORMException     映射异常
     * @throws SQLException     数据库异常
     */
    public int executeUpdateId(String id, Object param) throws ORMException, SQLException;
    
    /**
     * executeUpdate执行
     * 
     * @param id                SQL配置编号
     * @param param             参数,需和executeUpdate配置对应，否则将报错
     * @param replaceMap        字符替换表
     * @return int              执行条数
     * @throws ORMException     映射异常
     * @throws SQLException     数据库异常
     */
    public int executeUpdateId(String id, Object param, MapSS replaceMap) throws ORMException, SQLException;
    
    /**
     * executeUpdate执行
     * 
     * @param sql           SQL语句
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int executeUpdate(String sql) throws ORMException, SQLException;
    
    /**
     * executeUpdate执行
     * 
     * @param sql           SQL语句
     * @param param         参数,需和update配置对应，否则将报错
     * @return int          执行条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int executeUpdate(String sql, Object param) throws ORMException, SQLException;
    
    /**
     * executeUpdate执行
     * 
     * @param sql           SQL语句
     * @param param         参数,需和update配置对应，否则将报错
     * @param replaceMap    字符替换表
     * @return int          执行条数
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int executeUpdate(String sql, Object param, MapSS replaceMap) throws ORMException, SQLException;
    
    /************************************************************************************************/
    //executeBatch
    /************************************************************************************************/
    
    /**
     * executeBatch执行
     * 
     * @param id            SQL配置编号
     * @param paramList     参数列表
     * @return int[]        执行结果数组
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int[] executeBatchId(String id, List<?> paramList) throws ORMException, SQLException;
    
    /**
     * executeBatch执行
     * 
     * @param id            SQL配置编号
     * @param paramList     参数列表
     * @param replaceMap    字符替换表
     * @return int[]        执行结果数组
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int[] executeBatchId(String id, List<?> paramList, MapSS replaceMap) throws ORMException, SQLException;
    
    /**
     * executeBatch执行
     * 
     * @param sql           SQL语句
     * @param paramList     参数列表
     * @return              int[] 执行结果数组
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int[] executeBatch(String sql, List<?> paramList) throws ORMException, SQLException;
    
    /**
     * executeBatch执行
     * 
     * @param sql           SQL语句
     * @param paramList     参数列表
     * @param replaceMap    字符替换表
     * @return int[]        执行结果数组
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public int[] executeBatch(String sql, List<?> paramList, MapSS replaceMap) throws ORMException, SQLException;
    
    /************************************************************************************************/
    //executeQuery
    /************************************************************************************************/
    
    /**
     * executeQuery执行
     * 
     * @param id            SQL配置编号
     * @param resultClass   结果类结构
     * @return              结果集
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> executeQueryId(String id, Class<T> resultClass) throws ORMException, SQLException;
    
    /**
     * executeQuery执行
     * 
     * @param id            SQL配置编号
     * @param resultClass   结果类结构
     * @param param         参数
     * @return              结果集
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> executeQueryId(String id, Class<T> resultClass, Object param) throws ORMException, SQLException;
    
    /**
     * executeQuery执行
     * 
     * @param id            SQL配置编号
     * @param resultClass   结果类结构
     * @param param         参数
     * @param replaceMap    字符替换表
     * @return              结果集
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> executeQueryId(String id, Class<T> resultClass, Object param, MapSS replaceMap) throws ORMException, SQLException;
    
    /**
     * executeQuery执行
     * 
     * @param sql           SQL语句
     * @return              结果集
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public List<LinkedMapSO> executeQuery(String sql) throws ORMException, SQLException;
    
    /**
     * executeQuery执行
     * 
     * @param sql           SQL语句
     * @param resultClass   结果类结构
     * @return              结果集
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> executeQuery(String sql, Class<T> resultClass) throws ORMException, SQLException;
    
    /**
     * executeQuery执行
     * 
     * @param sql           SQL语句
     * @param resultClass   结果类结构
     * @param param         参数
     * @return              结果集
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> executeQuery(String sql, Class<T> resultClass, Object param) throws ORMException, SQLException;
    
    /**
     * executeQuery执行
     * 
     * @param sql           SQL语句
     * @param resultClass   结果类结构
     * @param param         参数
     * @param replaceMap    字符替换表
     * @return              结果集
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> executeQuery(String sql, Class<T> resultClass, Object param, MapSS replaceMap) throws ORMException, SQLException;
    
    /************************************************************************************************/
    //call(函数&存储过程调用)
    /************************************************************************************************/
    
    /**
     * call 函数&存储过程执行，无结果集，一般用于函数或count/sum等简单的存储过程
     * 
     * @param sql           SQL语句
     * @param paramList     参数列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void call(String sql, List<CallParam> paramList) throws ORMException, SQLException;
    
    
    /**
     * call 函数&存储过程执行，支持替换表，无结果集，一般用于函数或count/sum等简单的存储过程
     * 
     * @param sql           SQL语句
     * @param paramList     参数列表
     * @param replaceMap    字符替换表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void call(String sql, List<CallParam> paramList, MapSS replaceMap) throws ORMException, SQLException;
    
    /**
     * call 存储过程执行，返回一个结果集
     * 
     * @param sql           SQL语句
     * @param paramList     参数列表
     * @param replaceMap    字符替换表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public <T> List<T> call(String sql, List<CallParam> paramList, Class<T> resultClass) throws ORMException, SQLException;
    
    /**
     * call 函数&存储过程执行，支持多结果集
     * 
     * @param sql           SQL语句
     * @param paramList     参数列表
     * @param resultList    结果集列表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void call(String sql, List<CallParam> paramList, List<CallResult> resultList) throws ORMException, SQLException;
    
    /**
     * call 函数&存储过程执行，支持多结果集
     * 
     * @param sql           SQL语句
     * @param paramList     参数列表
     * @param resultList    结果集列表
     * @param replaceMap    字符替换表
     * @throws ORMException 映射异常
     * @throws SQLException 数据库异常
     */
    public void call(String sql, List<CallParam> paramList, List<CallResult> resultList, MapSS replaceMap) throws ORMException, SQLException;
}
