/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.sessioner;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.httpd.HttpSession;
import org.zhiqim.httpd.HttpSessionManager;
import org.zhiqim.httpd.HttpSessionUser;
import org.zhiqim.kernel.extend.HashMapSO;
import org.zhiqim.kernel.json.Jsons;
import org.zhiqim.kernel.util.DateTimes;
import org.zhiqim.kernel.util.Ints;
import org.zhiqim.kernel.util.Longs;
import org.zhiqim.kernel.util.Strings;

/**
 * HTTP会话对象：
 * 1.sessionId 会话唯一标识,用于SessionManager管理
 * 2.created 会话创建时间
 * 3.accessed 会话访问时间
 * 4.sessionName 会话名称,用于业务上作为唯一标识
 * 5.sessionUser 会话对象,用于业务对象模型,作为存储
 * 6.attributes 属性列表
 * 持久化时,6个值都将保存到持久化层
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class HashSession implements HttpSession
{
    //会话管理器
    private transient HttpContext context;
    private transient HttpSessionManager sessionManager;
    
    //定义浸入式更换sessionId的保留老编号，由HttpResponse时清理
    private String oldSessionId;
    
    private String sessionId;
    private long created;
    private long accessed;
    private int maxInactiveInterval;
    private String sessionIp;
    private String userAgent;

    private HttpSessionUser sessionUser;
    private HashMapSO attributes;
    
    public HashSession()
    {
        this.created = System.currentTimeMillis();
        this.accessed = created;
        this.attributes = new HashMapSO();
    }
    
    public HashSession setSessionManager(HttpSessionManager sessionManager)
    {
        this.sessionManager = sessionManager;
        this.context = sessionManager.getContext();
        this.maxInactiveInterval = sessionManager.getSessionTimeout();
        return this;
    }
    
    public HashSession setSessionId(String sessionId)
    {
        this.sessionId = sessionId;
        return this;
    }
    
    public void setSessionIp(String sessionIp)
    {
        this.sessionIp = sessionIp;
    }
    
    public void setUserAgent(String userAgent)
    {
        this.userAgent = userAgent;
    }
    
    /*********************************************************/
    //会话参数设置（最后访问时间、IP地址等） & 会话主动失效调用
    /*********************************************************/
    
    public void updateLastAccessed()
    {
        this.accessed = System.currentTimeMillis();
    }
    
    public void updateInfo(String sessionIp, String userAgent)
    {
        this.accessed = System.currentTimeMillis();
        this.sessionIp = sessionIp;
        this.userAgent = userAgent;
    }
    
    public void invalidate()
    {
        if (sessionUser != null)
        {//解除绑定
            sessionUser.setSession(null);
            sessionUser = null;
        }
        
        if (attributes != null)
        {//清除属性
            attributes.clear();
            attributes = null;
        }
        
        if (oldSessionId != null)
        {//清除旧会话
            sessionManager.invalidateSession(oldSessionId);
            oldSessionId = null;
        }
        
        if (sessionId != null)
        {//清除当前会话
            sessionManager.invalidateSession(sessionId);
            sessionId = null;
        }
    }
    
    /******************************************************************************************/
    //用于业务浸入式更改sessionId（注：谨慎使用），HttpResponse会oldSessionId进行处理
    /******************************************************************************************/
    
    /** 用于业务修改ID值，请谨慎使用 */
    public void setNewSessionId(String newSessionId)
    {
        if (newSessionId.equals(this.sessionId))
            return;
        
        this.oldSessionId = this.sessionId;
        this.sessionId = newSessionId;
    }
    
    public String getOldSessionId()
    {
        return oldSessionId;
    }
    
    public void clearOldSessionId()
    {
        this.oldSessionId = null;
    }
    
    /**********************************************/
    //获取参数判断值
    /**********************************************/
    
    public HttpContext getContext()
    {
        return context;
    }
    
    public String getId()
    {
        return sessionId;
    }
    
    public boolean isNew()
    {
        return created == accessed;
    }
    
    public String getSessionIp()
    {
        return sessionIp;
    }
    
    public String getUserAgent()
    {
        return this.userAgent;
    }
    
    public int getMaxInactiveInterval()
    {
        return maxInactiveInterval;
    }
    
    public long getCreated()
    {
        return created;
    }
    
    public String getCreateTime()
    {
        return DateTimes.toDateTimeString(created);
    }
    
    public long getLastAccessed()
    {
        return accessed;
    }
    
    public String getLastAccessTime()
    {
        return DateTimes.toDateTimeString(accessed);
    }
    
    public long getRemainSecond()
    {
        long remain = System.currentTimeMillis() - getLastAccessed();
        return maxInactiveInterval - remain / 1000;
    }
    
    public long getRemainMinute()
    {
        return getRemainSecond() / 60;
    }
    
    public long[] getRemainTime()
    {
        long time = getRemainSecond();
        long day = time / 86400;
        long hour = time % 86400 / 3600;
        long minute = time % 3600 / 60;
        long second = time % 60;
        
        return new long[]{day, hour, minute, second};
    }
    
    public String getRemainTimeDesc()
    {
        long[] times = getRemainTime();
        
        StringBuilder strb = new StringBuilder();
        if (times[0] > 0)
            strb.append(times[0]).append("天");
        if (times[1] > 0)
            strb.append(times[1]).append("小时");
        if (times[2] > 0)
            strb.append(times[2]).append("分");
        strb.append(times[3]).append("秒");
        
        return strb.toString();
    }
    
    /**********************************************/
    //SessionUser相关
    /**********************************************/    

    /** 绑定会话用户信息 */
    public void bindSessionUser(HttpSessionUser sessionUser)
    {
        if (this.sessionUser != null && !this.sessionUser.equals(sessionUser))
        {//清理原关系
            this.sessionUser.setSession(null);
            this.sessionUser = null;
        }
        
        if (sessionUser != null)
        {//增加新关系
            this.sessionUser = sessionUser;
            this.sessionUser.setSession(this);
        }
    }

    /** 解绑会话用户信息 */
    public void unbindSessionUser()
    {
        if (this.sessionUser == null)
            return;
        
        this.sessionUser.setSession(null);
        this.sessionUser = null;
    }
    
    /** 判断会话中是否有用户信息 */
    public boolean hasSessionUser()
    {
        return sessionUser != null;
    }
    
    /** 获取会话用户信息中用户类，多个时抛异常，多会话用户时请谨慎调用 */
    public HttpSessionUser getSessionUser()
    {
        return sessionUser;
    }
    
    /** 判断会话中是否有用户信息且指定的用户类 */
    public <T extends HttpSessionUser> boolean hasSessionUser(Class<T> clazz)
    {
        return sessionUser != null && sessionUser.getClass() == clazz;
    }
    
    /** 获取会话用户信息且指定的用户类 */
    @SuppressWarnings("unchecked")
    public <T extends HttpSessionUser> T getSessionUser(Class<T> clazz)
    {
        return sessionUser == null?null:sessionUser.getClass() == clazz?(T)sessionUser:null;
    }
    
    /** 获取会话用户名称，多个时抛异常，多会话用户时请谨慎调用 */
    public String getSessionName()
    {
        return sessionUser == null?null:sessionUser.getSessionName();
    }
    
    /**********************************************/
    //读写关联到session中attribute列表
    /**********************************************/
    
    public HashMapSO getAttributes()
    {
        return attributes;
    }
    
    public Object removeAttribute(String name)
    {
        return attributes.remove(name);
    }
    
    public void setAttribute(String name, Object value)
    {
        attributes.put(name, value);
    }
    
    public boolean hasAttribute(String name)
    {
        return attributes.containsKey(name);
    }
    
    public Object getAttribute(String name)
    {
        return attributes.get(name);
    }
    
    public Object getAttribute(String name, Object defaultValue)
    {
        Object value = attributes.get(name);
        return (value == null)?defaultValue:value;
    }
    
    public String getAttributeString(String name)
    {
        return getAttributeString(name, null);
    }
    
    public String getAttributeString(String name, String defaultValue)
    {
        Object value = getAttribute(name);
        return (value == null)?defaultValue:Strings.trim(String.valueOf(value));
    }
    
    public int getAttributeInt(String name)
    {
        return getAttributeInt(name, -1);
    }
    
    public int getAttributeInt(String name, int defaultValue)
    {
        return Ints.toInt(attributes.get(name), defaultValue);
    }
    
    public long getAttributeLong(String name)
    {
        return getAttributeLong(name, -1);
    }
    
    public long getAttributeLong(String name, long defaultValue)
    {
        return Longs.toLong(attributes.get(name), defaultValue);
    }
    
    public boolean getAttributeBoolean(String name, boolean defaultValue)
    {
        Object value = attributes.get(name);
        if (value == null)
            return defaultValue;
        else if (value instanceof Boolean)
            return (Boolean)value;
        else if (value instanceof String)
            return Boolean.parseBoolean((String)value);
        else
            return defaultValue;
    }
    
    /**********************************************/
    //清理和转换成字符串
    /**********************************************/
    
    public String toString()
    {
        return Jsons.toString(this);
    }
}
