/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.nio;

import java.nio.ByteBuffer;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.httpd.HttpInputStream;
import org.zhiqim.httpd.HttpWebsocketConnectionAbs;
import org.zhiqim.httpd.HttpWebsocketMessage;
import org.zhiqim.httpd.HttpWebsocketMessageAsync;

/**
 * HttpWebsocket连接线程
 *
 * @version v1.0.0 @author zouzhigang 2017-6-9 新建与整理
 */
public class HttpNioWebsocketConnection extends HttpWebsocketConnectionAbs
{
    private HttpNioConnection conn;
    private HttpInputStream input;
    private HttpWebsocketMessageAsync data;//最后一个消息
    
    //SSL属性
    private boolean hasSsl;
    private HttpNioSslBuffer buffer;
    
    public HttpNioWebsocketConnection(HttpNioConnection conn, HttpContext context, String protocol, String sessionId)
    {
        super(context, protocol, sessionId);
        
        this.conn = conn;
        this.input = new HttpInputStream();
        
        if (conn instanceof HttpNioSslConnection)
        {//对SSL属性初始化
            HttpNioSslConnection sslconn = (HttpNioSslConnection)conn;
            this.hasSsl = true;
            this.buffer = new HttpNioSslBuffer(sslconn.getSSLEngine());
        }
    }
    
    public HttpNioConnection getConnection()
    {
        return conn;
    }
    
    public synchronized void open()
    {
        if (isRunning)
            return;
        
        isRunning = true;
    }
    
    public synchronized void close()
    {
        if (!isRunning)
            return;
        
        isRunning = false;
        if (conn != null)
        {
            conn.close();
            conn = null;
        }
        
        super.close();
    }
    
    public synchronized void exception(Throwable e)
    {
        if (!isRunning)
            return;
        
        isRunning = false;
        if (conn != null)
        {
            conn.close();
            conn = null;
        }
        
        super.exception(e);
    }
    
    /***********************************************************************************/
    //异步接受数据
    /***********************************************************************************/
    
    public int execute(ByteBuffer buf)
    {
        int result = buffer(buf);
        if (result <= 0)
        {//NIO返回值为1，NIO-SSL可能返回-1/0/1，针对-1/0的返回连接处理
            return result;
        }
        
        if (data == null || data.isParsed())
            data = new HttpWebsocketMessageAsync();
        
        boolean readNext = parse();
        while (readNext && input.available() > 0)
        {
            data = new HttpWebsocketMessageAsync();
            readNext = parse();
        }
        
        return 1;
    }
    
    /** 解析缓冲流 */
    private int buffer(ByteBuffer buf)
    {
        int result = 1;
        if (hasSsl)
            result = buffer.parseBuffer(buf, input);
        else
            input.addBuffer(buf);
        return result;
    }
    
    /** 解析数据流 */
    private boolean parse()
    {
        try
        {
            if (!data.parse(input))
            {//未完成解析
                return false;
            }
            
            if (data.isPong())
            {//心跳响应
                return true;
            }
            
            if (data.isPing())
            {//心跳请求
                HttpWebsocketMessage message = new HttpWebsocketMessage().buildPong();
                send(message.toBytes());
                return true;
            }
            
            if (data.isClose())
            {//接收到退出
                close();
                return false;
            }
            
            if (data.isText())
            {//接收到消息
                context.getSessionManager().updateLastAccessed(sessionId);
                if (websocket != null){
                    websocket.onMessage(this, data.getText());
                }
                return true;
            }
            
            //TODO 其他的后续再处理
            return true;
        }
        catch (Throwable e)
        {
            exception(e);
            return false;
        }
    }
}
