/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.nio;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.StandardSocketOptions;
import java.nio.channels.SelectionKey;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;

import org.zhiqim.httpd.HttpServer;
import org.zhiqim.httpd.HttpSslConfig;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Threads;

/**
 * HTTPS非阻塞式监听器<br>
 * 1、继承自HttpNioSslListener
 * 2、重写getScheme方法，返回结果HTTPS
 * 3、线程开启前创建SSLContext
 * 4、重写newConnection增加SSLEngine参数
 * 
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 * @version v1.5.0 @author zouzhigang 2018-11-24 修改为完成从HttpNioListener继承，优化重写的方法
 */
public class HttpNioSslListener extends HttpNioListener
{
    private final HttpSslConfig ssl;
    private SSLContext sslContext;
    
    /** 构造函数，初始新建SSL配置类 */
    public HttpNioSslListener(HttpServer server)
    {
        super(server);
        this.ssl = new HttpSslConfig();
    }
    
    @Override /** 协议 */
    public String getScheme()
    {
        return _HTTPS_;
    }
    
    @Override /** 线程开启前 */
    protected void openBeforeInner()
    {
        try
        {
            //优先SSL参数（五项，其中Keystore两项必须）
            ssl.initParam(listenerGroup);
            sslContext = ssl.createSSLContext();
        }
        catch(Exception e)
        {
            throw Asserts.exception(e);
        }
    }
    
    /** 接受客户端连接 */
    protected void onAccept(SelectionKey key)
    {
        ServerSocketChannel server = (ServerSocketChannel)key.channel();
        
        HttpNioSslConnection conn = null;
        SocketChannel socket = null;
        
        try
        {
            socket = server.accept();
            socket.setOption(StandardSocketOptions.TCP_NODELAY, true);
            socket.setOption(StandardSocketOptions.SO_LINGER, soLingerS);
            socket.setOption(StandardSocketOptions.SO_KEEPALIVE, true);
            socket.setOption(StandardSocketOptions.SO_RCVBUF, soRecvBuf);
            socket.setOption(StandardSocketOptions.SO_SNDBUF, soSendBuf);
            
            //先默认阻塞，并检查握手
            conn = newConnection(socket);
            if (!ssl.doHandshake(conn))
            {//SSL的握手失败
                conn.close();
                return;
            }
            
            //握手成功置为非阻塞，注册读事件
            socket.configureBlocking(false);
            socket.register(selector, SelectionKey.OP_READ, add(conn));
        }
        catch (Throwable e)
        {
            if (conn != null)
            {//连接创建成功异常，关连接
                conn.close();
            }
            else if (socket != null)
            {//Socket异常，关Socket
                Threads.closeIgnoreException(socket);
            }
        }
    }
    
    @Override /** 新建封装的连接对象 */
    protected HttpNioSslConnection newConnection(SocketChannel socket) throws IOException
    {
        InetAddress remoteAddress = ((InetSocketAddress)socket.getRemoteAddress()).getAddress();
        
        SSLEngine engine = sslContext.createSSLEngine();
        engine.setUseClientMode(ssl.isClientAuth());
        engine.setWantClientAuth(ssl.isClientAuth());
        engine.setEnableSessionCreation(true);
        
        return new HttpNioSslConnection(this, socket, remoteAddress, engine);
    }
}
