/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.context.service;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.httpd.HttpExecutor;
import org.zhiqim.httpd.HttpRequest;
import org.zhiqim.httpd.HttpResponse;
import org.zhiqim.httpd.context.ZmlContextConstants;
import org.zhiqim.httpd.util.Responses;

import org.zhiqim.kernel.logging.Log;
import org.zhiqim.kernel.logging.LogFactory;
import org.zhiqim.kernel.util.Files;
import org.zhiqim.kernel.util.Ids;
import org.zhiqim.kernel.util.Streams;
import org.zhiqim.kernel.util.Strings;
import org.zhiqim.kernel.util.Urls;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.kernel.util.codes.Base64;

/**
 * 上传服务，用于固定访问地址为/service/upload，实际处理在forward中
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class UploadService implements HttpExecutor, ZmlContextConstants
{
    private static final Log log = LogFactory.getLog(UploadService.class);
    private static final String[] FILE_DIR_NOT_FORMAT = {".", "\\", "?", ":", "*", "\'", "\"", "<", ">", "|"};
    
    private String uploadRootDir;
    
    @Override
    public boolean isMatch(String pathInContext)
    {
        return _PATH_SERVICE_UPLOAD_.equals(pathInContext);
    }
    
    @Override
    public void handle(HttpRequest request, HttpResponse response) throws IOException
    {
        String sessionName = Strings.toString(request.getSessionName(), "guest");
        
        /**********************************************************/
        //第一步，判断上传参数
        /**********************************************************/
        int mode = request.getHeaderInt(_X_RESPONSED_MODE_, 0);
        String fileDir = request.getHeader(_X_UPLOAD_FILE_DIR_);
        String fileName = request.getHeader(_X_UPLOAD_FILE_NAME_);
        int contentLength = request.getContentLength();
        
        fileDir = Urls.decodeUTF8(fileDir);
        fileDir = Strings.removeStartsWith(fileDir, "/");
        fileDir = Strings.removeEndsWith(fileDir, "/");
        fileName = Urls.decodeUTF8(fileName);
        
        if (mode < 0 || mode > 1                                            //响应模式，默认0
            || Validates.isStrContainStrArr(fileDir, FILE_DIR_NOT_FORMAT)   //文件目录不支持.\:*?'"<>|
            || Validates.isEmptyBlank(fileName)                             //文件名不能为空
            || contentLength <= 0)                                          //内容不能为空
        {
            log.error("[Upload][%s][参数不正确]", sessionName);
            response.sendError(_412_PRECONDITION_FAILED_);
            return;
        }
        
        /**********************************************************/
        //第二步，跳转优先
        /**********************************************************/
        HttpContext context = request.getContext();
        String forward = (String)context.getAttribute(_SERV_UPLOAD_FORWARD_);
        
        if (Validates.isNotEmpty(forward))
        {//跳转优先，成功则跳转到对应的连接
            try{request.forwardTo(forward);}
            catch(Exception e){try{response.sendHeader(_500_INTERNAL_SERVER_ERROR_);}catch(Exception e2){}}
            return;
        }
        
        /**********************************************************/
        //第三步，判断文件PATH和URL，生成并返回
        //1.文件根目录配置必须
        //2.对文件目录进行检查，并生成URL对应的BASE64编码的目录名
        //3.写入文件并返回URL
        /**********************************************************/
        
        if (uploadRootDir == null)
        {//1.文件根目录配置必须
            String rootDir = (String)context.getAttribute(_SERV_UPLOAD_ROOT_DIR_);
            if (Validates.isEmptyBlank(rootDir) || !Files.mkDirectory(rootDir))
            {
                log.error("[Upload][%s][配置处理器处理时必须配置上传根目录][%s]", sessionName, _SERV_UPLOAD_ROOT_DIR_);
                response.sendError(_403_FORBIDDEN_);
                return;
            }
            
            uploadRootDir = Files.toLinuxPath(new File(rootDir).getCanonicalPath());
            uploadRootDir = Strings.addEndsWith(uploadRootDir, "/");
        }
        
        //2.对文件目录进行分析并创建
        String fileDirPath = null;
        if (Validates.isEmptyBlank(fileDir))    
            fileDirPath = uploadRootDir;
        else
        {
            fileDirPath = uploadRootDir + fileDir + "/";
            if (!Files.mkDirectory(fileDirPath))
            {
                log.error("[Upload][%s][上传的文件目录无法创建][%s]", sessionName, request.getHeader(_X_UPLOAD_FILE_DIR_));
                response.sendError(_403_FORBIDDEN_);
                return;
            }
        }
        
        String fileExt = Files.getFileExt(fileName);
        fileName  = Ids.uuid() + Files.fixFileExt(fileExt);
        
        String filePath = fileDirPath + fileName;
        
        String fileUrl = null;
        if (Validates.isEmptyBlank(fileDir))
            fileUrl = _PATH_UPLOAD_PREFIX_ + fileName;
        else
        {
            String fileDirUrl = Base64.encodeFileUrlUTF8(fileDir);
            fileUrl = _PATH_UPLOAD_PREFIX_ + fileDirUrl + "/" + fileName;
        }
        
        //写入文件
        byte[] data = Streams.getBytes(request.getInputStream(), contentLength);
        Files.writeReturnSize(filePath, data);
        
        //返回URL
        log.info("[Upload][%s][上传完成][%s]", sessionName, fileName);
        HashMap<String, String> headerMap = new HashMap<String, String>();
        headerMap.put(_X_UPLOAD_FILE_URL_, fileUrl);
        
        Responses.doReturnMessage(response, mode, headerMap);
    }
}
