/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.context;

import java.util.ArrayList;
import java.util.List;

import org.zhiqim.httpd.HttpContext;
import org.zhiqim.httpd.HttpWebsocket;
import org.zhiqim.httpd.HttpWebsocketConnection;
import org.zhiqim.httpd.HttpWebsocketManager;
import org.zhiqim.kernel.extend.HashMapSV;
import org.zhiqim.kernel.schedule.Interval;
import org.zhiqim.kernel.schedule.Task;
import org.zhiqim.kernel.util.Lists;
import org.zhiqim.kernel.util.Validates;

/**
 * ZML对Websocket的管理器
 *
 * @version v1.0.0 @author Zouzhigang 2018-9-27 新建与整理
 */
public class ZmlWebsocketManager implements HttpWebsocketManager, Task
{
    private final ZmlContext context;
    private final List<HttpWebsocketConnection> connList;
    private final HashMapSV<HttpWebsocket> websocketMap;
    private Interval interval;
    
    public ZmlWebsocketManager(ZmlContext context)
    {
        this.context = context;
        this.connList = new ArrayList<>();
        this.websocketMap = new HashMapSV<>(); 
    }
    
    public void open()
    {////定义29秒检查一次，保证最小1分钟清除即可，太频繁也不大合适
        this.interval = Interval.shedule(this, 29 * 1000);
    }

    public void close()
    {
        if (this.interval != null)
        {
            this.interval.close();
            this.interval = null;
        }
    }
    
    public HttpContext getContext()
    {
        return context;
    }
    
    @Override
    public HttpWebsocket getWebsocket(String protocol)
    {
        if (Validates.isEmptyBlank(protocol))
            return null;
        
        if (websocketMap.containsKey(protocol))
            return websocketMap.get(protocol);
        
        HttpWebsocket websocket = context.getConfig().getWebsocket(protocol);
        websocketMap.put(protocol, websocket);
        return websocket;
    }

    /** 未设置协议名的情况 */
    public List<HttpWebsocketConnection> get(String protocol)
    {
        if (protocol == null)
            return connList;
        
        List<HttpWebsocketConnection> sList = new ArrayList<>();
        for (HttpWebsocketConnection conn : connList)
        {
            if (protocol.equals(conn.getProtocol()))
                sList.add(conn);
        }
        
        return sList;
    }
    
    /** 指定协议名的情况 */
    public List<HttpWebsocketConnection> get(String protocol, String sessionId)
    {
        if (sessionId == null)
            return get(protocol);
        
        List<HttpWebsocketConnection> sList = new ArrayList<>();
        if (protocol == null)
        {
            for (HttpWebsocketConnection conn : connList)
            {
                if (sessionId.equals(conn.getSessionId()))
                    sList.add(conn);
            }
        }
        else
        {
            for (HttpWebsocketConnection conn : connList)
            {
                if (protocol.equals(conn.getProtocol()) && sessionId.equals(conn.getSessionId()))
                    sList.add(conn);
            }
        }
        
        return Lists.trim(sList);
    }

    /*******************************************************************************/
    //增加 & 删除
    /*******************************************************************************/
    
    /** 由HttpWebsocketService增加连接关联到会话 */
    public void add(HttpWebsocketConnection conn)
    {
        synchronized (connList)
        {
            connList.add(conn);
        }
    }
    
    /** 从连接中收到结束取消关联会话 */
    public void remove(HttpWebsocketConnection conn)
    {
        synchronized (connList)
        {
            connList.remove(conn);
        }
    }
    
    /*******************************************************************************/
    //定时检查
    /*******************************************************************************/
    
    @Override
    public void execute()
    {
        //1.查出需要关闭的连接
        List<HttpWebsocketConnection> closes = new ArrayList<>();
        for (HttpWebsocketConnection conn : connList)
        {
            if (!context.getSessionManager().hasSession(conn.getSessionId()))
            {//用户超时，则关闭连接
                closes.add(conn);
            }
            else if (!conn.sendPing())
            {//发送PING失败，异常连接关闭
                closes.add(conn);
            }
        }
        
        //2.逐个关闭连接
        for (HttpWebsocketConnection conn : closes)
        {
            conn.close();
        }
        closes.clear();
    }
}
