/*
 * 版权所有 (C) 2015 知启蒙(ZHIQIM) 保留所有权利。[遇见知启蒙，邂逅框架梦]
 * 
 * 知启蒙WEB容器（zhiqim_httpd）在LGPL3.0协议下开源：https://www.zhiqim.com/gitcan/zhiqim/zhiqim_httpd.htm
 *
 * This file is part of [zhiqim_httpd].
 * 
 * [zhiqim_httpd] is free software: you can redistribute
 * it and/or modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * [zhiqim_httpd] is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with [zhiqim_httpd].
 * If not, see <http://www.gnu.org/licenses/>.
 */
package org.zhiqim.httpd.bio;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;
import java.security.cert.X509Certificate;

import javax.net.ssl.SSLServerSocket;
import javax.net.ssl.SSLServerSocketFactory;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;

import org.zhiqim.httpd.HttpCertificate;
import org.zhiqim.httpd.HttpListener;
import org.zhiqim.httpd.HttpServer;
import org.zhiqim.httpd.HttpSslConfig;
import org.zhiqim.kernel.util.Asserts;
import org.zhiqim.kernel.util.Hexs;
import org.zhiqim.kernel.util.Sockets;
import org.zhiqim.kernel.util.Threads;
import org.zhiqim.kernel.util.Validates;
import org.zhiqim.kernel.util.codes.SSL;

/**
 * HTTPS阻塞式监听器
 *
 * @version v1.0.0 @author zouzhigang 2014-3-21 新建与整理
 */
public class HttpBioSslListener extends HttpListener
{
    private static final String SSL_SESSION_CACHE = HttpCertificate.class.getName();
    private static final String SSL_NULL_WITH_NULL_NULL = "SSL_NULL_WITH_NULL_NULL";
    
    private final HttpSslConfig ssl;
    private SSLServerSocket serverSocket;
    
    /** 构造函数 */
    public HttpBioSslListener(HttpServer server)
    {
        super(server);
        this.ssl = new HttpSslConfig();
    }
    
    @Override /** 协议 */
    public String getScheme()
    {
        return _HTTPS_;
    }

    @Override /** 线程启动前 */
    protected boolean openBefore()
    {
        try
        {
            //优先SSL参数（五项，其中Keystore两项必须）
            this.ssl.initParam(listenerGroup);
            
            //监听参数和线程池
            super.openBefore();
            
            //监听端口和守护线程
            SSLServerSocketFactory factory = ssl.createFactory();
            if (Validates.isEmpty(listenerIp))
                serverSocket = (SSLServerSocket)factory.createServerSocket(listenerPort);
            else
                serverSocket = (SSLServerSocket)factory.createServerSocket(listenerPort, 50, InetAddress.getByName(listenerIp));
            serverSocket.setReuseAddress(true);
            serverSocket.setNeedClientAuth(ssl.isClientAuth());
            serverSocket.setWantClientAuth(ssl.isClientAuth());
            return true;
        }
        catch(Exception e)
        {
            throw Asserts.exception(e);
        }
    }

    @Override /** 线程关闭之后 */
    protected void closeAfter()
    {
        //1.关闭监听
        if (serverSocket != null)
        {
            Threads.closeIgnoreException(serverSocket);
            serverSocket = null;
        }
        
        //2.关闭定时任务等父类定义
        super.closeAfter();
        
        //3.打印退出日志
        log.info(toString() + "退出");
    }

    @Override /** 线程持续运行 */
    protected void loop()
    {
        try
        {
            Socket socket = serverSocket.accept();
            Sockets.setOption(socket, true, soTimeoutMs, soLingerS, false);
            
            SSLSocket sslSocket = (SSLSocket)socket;
            SSLSession sslSession = sslSocket.getSession();
            String cipherSuite = sslSession.getCipherSuite();
            if (cipherSuite == null || SSL_NULL_WITH_NULL_NULL.equals(cipherSuite))
            {//HTTP访问，未使用SSL
                socket.close();
                return;
            }
            
            String sslSessionId = null;
            X509Certificate[] x509Certs = null;
            HttpCertificate sslSessionCache = (HttpCertificate)sslSession.getValue(SSL_SESSION_CACHE);
            if (sslSessionCache != null)
            {
                x509Certs = sslSessionCache.getCerts();
                sslSessionId = sslSessionCache.getSessionId();
            }
            else
            {
                x509Certs = SSL.getCertChain(sslSession);
                byte[] bytes = sslSession.getId();
                sslSessionId = Hexs.toHexString(bytes);
                sslSessionCache = new HttpCertificate(sslSessionId, x509Certs);
                sslSession.putValue(SSL_SESSION_CACHE, sslSessionCache);
            }
            
            if (x509Certs == null && ssl.isClientAuth())
            {
                socket.close();
                return;
            }
            
            //连接交给控制器多线程处理
            controller.execute(add(newConnection(socket, x509Certs)));
        }
        catch (SocketException e)
        {
            if (!_SOCKET_CLOSED_.equalsIgnoreCase(e.getMessage()))
            {//非服务端自己关闭连接导致的异常打印
                log.error(toString() + "SocketException异常", e);
            }
        }
        catch (IOException e)
        {
            log.error(toString() + "IO异常", e);
        }
    }
    
    /** 新建连接 */
    private HttpBioConnection newConnection(Socket socket, X509Certificate[] x509Certs) throws IOException
    {//把输入输出流的引用先拿出来
        return new HttpBioConnection(this, socket, socket.getInputStream(), socket.getOutputStream(), x509Certs);
    }
}
